#include "endpoint_storage_host.h"
#include "proxy_factory.h"

#include <yandex_io/sdk/private/endpoint_storage/converters/remote_object_id_factory.h>
#include <yandex_io/sdk/private/endpoint_storage/converters/converters.h>
#include <yandex_io/sdk/private/endpoint_storage/converters/remoting_message_builder.h>

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/proto_trace.h>

#include <yandex_io/protos/quasar_proto.pb.h>

#include <util/system/yassert.h>
#include <util/generic/yexception.h>

#include <algorithm>
#include <sstream>

YIO_DEFINE_LOG_MODULE("endpoint_storage_host");

namespace YandexIO {

    static const std::string s_remotingRegistryName = "EndpointStorage";

    EndpointStorageHost::EndpointStorageHost(std::weak_ptr<IRemotingRegistry> remotingRegistry,
                                             std::string deviceId,
                                             NAlice::TEndpoint::EEndpointType type)
        : IRemoteObject(std::move(remotingRegistry))
        , proxyConnections_(std::make_shared<ConnectionRegistry>())
    {
        auto iendpoint = createEndpoint(deviceId, type, {}, nullptr);
        localEndpoint_ = findCreatedEndpoint(iendpoint);
        Y_VERIFY(localEndpoint_);
    }

    EndpointStorageHost::~EndpointStorageHost()
    {
        if (auto remotingRegistry = getRemotingRegistry().lock()) {
            remotingRegistry->removeRemoteObject(remoteObjectId_);
        }
    }

    void EndpointStorageHost::init()
    {
        if (auto remotingRegistry = getRemotingRegistry().lock()) {
            remoteObjectId_ = RemoteObjectIdFactory::createId(shared_from_this());
            remotingRegistry->addRemoteObject(remoteObjectId_, weak_from_this());
        }
    }

    std::shared_ptr<IEndpoint> EndpointStorageHost::getLocalEndpoint() const {
        return localEndpoint_;
    }

    void EndpointStorageHost::addListener(std::weak_ptr<IListener> wlistener)
    {
        listeners_.push_back(wlistener);
    }

    void EndpointStorageHost::removeListener(std::weak_ptr<IListener> wlistener)
    {
        const auto iter = std::find_if(listeners_.begin(), listeners_.end(),
                                       [wlistener](const auto& wp) {
                                           return !(wp.owner_before(wlistener) || wlistener.owner_before(wp));
                                       });

        if (iter != listeners_.end()) {
            listeners_.erase(iter);
        }
    }

    void EndpointStorageHost::addEndpoint(const std::shared_ptr<IEndpoint>& iendpoint)
    {
        if (iendpoint == nullptr) {
            return;
        }
        const auto endpoint = findCreatedEndpoint(iendpoint);
        Y_ENSURE(endpoint != nullptr, "Endpoint was not created by this EndpointStorage instance");

        if (!addEndpointImpl(endpoint, hostStorageName_)) {
            return;
        }

        const auto message = RemotingMessageBuilder::buildAddEndpoint(shared_from_this(), hostStorageName_, endpoint->toProto());
        proxyConnections_->broadcastMessage(message, {});
    }

    void EndpointStorageHost::removeEndpoint(const std::shared_ptr<IEndpoint>& iendpoint)
    {
        dropDeadCreatedEndpoints();
        if (iendpoint == nullptr) {
            return;
        }
        Y_ENSURE(iendpoint->getId() != localEndpoint_->getId(), "Can't remove local endpoint");

        const auto endpoint = findCreatedEndpoint(iendpoint);
        Y_ENSURE(endpoint, "Can't remove endpoint created not by this EndpointStorage");

        removeEndpointImpl(endpoint->getId(), hostStorageName_);
        const auto message = RemotingMessageBuilder::buildRemoveEndpoint(shared_from_this(), hostStorageName_, endpoint->toProto());
        proxyConnections_->broadcastMessage(message, {});
    }

    std::list<std::shared_ptr<IEndpoint>> EndpointStorageHost::getEndpoints() const {
        std::list<std::shared_ptr<IEndpoint>> endpoints;
        endpoints.push_back(localEndpoint_);

        for (const auto& [proxyName, storage] : endpoints_) {
            for (const auto& endpoint : storage) {
                endpoints.push_back(endpoint);
            }
        }
        return endpoints;
    }

    void EndpointStorageHost::setCapabilityConfig(quasar::proto::CapabilityConfig config)
    {
        capabilityConfig_ = std::move(config);
        broadcastCapabilityConfig();
    }

    void EndpointStorageHost::broadcastCapabilityConfig()
    {
        YIO_LOG_INFO("broadcastCapabilityConfig proxyConnections_=" << proxyConnections_->getSize());
        const auto message = RemotingMessageBuilder::builCapabilityConfig(shared_from_this(), capabilityConfig_);
        proxyConnections_->broadcastMessage(message, {});
    }

    void EndpointStorageHost::handleRemotingMessage(
        const quasar::proto::Remoting& message,
        std::shared_ptr<IRemotingConnection> connection)
    {
        if (message.has_endpoint_storage_method()) {
            const auto& method = message.endpoint_storage_method();
            if (method.storage_name().empty()) {
                YIO_LOG_ERROR_EVENT("EndpointStorageHost.EmptyProxyName", "Proxy must provide storage name. Drop message");
                return;
            }
            const auto& proxyStorage = method.storage_name();

            if (method.method() == quasar::proto::Remoting::EndpointStorageMethod::ADD_ENDPOINT) {
                const std::string& endpointId = method.endpoints(0).endpoint().id();
                if (method.endpoints_size() != 1) {
                    YIO_LOG_ERROR_EVENT("EndpointStorageHost.AddEndpoint", "Failed to remove endpoint: endpoints_size != 1");
                } else {
                    auto endpoint = createEndpointProxy(method.endpoints(0), connection);
                    if (!addEndpointImpl(endpoint, method.storage_name())) {
                        YIO_LOG_ERROR_EVENT("EndpointStorageHost.AddEndpoint", "Failed to add endpoint with id=" << endpointId);
                    }
                    proxyConnections_->broadcastMessage(message, {proxyStorage});
                }
            } else if (method.method() == quasar::proto::Remoting::EndpointStorageMethod::REMOVE_ENDPOINT) {
                const std::string& endpointId = method.endpoints(0).endpoint().id();
                if (method.endpoints_size() != 1) {
                    YIO_LOG_ERROR_EVENT("EndpointStorageHost.RemoveEndpoint", "Failed to remove endpoint: endpoints_size != 1");
                } else {
                    // NOTE: Currently proxy can remove only it's own endpoints
                    removeEndpointImpl(endpointId, proxyStorage);
                    proxyConnections_->broadcastMessage(message, {proxyStorage});
                }

            } else if (method.method() == quasar::proto::Remoting::EndpointStorageMethod::SYNC_ENDPOINTS) {
                proxyConnections_->putConnection(proxyStorage, connection);

                YIO_LOG_INFO("EndpointStorageMethod::SYNC_ENDPOINTS for proxy:" << proxyStorage);
                removeProxyEndpoints(proxyStorage);

                // save proxy endpoints
                for (const auto& endpointProto : method.endpoints()) {
                    auto endpoint = createEndpointProxy(endpointProto, connection);
                    if (!addEndpointImpl(endpoint, proxyStorage)) {
                        YIO_LOG_ERROR_EVENT("EndpointStorageHost.SyncEndpoints", "Add Endpoint failed");
                    }
                }

                // send local endpoint
                {
                    const auto message = RemotingMessageBuilder::buildAddEndpoint(shared_from_this(), hostStorageName_, localEndpoint_->toProto());
                    connection->sendMessage(message);
                }
                // send stored endpoints to proxy
                for (const auto& [storage, endpoints] : endpoints_) {
                    if (storage == proxyStorage) {
                        continue; // skip proxy itself
                    }
                    for (const auto& endpoint : endpoints) {
                        const auto message = RemotingMessageBuilder::buildAddEndpoint(shared_from_this(), proxyStorage, endpoint->toProto());
                        connection->sendMessage(message);
                    }
                }

                // send new added endpoints to other proxies
                for (const auto& endpoint : endpoints_[proxyStorage]) {
                    const auto message = RemotingMessageBuilder::buildAddEndpoint(shared_from_this(), proxyStorage, endpoint->toProto());
                    proxyConnections_->broadcastMessage(message, {proxyStorage});
                }

                const auto configMessage = RemotingMessageBuilder::builCapabilityConfig(shared_from_this(), capabilityConfig_);
                connection->sendMessage(configMessage);
            }
        }
    }

    bool EndpointStorageHost::addEndpointImpl(const std::shared_ptr<Endpoint>& endpoint, const std::string& storageName)
    {
        Y_VERIFY(endpoint != nullptr);

        auto& endpoints = endpoints_[storageName];

        // todo: make sure that other proxies doesn't have endpoint with same id
        const auto iter = std::find_if(endpoints.cbegin(), endpoints.cend(), [&](const auto& item) {
            return endpoint->getId() == item->getId();
        });
        if (iter != endpoints.cend()) {
            YIO_LOG_WARN("Failed to addEndpointImpl: endpointId already exists");
            return false;
        }

        endpoints.push_back(endpoint);

        YIO_LOG_INFO("addEndpointImpl for storage: " << storageName << " endpointId: " << endpoint->getId() << ", listeners_=" << listeners_.size());
        forEachListener([&endpoint](const auto& listener) {
            listener->onEndpointAdded(endpoint);
        });

        return true;
    }

    bool EndpointStorageHost::removeEndpointImpl(const std::string& endpointId, const std::string& storageName)
    {
        auto& endpoints = endpoints_[storageName];
        const auto iter = std::find_if(endpoints.begin(), endpoints.end(), [&endpointId](const auto& endpoint) {
            return endpoint->getId() == endpointId;
        });
        if (iter == endpoints.end()) {
            return true;
        }

        auto endpoint = std::move(*iter);
        endpoints.erase(iter);

        forEachListener([&endpoint](const auto& listener) {
            listener->onEndpointRemoved(endpoint);
        });
        return true;
    }

    void EndpointStorageHost::removeProxyEndpoints(const std::string& proxyStorageName) {
        auto& endpoints = endpoints_[proxyStorageName];
        for (auto it = endpoints.begin(); it != endpoints.end();) {
            const auto endpoint = std::move(*it);
            it = endpoints.erase(it);

            forEachListener([&endpoint](const auto& listener) {
                listener->onEndpointRemoved(endpoint);
            });

            // request proxies to remove endpoints
            const auto message = RemotingMessageBuilder::buildRemoveEndpoint(shared_from_this(), proxyStorageName, endpoint->toProto());
            proxyConnections_->broadcastMessage(message, {proxyStorageName});
        }
    }

    void EndpointStorageHost::forEachListener(std::function<void(const std::shared_ptr<IEndpointStorage::IListener>&)> callback) {
        for (const auto& wlistener : listeners_) {
            if (auto slistener = wlistener.lock()) {
                callback(slistener);
            }
        }
    }

    std::shared_ptr<Endpoint> EndpointStorageHost::findCreatedEndpoint(const std::shared_ptr<IEndpoint>& iendpoint) {
        dropDeadCreatedEndpoints();
        for (const auto& wendpoint : createdEndpoints_) {
            auto endpoint = wendpoint.lock();
            if (endpoint && endpoint == iendpoint) {
                return endpoint;
            }
        }

        return nullptr;
    }

    void EndpointStorageHost::dropDeadCreatedEndpoints() {
        std::erase_if(createdEndpoints_, [](const auto& wendpoint) {
            return wendpoint.expired();
        });
    }

    std::shared_ptr<IEndpoint> EndpointStorageHost::createEndpoint(std::string id, NAlice::TEndpoint::EEndpointType type, NAlice::TEndpoint::TDeviceInfo deviceInfo, std::shared_ptr<IDirectiveHandler> directiveHandler) {
        if (localEndpoint_) {
            Y_ENSURE(id != localEndpoint_->getId(), "Can't create local endpoint");
        }
        dropDeadCreatedEndpoints();

        constexpr bool isEndpointHost = true;
        const auto endpoint = std::make_shared<Endpoint>(
            std::move(id),
            type,
            std::move(deviceInfo),
            std::move(directiveHandler),
            hostStorageName_,
            isEndpointHost,
            hostStorageName_,
            getRemotingRegistry(),
            proxyConnections_);

        endpoint->initRemoting();
        createdEndpoints_.push_back(endpoint);
        return endpoint;
    }

    std::shared_ptr<Endpoint> EndpointStorageHost::createEndpointProxy(
        const quasar::proto::Endpoint& message,
        const std::shared_ptr<IRemotingConnection>& connection)
    {
        constexpr bool isEndpointHost = false;
        const auto hostStorageName = proxyConnections_->getStorageName(connection);
        // can't create EndpointProxy without knowing it's host address
        Y_VERIFY(hostStorageName.has_value());
        std::shared_ptr<IDirectiveHandler> directiveHandler;
        if (message.has_directive_handler()) {
            directiveHandler = YandexIO::createDirectiveHandlerProxy(
                message.directive_handler(),
                RemoteObjectIdFactory::createEndpointId(message.endpoint().id()),
                proxyConnections_,
                *hostStorageName);
        }
        auto endpoint = std::make_shared<Endpoint>(
            message.endpoint().id(),
            message.endpoint().meta().type(),
            message.endpoint().meta().deviceinfo(),
            std::move(directiveHandler),
            hostStorageName_,
            isEndpointHost,
            *hostStorageName,
            getRemotingRegistry(),
            proxyConnections_);

        endpoint->initRemoting();
        endpoint->initProxyCapabilities(message);

        return endpoint;
    }

} // namespace YandexIO
