#include "endpoint_storage_proxy.h"
#include "proxy_factory.h"

#include <yandex_io/sdk/private/endpoint_storage/converters/remote_object_id_factory.h>
#include <yandex_io/sdk/private/endpoint_storage/converters/converters.h>
#include <yandex_io/sdk/private/endpoint_storage/converters/remoting_message_builder.h>

#include <yandex_io/libs/logging/logging.h>

#include <util/generic/yexception.h>
#include <util/system/yassert.h>

#include <future>

YIO_DEFINE_LOG_MODULE("endpoint_storage_proxy");

namespace YandexIO {
    EndpointStorageProxy::EndpointStorageProxy(std::string storageName, std::string deviceId,
                                               std::shared_ptr<quasar::ICallbackQueue> worker,
                                               std::weak_ptr<IRemotingRegistry> remotingRegistry)
        : IRemoteObject(std::move(remotingRegistry))
        , storageName_(std::move(storageName))
        , connections_(std::make_shared<ConnectionRegistry>())
        , localEndpoint_(std::make_shared<LocalEndpoint>(std::move(deviceId)))
        , worker_(std::move(worker))
    {
        Y_VERIFY(!storageName_.empty());
    }

    EndpointStorageProxy::~EndpointStorageProxy() = default;

    void EndpointStorageProxy::start()
    {
        // NOTE: Start is sync for now, so it will be guaranted that EndpointStorageProxy
        // registered in RemotingRegistry. Otherwise it can skip "handleRemotingConnect"
        if (!worker_->isWorkingThread()) {
            std::promise<void> promise;
            worker_->add([this, &promise]() {
                start();
                promise.set_value();
            });
            promise.get_future().get();
            return;
        }
        Y_VERIFY(isStarted_ == false);

        const auto remotingRegistry = getRemotingRegistry().lock();
        Y_VERIFY(remotingRegistry);

        remotingRegistry->addRemoteObject(RemoteObjectIdFactory::createId(shared_from_this()), weak_from_this());

        isStarted_ = true;
    }

    void EndpointStorageProxy::stop()
    {
        worker_->add([this]() {
            if (auto remotingRegistry = getRemotingRegistry().lock()) {
                remotingRegistry->removeRemoteObject(RemoteObjectIdFactory::createId(shared_from_this()));
            }
        });
    }

    std::shared_ptr<IEndpoint> EndpointStorageProxy::getLocalEndpoint() const {
        return localEndpoint_;
    }

    void EndpointStorageProxy::addListener(std::weak_ptr<IEndpointStorage::IListener> wlistener)
    {
        worker_->add([this, wlistener]() {
            listeners_.push_back(wlistener);
            if (const auto listener = wlistener.lock(); listener && capabilityConfig_.has_value()) {
                listener->onCapabilityConfigChanged(*capabilityConfig_);
            }
        });
    }

    void EndpointStorageProxy::removeListener(std::weak_ptr<IEndpointStorage::IListener> wlistener)
    {
        worker_->add([this, wlistener]() {
            const auto iter = std::find_if(listeners_.begin(), listeners_.end(),
                                           [wlistener](const auto& wp) {
                                               return !(wp.owner_before(wlistener) || wlistener.owner_before(wp));
                                           });

            if (iter != listeners_.end()) {
                listeners_.erase(iter);
            }
        });
    }

    void EndpointStorageProxy::addEndpoint(const std::shared_ptr<IEndpoint>& iendpoint)
    {
        if (iendpoint == nullptr) {
            return;
        }
        if (!worker_->isWorkingThread()) {
            worker_->add([this, iendpoint] {
                addEndpoint(iendpoint);
            });
            return;
        }

        YIO_LOG_INFO("[Proxy:" << storageName_ << "] addEndpoint " << iendpoint->getId());
        // Working Thread
        const auto endpoint = findCreatedEndpoint(iendpoint);
        Y_VERIFY(endpoint != nullptr, "Endpoint was not created by this EndpointStorage instance");

        const auto message = RemotingMessageBuilder::buildAddEndpoint(shared_from_this(), storageName_, endpoint->toProto());
        connections_->broadcastMessage(message, {});

        addEndpointImpl(endpoint, endpoints_);
    }

    void EndpointStorageProxy::removeEndpoint(const std::shared_ptr<IEndpoint>& iendpoint)
    {
        if (iendpoint == nullptr) {
            return;
        }
        if (!worker_->isWorkingThread()) {
            worker_->add([this, iendpoint]() {
                removeEndpoint(iendpoint);
            });
            return;
        }
        YIO_LOG_INFO("[" << storageName_ << "] removeEndpoint " << iendpoint->getId());
        Y_ENSURE(iendpoint->getId() != localEndpoint_->getId(), "Can't remove local endpoint");
        // Working Thread
        dropDeadCreatedEndpoints();
        const auto endpoint = findCreatedEndpoint(iendpoint);
        Y_ENSURE(endpoint, "Can't remove endpoint created not by this EndpointStorage");

        const auto message = RemotingMessageBuilder::buildRemoveEndpoint(shared_from_this(), storageName_, endpoint->toProto());
        connections_->broadcastMessage(message, {});

        removeEndpointImpl(endpoint->getId(), endpoints_);
    }

    std::list<std::shared_ptr<IEndpoint>> EndpointStorageProxy::getEndpoints() const {
        if (!worker_->isWorkingThread()) {
            std::promise<std::list<std::shared_ptr<IEndpoint>>> promise;
            worker_->add([this, &promise]() {
                promise.set_value(getEndpoints());
            });
            return promise.get_future().get();
        }

        // Working thread
        std::list<std::shared_ptr<IEndpoint>> endpoints = {endpoints_.begin(), endpoints_.end()};
        endpoints.push_back(localEndpoint_);
        for (const auto& endpoint : proxyEndpoints_) {
            endpoints.push_back(endpoint);
        }
        return endpoints;
    }

    void EndpointStorageProxy::handleRemotingConnect(std::shared_ptr<IRemotingConnection> connection) {
        if (!worker_->isWorkingThread()) {
            worker_->add([this, connection]() {
                handleRemotingConnect(connection);
            });
            return;
        }
        // Working thread

        // EndpointStorageProxy can be connected only to Host
        // so we know for sure we have only 1 connection
        connections_->putConnection("EndpointStorageHost", connection);
        syncEndpoints();
    }

    void EndpointStorageProxy::handleRemotingMessage(const quasar::proto::Remoting& message, std::shared_ptr<IRemotingConnection> connection)
    {
        if (!worker_->isWorkingThread()) {
            worker_->add([this, message, connection] {
                handleRemotingMessage(message, connection);
            });
            return;
        }

        // Working Thread

        if (message.has_endpoint_storage_send_capability_config_method()) {
            const auto& method = message.endpoint_storage_send_capability_config_method();
            capabilityConfig_ = method.capability_config();
            forEachListener([this](const auto& listener) {
                listener->onCapabilityConfigChanged(*capabilityConfig_);
            });
        } else if (message.has_endpoint_storage_method()) {
            const auto& method = message.endpoint_storage_method();
            if (method.endpoints_size() != 1) {
                YIO_LOG_ERROR_EVENT("EndpointStorageProxy.EndpointStorageMethod", "Invalid endpoint method: endpoints_size != 1");
                return;
            }

            const std::string& endpointId = method.endpoints(0).endpoint().id();
            if (method.method() == quasar::proto::Remoting::EndpointStorageMethod::ADD_ENDPOINT) {
                const auto endpoint = createEndpointProxy(method.endpoints(0), connection);
                if (endpoint->getId() == localEndpoint_->getId()) {
                    localEndpoint_->setLocalEndpoint(endpoint);
                } else {
                    // todo: ensure endoint is unique
                    addEndpointImpl(endpoint, proxyEndpoints_);
                }
            } else if (method.method() == quasar::proto::Remoting::EndpointStorageMethod::REMOVE_ENDPOINT) {
                removeEndpointImpl(endpointId, proxyEndpoints_);
            }
        }
    }

    void EndpointStorageProxy::syncEndpoints()
    {
        Y_ENSURE_THREAD(worker_);

        YIO_LOG_INFO("[" << storageName_ << "] syncEndpoints");
        dropProxyEndpoints();
        localEndpoint_->removeLocalEndpoint(); // host should resend real local endpoint

        // Drop all proxy capabilities from real endpoints
        // EndpointStorageHost should resend them all
        for (const auto& endpoint : proxyEndpoints_) {
            endpoint->dropProxyCapabilities();
        }

        // build protos only for EndpointHost for sync with EndpointStorageHost
        std::vector<quasar::proto::Endpoint> endpointsProtos;
        for (const auto& endpoint : endpoints_) {
            endpointsProtos.push_back(endpoint->toProto());
        }

        const auto message = RemotingMessageBuilder::buildSyncEndpoints(shared_from_this(), storageName_, endpointsProtos);
        connections_->broadcastMessage(message, {});
        YIO_LOG_INFO("[" << storageName_ << "] syncEndpoints sent. endpoints_.size=" << endpoints_.size());
    }

    bool EndpointStorageProxy::addEndpointImpl(const std::shared_ptr<Endpoint>& endpoint, std::list<std::shared_ptr<Endpoint>>& endpoints)
    {
        Y_ENSURE_THREAD(worker_);
        // todo: Ensure endpoints are unique
        YIO_LOG_INFO("[Proxy:" << storageName_ << "] endpoint added " << endpoint->getId() << " listeners_:" << listeners_.size());
        endpoints.push_back(endpoint);

        forEachListener([&endpoint](const auto& listener) {
            listener->onEndpointAdded(endpoint);
        });
        return true;
    }

    bool EndpointStorageProxy::removeEndpointImpl(const std::string& endpointId, std::list<std::shared_ptr<Endpoint>>& endpoints)
    {
        Y_ENSURE_THREAD(worker_);

        const auto iter = std::find_if(endpoints.begin(), endpoints.end(), [&](const std::shared_ptr<Endpoint>& endpoint) {
            return endpoint->getId() == endpointId;
        });
        if (iter == endpoints.end()) {
            return false;
        }

        const auto endpoint = std::move(*iter);
        endpoints.erase(iter);

        forEachListener([&endpoint](const auto& listener) {
            listener->onEndpointRemoved(endpoint);
        });

        return true;
    }

    void EndpointStorageProxy::dropProxyEndpoints() {
        Y_ENSURE_THREAD(worker_);
        while (!proxyEndpoints_.empty()) {
            const auto endpoint = proxyEndpoints_.front();
            removeEndpointImpl(endpoint->getId(), proxyEndpoints_);
        }
    }

    std::shared_ptr<Endpoint> EndpointStorageProxy::findCreatedEndpoint(const std::shared_ptr<IEndpoint>& iendpoint) {
        Y_ENSURE_THREAD(worker_);

        dropDeadCreatedEndpoints();
        for (const auto& wendpoint : createdEndpoints_) {
            auto endpoint = wendpoint.lock();
            if (endpoint && endpoint == iendpoint) {
                return endpoint;
            }
        }

        return nullptr;
    }

    void EndpointStorageProxy::dropDeadCreatedEndpoints() {
        Y_ENSURE_THREAD(worker_);
        std::erase_if(createdEndpoints_, [](const auto& wendpoint) {
            return wendpoint.expired();
        });
    }

    std::shared_ptr<IEndpoint> EndpointStorageProxy::createEndpoint(std::string id, NAlice::TEndpoint::EEndpointType type, NAlice::TEndpoint::TDeviceInfo deviceInfo, std::shared_ptr<IDirectiveHandler> directiveHandler) {
        Y_ENSURE(id != localEndpoint_->getId(), "Can't create local endpoint");

        if (!worker_->isWorkingThread()) {
            std::promise<std::shared_ptr<IEndpoint>> promise;
            worker_->add([this, &promise, id{std::move(id)}, type, deviceInfo{std::move(deviceInfo)}, directiveHandler{std::move(directiveHandler)}]() mutable {
                promise.set_value(createEndpoint(std::move(id), type, std::move(deviceInfo), std::move(directiveHandler)));
            });
            return promise.get_future().get();
        }

        // Working Thread
        dropDeadCreatedEndpoints();
        const bool isEndpointHost = true;
        const auto endpoint = std::make_shared<Endpoint>(
            id,
            type,
            deviceInfo,
            std::move(directiveHandler),
            storageName_,
            isEndpointHost,
            storageName_,
            getRemotingRegistry(),
            connections_);

        endpoint->initRemoting();
        createdEndpoints_.push_back(endpoint);

        return endpoint;
    }

    std::shared_ptr<Endpoint> EndpointStorageProxy::createEndpointProxy(
        const quasar::proto::Endpoint& message,
        const std::shared_ptr<IRemotingConnection>& connection)
    {
        Y_ENSURE_THREAD(worker_);

        const auto hostStorageName = connections_->getStorageName(connection);
        constexpr bool isEndpointHost = false;
        // can't create EndpointProxy without knowing it's host address
        Y_VERIFY(hostStorageName.has_value());
        std::shared_ptr<IDirectiveHandler> directiveHandler;
        if (message.has_directive_handler()) {
            directiveHandler = YandexIO::createDirectiveHandlerProxy(message.directive_handler(),
                                                                     RemoteObjectIdFactory::createEndpointId(message.endpoint().id()),
                                                                     connections_, *hostStorageName);
        }
        auto endpoint = std::make_shared<Endpoint>(
            message.endpoint().id(),
            message.endpoint().meta().type(),
            message.endpoint().meta().deviceinfo(),
            std::move(directiveHandler),
            storageName_,
            isEndpointHost,
            *hostStorageName,
            getRemotingRegistry(),
            connections_);
        endpoint->initRemoting();
        endpoint->initProxyCapabilities(message);

        return endpoint;
    }

    void EndpointStorageProxy::forEachListener(std::function<void(const std::shared_ptr<IEndpointStorage::IListener>&)> callback) {
        for (const auto& wlistener : listeners_) {
            if (auto slistener = wlistener.lock()) {
                callback(slistener);
            }
        }
    }

} // namespace YandexIO
