#include "local_endpoint.h"

#include <yandex_io/sdk/private/endpoint_storage/converters/converters.h>

namespace YandexIO {

    LocalEndpoint::LocalEndpoint(std::string id)
        : id_(std::move(id))
    {
        state_.set_id(TString(id_));
        // start with unkown type. It will be provided by host
        state_.mutable_meta()->set_type(NAlice::TEndpoint::UnknownEndpointType);
        state_.mutable_meta()->mutable_deviceinfo()->CopyFrom(NAlice::TEndpoint::TDeviceInfo{});
        NAlice::TEndpoint::TStatus status;
        state_.mutable_status()->CopyFrom(status);
    }

    LocalEndpoint::~LocalEndpoint() = default;

    void LocalEndpoint::setLocalEndpoint(std::shared_ptr<IEndpoint> endpoint) {
        Y_VERIFY(realEndpoint_ == nullptr, "Can't reinstall local endpoint");
        Y_VERIFY(endpoint->getId() == state_.id());
        realEndpoint_ = std::move(endpoint);
        for (const auto& capability : realEndpoint_->getCapabilities()) {
            // notify localEndpoint listeners about installed "remote" capability
            const auto sthis = shared_from_this();
            forEachListener([&sthis, &capability](const auto& listener) {
                listener->onCapabilityAdded(sthis, capability);
            });
        }

        // install "local" capabilities into real endpoint
        for (const auto& capability : capabilities_) {
            realEndpoint_->addCapability(capability);
        }
        // install self as dispatcher for own listeners
        realEndpoint_->addListener(weak_from_this());
    }

    void LocalEndpoint::removeLocalEndpoint() {
        if (!realEndpoint_) {
            return;
        }
        const auto capabilities = realEndpoint_->getCapabilities();
        for (const auto& capability : capabilities) {
            const auto it = std::find(capabilities_.begin(), capabilities_.end(), capability);
            if (it != capabilities_.end()) {
                // do not "remove" Capabilities installed into this LocalEndpoint instance
                // notify listeners about remote Capabilities remove only
                continue;
            }
            const auto sthis = shared_from_this();
            forEachListener([&sthis, &capability](const auto& listener) {
                listener->onCapabilityRemoved(sthis, capability);
            });
        }
        realEndpoint_->removeListener(weak_from_this());
        realEndpoint_.reset();
    }

    const std::string& LocalEndpoint::getId() const {
        return id_;
    }

    NAlice::TEndpoint LocalEndpoint::getState() const {
        if (realEndpoint_) {
            return realEndpoint_->getState();
        }

        NAlice::TEndpoint state = state_;
        appendCapabilities(state, capabilities_);
        return state;
    }

    void LocalEndpoint::setStatus(const NAlice::TEndpoint::TStatus& /*status*/) {
        Y_FAIL("Can't update status of local endpoint");
    }

    NAlice::TEndpoint::TStatus LocalEndpoint::getStatus() const {
        if (realEndpoint_) {
            return realEndpoint_->getStatus();
        }
        return state_.status();
    }

    void LocalEndpoint::addCapability(const std::shared_ptr<ICapability>& capability) {
        capabilities_.push_back(capability);

        if (realEndpoint_) {
            // real endpoint will notify listeners
            realEndpoint_->addCapability(capability);
            return;
        }

        const auto sthis = shared_from_this();
        forEachListener([&sthis, &capability](const auto& listener) {
            listener->onCapabilityAdded(sthis, capability);
        });
    }

    void LocalEndpoint::removeCapability(const std::shared_ptr<ICapability>& capability)
    {
        if (realEndpoint_) {
            realEndpoint_->removeCapability(capability);
        }

        if (realEndpoint_) {
            // all listeners are already notified via dispatcher
            return;
        }
        removeCapabilityLocal(capability);

        const auto sthis = shared_from_this();
        forEachListener([&sthis, &capability](const auto& listener) {
            listener->onCapabilityRemoved(sthis, capability);
        });
    }

    void LocalEndpoint::removeCapabilityLocal(const std::shared_ptr<ICapability>& capability) {
        const auto iter = std::find(capabilities_.begin(), capabilities_.end(), capability);
        if (iter == capabilities_.end()) {
            return;
        }
        capabilities_.erase(iter);
    }

    std::list<std::shared_ptr<ICapability>> LocalEndpoint::getCapabilities() const {
        if (realEndpoint_) {
            return realEndpoint_->getCapabilities();
        }
        return capabilities_;
    }

    void LocalEndpoint::addListener(std::weak_ptr<IListener> wlistener) {
        wlisteners_.push_back(wlistener);
    }

    void LocalEndpoint::removeListener(std::weak_ptr<IListener> wlistener) {
        const auto iter = std::find_if(wlisteners_.begin(), wlisteners_.end(),
                                       [wlistener](const auto& wp) {
                                           return !(wp.owner_before(wlistener) || wlistener.owner_before(wp));
                                       });

        if (iter != wlisteners_.end()) {
            wlisteners_.erase(iter);
        }
    }

    std::shared_ptr<IDirectiveHandler> LocalEndpoint::getDirectiveHandler() const {
        if (realEndpoint_) {
            return realEndpoint_->getDirectiveHandler();
        }
        return nullptr;
    }

    void LocalEndpoint::onCapabilityAdded(const std::shared_ptr<IEndpoint>& /*enpdoint*/, const std::shared_ptr<ICapability>& capability) {
        const auto sthis = shared_from_this();
        forEachListener([&sthis, &capability](const auto& listener) {
            listener->onCapabilityAdded(sthis, capability);
        });
    }

    void LocalEndpoint::onCapabilityRemoved(const std::shared_ptr<IEndpoint>& /*enpdoint*/, const std::shared_ptr<ICapability>& capability) {
        removeCapabilityLocal(capability); // drop capability localy
        const auto sthis = shared_from_this();
        forEachListener([&sthis, &capability](const auto& listener) {
            listener->onCapabilityRemoved(sthis, capability);
        });
    }

    void LocalEndpoint::onEndpointStateChanged(const std::shared_ptr<IEndpoint>& endpoint) {
        state_ = endpoint->getState();
        const auto sthis = shared_from_this();
        forEachListener([&sthis](const auto& listener) {
            listener->onEndpointStateChanged(sthis);
        });
    }

    void LocalEndpoint::forEachListener(std::function<void(const std::shared_ptr<IEndpoint::IListener>&)> callback) {
        for (const auto& wlistener : wlisteners_) {
            if (auto slistener = wlistener.lock()) {
                callback(slistener);
            }
        }
    }

} // namespace YandexIO
