#include "remoting_message_router.h"

#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/protos/quasar_proto.pb.h>

YIO_DEFINE_LOG_MODULE("remoting_message_router");

using namespace quasar;

namespace YandexIO {

    bool RemotingMessageRouter::addRemoteObject(
        const std::string& key, std::weak_ptr<IRemoteObject> object)
    {
        if (remoteObjects_.contains(key)) {
            YIO_LOG_ERROR_EVENT("RemotingMessageRouter.FailedToAddRemoteObject", "key " << key << " already exists");
            return false;
        }

        YIO_LOG_INFO("addRemoteObject key=" << key << ", object=" << object.lock().get());
        remoteObjects_.emplace(key, std::move(object));
        return true;
    }

    void RemotingMessageRouter::removeRemoteObject(const std::string& key)
    {
        YIO_LOG_INFO("removeRemoteObject key=" << key);
        remoteObjects_.erase(key);
    }

    // connector api
    void RemotingMessageRouter::routeRemotingMessage(const quasar::proto::Remoting& message,
                                                     const std::shared_ptr<ipc::IConnector>& connection) {
        routeRemotingMessageImpl(message, findOrPutConnection(connection));
    }

    void RemotingMessageRouter::routeRemotingConnect(const std::shared_ptr<ipc::IConnector>& connection) {
        routeRemotingConnectImpl(findOrPutConnection(connection));
    }

    // server api
    void RemotingMessageRouter::routeRemotingMessage(const quasar::proto::Remoting& message,
                                                     const std::shared_ptr<ipc::IServer::IClientConnection>& connection) {
        routeRemotingMessageImpl(message, findOrPutConnection(connection));
    }

    void RemotingMessageRouter::routeRemotingConnect(const std::shared_ptr<ipc::IServer::IClientConnection>& connection) {
        routeRemotingConnectImpl(findOrPutConnection(connection));
    }

    // real impl
    void RemotingMessageRouter::routeRemotingMessageImpl(
        const quasar::proto::Remoting& message,
        const std::shared_ptr<IRemotingConnection>& connection) const {
        if (message.has_remote_object_id()) {
            auto iter = remoteObjects_.find(message.remote_object_id());
            if (iter != remoteObjects_.end()) {
                if (auto remoteObject = iter->second.lock()) {
                    remoteObject->handleRemotingMessage(message, connection);
                }
            }
        }

        const auto wildCardIt = remoteObjects_.find("*");
        if (wildCardIt != remoteObjects_.end()) {
            if (const auto object = wildCardIt->second.lock()) {
                object->handleRemotingMessage(message, connection);
            }
        }
    }

    void RemotingMessageRouter::routeRemotingConnectImpl(const std::shared_ptr<IRemotingConnection>& connection) const {
        for (const auto& [_, weakObject] : remoteObjects_) {
            if (const auto object = weakObject.lock()) {
                object->handleRemotingConnect(connection);
            }
        }
    }

    std::shared_ptr<IRemotingConnection> RemotingMessageRouter::findOrPutConnection(const std::weak_ptr<quasar::ipc::IConnector>& connection) {
        std::shared_ptr<IRemotingConnection> remotingConnection;
        const auto it = connectorConnections_.find(connection);
        if (it == connectorConnections_.end()) {
            remotingConnection = std::make_shared<RemotingConnection>(connection);
            connectorConnections_[connection] = remotingConnection;
        } else {
            remotingConnection = it->second;
        }
        return remotingConnection;
    }

    std::shared_ptr<IRemotingConnection> RemotingMessageRouter::findOrPutConnection(const std::weak_ptr<quasar::ipc::IServer::IClientConnection>& connection) {
        std::shared_ptr<IRemotingConnection> remotingConnection;
        const auto it = serverConnections_.find(connection);
        if (it == serverConnections_.end()) {
            remotingConnection = std::make_shared<RemotingConnection>(connection);
            serverConnections_[connection] = remotingConnection;
        } else {
            remotingConnection = it->second;
        }
        return remotingConnection;
    }

    // RemotingConnection
    RemotingConnection::RemotingConnection(std::weak_ptr<ipc::IServer::IClientConnection> connection)
        : serverConnection_(std::move(connection))
    {
    }

    RemotingConnection::RemotingConnection(std::weak_ptr<ipc::IConnector> connection)
        : connector_(std::move(connection))
    {
    }

    void RemotingConnection::sendMessage(const quasar::proto::Remoting& remoting) {
        auto message = quasar::ipc::buildMessage([&](auto& msg) {
            msg.mutable_remoting()->CopyFrom(remoting);
        });

        if (auto connector = connector_.lock()) {
            connector->sendMessage(std::move(message));
        } else if (auto connection = serverConnection_.lock()) {
            connection->send(std::move(message));
        }
    }

} // namespace YandexIO
