#pragma once

//
// © YANDEX LLC, 2018
//

#pragma once

#include "alarm_observer.h"
#include "audio_client_event_observer.h"
#include "auth_observer.h"
#include "backend_config_observer.h"
#include "bluetooth_media_observer.h"
#include "brick_status_observer.h"
#include "clock_display_state.h"
#include "device_group_state_observer.h"
#include "device_mode_observer.h"
#include "directive_observer.h"
#include "discovery_observer.h"
#include "equalizer_info.h"
#include "media_observer.h"
#include "music_state_observer.h"
#include "notification_observer.h"
#include "push_notification_observer.h"
#include "registration_result.h"
#include "sdk_state_observer.h"
#include "sound_command_observer.h"
#include "spectrum_observer.h"
#include "tv_set_state.h"
#include "update_observer.h"
#include "wifi_info.h"

#include <yandex_io/capabilities/alarm/interfaces/i_alarm_capability.h>
#include <yandex_io/capabilities/alice/interfaces/i_alice_capability.h>
#include <yandex_io/capabilities/file_player/interfaces/i_file_player_capability.h>
#include <yandex_io/capabilities/playback_control/interfaces/i_playback_control_capability.h>
#include <yandex_io/capabilities/spotter/interfaces/i_spotter_capability.h>
#include <yandex_io/capabilities/device_state/interfaces/i_device_state_capability.h>
#include <yandex_io/sdk/interfaces/i_endpoint_storage.h>
#include <yandex_io/sdk/proto/device_sdk.pb.h>

#include <memory>
#include <optional>
#include <string>
#include <string_view>

namespace YandexIO {
    class SDKInterface {
    public:
        /**
         * @param input -- vins_response json string received from the server
         * @param output -- new value of vins_response json string that will be processed
         */
        using VinsResponsePreprocessorHandler = std::function<std::string(std::string)>;

    public:
        SDKInterface() = default;
        virtual ~SDKInterface();

        /**
         * @brief Method used to postpone starting device configurtion cycle (i.e. in order to play power on sound)
         *        Alice service will not respond to user until this method is called (since sdk will not have CONFIGURED state)
         */
        virtual void allowInitConfigurationState() = 0;

        virtual void addDeviceModeObserver(std::weak_ptr<DeviceModeObserver> observer) = 0;
        virtual void addAlarmObserver(std::weak_ptr<AlarmObserver> observer) = 0;
        virtual void addMediaObserver(std::weak_ptr<MediaObserver> observer) = 0;
        virtual void addUpdateObserver(std::weak_ptr<UpdateObserver> observer) = 0;
        virtual void addDirectiveObserver(std::weak_ptr<DirectiveObserver> observer) = 0;
        virtual void addBackendConfigObserver(std::weak_ptr<BackendConfigObserver> observer) = 0;
        virtual void addAuthObserver(std::weak_ptr<AuthObserver> observer) = 0;
        virtual void addSDKStateObserver(std::weak_ptr<SDKStateObserver> observer) = 0;
        virtual void addNotificationObserver(std::weak_ptr<NotificationObserver> observer) = 0;
        virtual void addSpectrumObserver(std::weak_ptr<SpectrumObserver> observer) = 0;
        virtual void addSoundCommandObserver(std::weak_ptr<SoundCommandObserver> observer) = 0;
        virtual void addMusicStateObserver(std::weak_ptr<MusicStateObserver> observer) = 0;
        virtual void addAudioClientEventObserver(std::weak_ptr<AudioClientEventObserver> observer) = 0;
        virtual void addBluetoothMediaObserver(std::weak_ptr<BluetoothMediaObserver> observer) = 0;
        virtual void addPushNotificationObserver(std::weak_ptr<PushNotificationObserver> observer) = 0;
        virtual void addBrickStatusObserver(std::weak_ptr<BrickStatusObserver> observer) = 0;
        virtual void addDeviceGroupStateObserver(std::weak_ptr<DeviceGroupStateObserver> observer) = 0;
        virtual void addDiscoveryObserver(std::weak_ptr<DiscoveryObserver> observer) = 0;

        /**
         * @brief Performs registration of this device on quasar backend.
         *        This method could be called only if sdk is bundled with GenericFirstRunService.
         * @note: This function should be called by trusted sources (expect valid match of oauthToken and uid)
         * @fixme: Make this method -- private SDK (for our "trusted" devices)
         */
        virtual void registerDeviceInBackend(std::string oauthToken, std::string uid) = 0;

        /**
         * @brief Performs sync registration of this device on quasar backend.
         *        This method could be called only if sdk is bundled with GenericFirstRunService.
         * @note: This function should be called by trusted sources (expect valid match of oauthToken and uid)
         * @fixme: Make this method -- private SDK (for our "trusted" devices)
         */
        virtual RegistrationResult registerDeviceInBackendSync(std::string oauthToken, std::string uid) = 0;

        /**
         * @brief Provide User account info to sdk core
         * @note: This function should be called by trusted sources (expect valid match of oauthToken and uid)
         * @fixme: Do not trust that oauthToken and uid match. Get oauth token and check it via passport API.
         *         Make this method -- private SDK (for our "trusted" devices)
         */
        virtual void provideUserAccountInfo(std::string oauthToken, std::string uid) = 0;

        /**
         * @brief Revoke User account info provided by provideUserAccountInfo.
         *        SDK will drop token and uid and will start wait for new account data
         */
        virtual void revokeUserAccountInfo() = 0;

        /**
         * @brief subscribes to device config, coming from backend. If we subscribe to specified config, backend config observers will handle it now
         * @param configName -- json field name in backend config, needed to be subscribed
         */
        virtual void subscribeToDeviceConfig(const std::string& configName) = 0;
        /**
         * @brief unsubscribes from device config, coming from backend. If we unsubscribe from specified config, backend config observers will not handle it anymore
         * @param configName -- json field name in device config, needed to be unsubscribed
         */
        virtual void unsubscribeFromDeviceConfig(const std::string& configName) = 0;

        /**
         * @brief subscribes to system config, coming from backend. If we subscribe to specified config, backend config observers will handle it now
         * @param configName -- json field name in system config, needed to be subscribed
         * @note If configName disappeared from system_config Json::nullValue will be sent
         */
        virtual void subscribeToSystemConfig(const std::string& configName) = 0;
        /**
         * @brief unsubscribes from system config, coming from backend. If we unsubscribe from specified config, backend config observers will not handle it anymore
         * @param configName -- json field name in system config, needed to be unsubscribed
         */
        virtual void unsubscribeFromSystemConfig(const std::string& configName) = 0;

        /**
         * @brief subscribes to account config, coming from backend. If we subscribe to specified config, backend config observers will handle it now
         * @param configName -- json field name in account config, needed to be subscribed
         * @note If configName disappeared from account_config Json::nullValue will be sent
         */
        virtual void subscribeToAccountConfig(const std::string& configName) = 0;
        /**
         * @brief unsubscribes from account config, coming from backend. If we unsubscribe from specified config, backend config observers will not handle it anymore
         * @param configName -- json field name in account config, needed to be unsubscribed
         */
        virtual void unsubscribeFromAccountConfig(const std::string& configName) = 0;

        virtual void setSetupMode() = 0;
        virtual void stopSetupMode() = 0;
        virtual void toggleSetupMode() = 0;

        virtual void reportTvPolicyInfo(const TvPolicyInfo& tvPolicyInfo) = 0;
        virtual void setActiveActions(const NAlice::TDeviceState::TActiveActions& activeActions) = 0;
        virtual void setActiveActionsSemanticFrames(const std::optional<std::string>& payload) = 0;

        /**
         * @brief Indicates that device is ready to play notification sound
         * This is synchronization mechanism. Notifies "notificationd" service that all required settings were set
         * (e.g. proper volume).
         * Without being notified, sound will be played after timeout anyway, though it can lead to data race
         * Use this method handling NotificationObserver::onNotificationPending()
         */
        virtual void notifyPreparedForNotification() = 0;

        /**
         * @brief Allow of prohibit updates on device
         * @param allowAllUpdates - if true, we can apply all updates.
         * @param allowCritical - if true, we can apply critical updates.
         * Critical updates are allowed if allowAllUpdates is true OR allowCritical is true.
         * Non-critical updates are allowed ONLY if allowAllUpdates is true.
         */
        virtual void setAllowUpdate(bool allowUpdateAll, bool allowCritical) = 0;

        /**
         * Approve alarm to start playing after making all pre-settings (e.g. set proper volume).
         * This is just synchronization mechanism. Without approval alarm will be fired as well after little timeout.
         * Call this method on handling AlarmObserver::onAlarmEnqueued()
         * @note Disabled by default. Can be turned on in quasar.cfg ( "alarmd" : { "needAlarmApproval": true } )
         * @param alarmId - id of the approved alarm
         */
        virtual void approveAlarm(const std::string& alarmId) = 0;

        /**
         * @brief Start changing account with provided code
         * @param oauthCode -- oauth code
         */
        virtual void authenticate(const std::string& oauthCode) = 0;

        virtual void bluetoothMediaSinkPause() = 0;
        virtual void bluetoothMediaSinkStart() = 0;
        virtual void bluetoothSinkConnected(const std::string& networkAddr, const std::string& networkName) = 0;
        virtual void bluetoothSinkDisconnected(const std::string& networkAddr, const std::string& networkName) = 0;
        virtual void bluetoothMediaSinkTrackInfo(const std::string& title, const std::string& artist, const std::string& album, const std::string& genre, int songLenMs, int currPosMs) = 0;

        virtual void acceptIncomingCall() = 0;
        virtual void declineIncomingCall() = 0;
        virtual void declineCurrentCall() = 0;

        /**
         * @brief Send numeric metrics to Yandex metric daemon. This daemon accumulates received data for some period
         * and sends number statistics (max, min, mean, last value etc)
         */
        virtual void sendNumericMetrics(const std::string& key, double value) = 0;
        /**
         * @brief Send categorical metrics to Yandex metric daemon. This daemon with some period sends received data
         */
        virtual void sendCategoricalMetrics(const std::string& key, const std::string& value) = 0;

        /**
         * Blocks any interaction with Alice. Can be called from different sources. Blocks Alice if at least one source blocks it.
         * @param sourceId unique source identifier.
         * @param errorSound sound which should be played when assistant is called instead of activating. If nullopt sound not played.
         *                   If several sounds are passed by different block sources one of them played.
         */
        virtual void blockVoiceAssistant(const std::string& sourceId, const std::optional<std::string>& errorSound) = 0;
        /**
         * Unblocks any interaction with Alice. Can be called from different sources. Unblocks Alice only if all sources unblock it.
         * @param sourceId unique source identifier.
         */
        virtual void unblockVoiceAssistant(const std::string& sourceId) = 0;

        /**
         * Reports to SDK from some kind of interface, that some screen connected via HDMI or via internals
         * @param isScreenActive status of screen active state
         */
        virtual void reportScreenActive(bool isScreenActive) = 0;

        /**
         * Provide part of device state for future Alice interaction
         * @param statePart
         */
        virtual void provideState(const yandex_io::proto::TDeviceStatePart& statePart) = 0;

        /**
         * Provide media service params for content recommendations
         * @param params
         */
        virtual void provideMediaDeviceIdentifier(const NAlice::TClientInfoProto::TMediaDeviceIdentifier& identifier) = 0;

        /**
         * Set Location data to SDK
         * @param latitude latitude
         * @param longitude longitude
         * @param accuracy accuracy
         */
        virtual void setLocation(double latitude, double longitude, std::optional<double> accuracy) = 0;

        /**
         * Set Timezone data to SDK
         * @param latitude latitude
         * @param longitude longitude
         * @param accuracy accuracy
         */
        virtual void setTimezone(const std::string& timezone, int32_t offsetSec) = 0;

        /**
         * Set Wifi info to SDK
         * @param wifiList list of wifi info
         */
        virtual void setWifiList(const std::vector<WifiInfo>& wifiList) = 0;

        /**
         * Set handler to preprocess vins_response
         * Handler receives vins_response json string and returns new value of it
         */
        virtual void setVinsResponsePreprocessorHandler(VinsResponsePreprocessorHandler value) = 0;

        /**
         * Sends vins response to sdk
         * @param responseJson vins response serialized to json
         */
        virtual void sendVinsResponse(const std::string& responseJson) = 0;

        /**
         * Reports state of clock display
         * @param clockDisplayState
         */
        virtual void reportClockDisplayState(ClockDisplayState clockDisplayState) = 0;

        virtual const std::shared_ptr<IEndpointStorage>& getEndpointStorage() const = 0;

        virtual void setEqualizerConfig(const EqualizerInfo& equalizerInfo) = 0;

        virtual std::shared_ptr<IAlarmCapability> getAlarmCapability() const = 0;
        virtual std::shared_ptr<IAliceCapability> getAliceCapability() const = 0;
        virtual std::shared_ptr<IFilePlayerCapability> getFilePlayerCapability() const = 0;
        virtual std::shared_ptr<IPlaybackControlCapability> getPlaybackControlCapability() const = 0;
        virtual std::shared_ptr<ISpotterCapability> getActivationSpotterCapability() const = 0;
        virtual std::shared_ptr<ISpotterCapability> getCommandSpotterCapability() const = 0;
        virtual std::shared_ptr<ISpotterCapability> getNaviOldSpotterCapability() const = 0;
        virtual std::shared_ptr<IDeviceStateCapability> getDeviceStateCapability() const = 0;
    };

} // namespace YandexIO
