#include "yandex_iosdk.h"

#include "converters/audio_client_event_converter.h"
#include "converters/device_group_state_converter.h"
#include "converters/glagol_discovery_result_converter.h"
#include "converters/wifi_info_converter.h"

#include <yandex_io/capabilities/alarm/alarm_capability_proxy.h>
#include <yandex_io/capabilities/alice/alice_capability_proxy.h>
#include <yandex_io/capabilities/file_player/file_player_capability_proxy.h>
#include <yandex_io/capabilities/playback_control/playback_control_capability_proxy.h>
#include <yandex_io/capabilities/spotter/spotter_capability_proxy.h>
#include <yandex_io/capabilities/device_state/proxy/device_state_capability_proxy.h>
#include <yandex_io/libs/base/named_callback_queue.h>
#include <yandex_io/libs/device/defines.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/proto_trace.h>
#include <yandex_io/protos/model_objects.pb.h>
#include <yandex_io/protos/quasar_proto.pb.h>

#include <mutex>
#include <optional>
#include <sstream>
#include <vector>

YIO_DEFINE_LOG_MODULE("sdk");

using namespace YandexIO;
using namespace quasar;

namespace {
    SDKState::AliceState convertAliceState(const proto::AliceState& aliceState) {
        SDKState::AliceState resultState;
        switch (aliceState.state()) {
            case proto::AliceState_State::AliceState_State_NONE:
            case proto::AliceState_State::AliceState_State_IDLE:
                resultState.state = SDKState::AliceState::State::IDLE;
                break;
            case proto::AliceState_State::AliceState_State_LISTENING:
                resultState.state = SDKState::AliceState::State::LISTENING;
                break;
            case proto::AliceState_State::AliceState_State_BUSY:
                resultState.state = SDKState::AliceState::State::THINKING;
                break;
            case proto::AliceState_State::AliceState_State_SPEAKING:
                resultState.state = SDKState::AliceState::State::SPEAKING;
                break;
            case proto::AliceState_State::AliceState_State_SHAZAM:
                resultState.state = SDKState::AliceState::State::SHAZAM;
                break;
            default:
                resultState.state = SDKState::AliceState::State::IDLE;
                break;
        }

        if (aliceState.has_recognized_phrase()) {
            resultState.recognizedPhrase = aliceState.recognized_phrase();
        }

        if (aliceState.has_vins_response()) {
            resultState.vinsResponse = SDKState::AliceState::VinsResponse();
            if (aliceState.vins_response().has_output_speech()) {
                resultState.vinsResponse->outputSpeech = aliceState.vins_response().output_speech();
            }
            if (aliceState.vins_response().has_text_only()) {
                resultState.vinsResponse->textOnly = aliceState.vins_response().text_only();
            }
            if (aliceState.vins_response().has_div_card()) {
                resultState.vinsResponse->divCard = aliceState.vins_response().div_card();
            }
            resultState.vinsResponse->suggests.reserve(aliceState.vins_response().suggests().size());
            for (const auto& protoSuggest : aliceState.vins_response().suggests()) {
                SDKState::AliceState::VinsResponse::Suggest suggest;
                if (protoSuggest.has_text()) {
                    suggest.text = protoSuggest.text();
                }
                resultState.vinsResponse->suggests.push_back(suggest);
            }
        }

        return resultState;
    }

    SDKState::NotificationState convertNotificationState(proto::IOEvent_SDKState_NotificationState notificationState) {
        if (notificationState == quasar::proto::IOEvent_SDKState_NotificationState_AVAILABLE) {
            return SDKState::NotificationState::AVAILABLE;
        }
        if (notificationState == quasar::proto::IOEvent_SDKState_NotificationState_PASSIVE) {
            return SDKState::NotificationState::PASSIVE;
        }

        return SDKState::NotificationState::NONE;
    }

    SDKState::ConfigurationState convertConfigurationState(proto::ConfigurationState cs) {
        switch (cs) {
            case proto::ConfigurationState::UNKNOWN_STATE: {
                return SDKState::ConfigurationState::UNKNOWN;
            }
            case proto::ConfigurationState::CONFIGURING: {
                return SDKState::ConfigurationState::CONFIGURING;
            }
            case proto::ConfigurationState::CONFIGURED: {
                return SDKState::ConfigurationState::CONFIGURED;
            }
            default:
                break;
        }
        return SDKState::ConfigurationState::UNKNOWN;
    }

    SDKState::WifiState convertWifiState(proto::WifiStatus protoWifiStatus) {
        SDKState::WifiState wifiState;
        // We now have 5 statuses, numbering starts with 0
        static_assert(proto::WifiStatus::Status_MAX == 4, "There must be 5 wifi statuses");
        auto wifiStatus = protoWifiStatus.status();
        switch (wifiStatus) {
            case proto::WifiStatus::NOT_CONNECTED:
            case proto::WifiStatus::CONNECTING:
            case proto::WifiStatus::NOT_CHOSEN:
                wifiState.isWifiConnected = false;
                break;
            case proto::WifiStatus::CONNECTED_NO_INTERNET:
            case proto::WifiStatus::CONNECTED:
                wifiState.isWifiConnected = true;
                break;
        }
        wifiState.isInternetReachable = protoWifiStatus.internet_reachable();
        return wifiState;
    }

    SDKState::CallState convertCallState(proto::CalldSessionState calldState) {
        SDKState::CallState st;

        switch (calldState.status()) {
            case proto::CalldSessionState::NEW:
                st.status = SDKState::CallState::Status::NEW;
                break;

            case proto::CalldSessionState::DIALING:
                st.status = SDKState::CallState::Status::DIALING;
                break;

            case proto::CalldSessionState::RINGING:
                st.status = SDKState::CallState::Status::RINGING;
                break;

            case proto::CalldSessionState::ACCEPTING:
                st.status = SDKState::CallState::Status::ACCEPTING;
                break;

            case proto::CalldSessionState::CONNECTING:
                st.status = SDKState::CallState::Status::CONNECTING;
                break;

            case proto::CalldSessionState::CONNECTED:
                st.status = SDKState::CallState::Status::CONNECTED;
                break;

            case proto::CalldSessionState::ENDED:
                if (calldState.has_ended_with_error()) {
                    st.status = SDKState::CallState::Status::ENDED_FAILURE;

                } else {
                    st.status = SDKState::CallState::Status::ENDED_OK;
                }

                break;

            case proto::CalldSessionState::NOCALL:
                st.status = SDKState::CallState::Status::NOCALL;
                break;
        }

        switch (calldState.direction()) {
            case proto::CalldSessionState::INCOMING:
                st.direction = SDKState::CallState::Direction::INCOMING;
                break;

            case proto::CalldSessionState::OUTGOING:
                st.direction = SDKState::CallState::Direction::OUTGOING;
                break;
        }

        return st;
    }

    SDKState::NtpdState convertNtpdState(const proto::IOEvent::NtpdState& st) {
        SDKState::NtpdState ntpdState;

        if (st.has_clock_synchronized()) {
            ntpdState.clockSynchronized = st.clock_synchronized();
        }

        return ntpdState;
    }

    SDKState::UpdateState convertUpdateState(const proto::UpdateState& us) {
        SDKState::UpdateState updateState;
        updateState.downloadProgress = us.download_progress();
        updateState.isCritical = us.is_critical();
        switch (us.state()) {
            case proto::UpdateState_State_DOWNLOADING: {
                updateState.state = SDKState::UpdateState::State::DOWNLOADING;
                break;
            }
            case proto::UpdateState_State_APPLYING: {
                updateState.state = SDKState::UpdateState::State::APPLYING;
                break;
            }
            case proto::UpdateState_State_NONE:
            default: {
                updateState.state = SDKState::UpdateState::State::NONE;
                break;
            }
        }
        return updateState;
    }

    SDKState::PlayerState convertPlayerState(const proto::AppState& as) {
        SDKState::PlayerState playerState;
        playerState.audio.isPlaying = as.has_audio_player_state() && as.audio_player_state().has_state() && as.audio_player_state().state() == proto::AudioClientState::PLAYING;
        if (as.has_music_state()) {
            playerState.music.currentTrackId = as.music_state().current_track_id();
            playerState.music.isPlaying = !as.music_state().is_paused();
            playerState.music.title = as.music_state().title();
            playerState.music.artists = as.music_state().artists();
            playerState.music.coverURL = as.music_state().cover_uri();
        }
        playerState.radio.isPlaying = as.has_radio_state() && !as.radio_state().is_paused();
        playerState.video.isPlaying = as.has_video_state() && as.video_state().has_is_paused() && !as.video_state().is_paused();

        return playerState;
    }

    SDKState::ScreenState convertScreenState(const proto::AppState& as) {
        SDKState::ScreenState screenState;
        if (as.has_screen_state() && as.screen_state().has_is_screensaver_on()) {
            screenState.isScreensaverOn = as.screen_state().is_screensaver_on();
        }

        return screenState;
    }

    std::vector<SDKState::TimerState> convertTimersState(const ::google::protobuf::RepeatedPtrField<proto::IOEvent::TimersTimingsState>& timers) {
        std::vector<SDKState::TimerState> result;
        result.reserve(timers.size());
        for (const auto& item : timers) {
            const auto startTs = std::chrono::time_point<std::chrono::system_clock>() + std::chrono::milliseconds(item.start_timer_ms());
            const auto endTs = std::chrono::time_point<std::chrono::system_clock>() + std::chrono::milliseconds(item.end_timer_ms());
            result.push_back({item.id(), startTs, endTs});
        }
        return result;
    }

    AlarmObserver::AlarmType convertAlarmType(const proto::IOEvent_AlarmType& alarmType) {
        // We now have 3 statuses, numbering starts with 0
        static_assert(proto::IOEvent::AlarmType_MAX == 2, "There must be 3 alarm statuses");
        switch (alarmType) {
            case proto::IOEvent::CLASSIC_ALARM:
                return AlarmObserver::AlarmType::CLASSIC_ALARM;
            case proto::IOEvent::MEDIA_ALARM:
                return AlarmObserver::AlarmType::MEDIA_ALARM;
            case proto::IOEvent::TIMER:
                return AlarmObserver::AlarmType::TIMER;
        }
        return AlarmObserver::AlarmType::CLASSIC_ALARM;
    }

    AlarmObserver::AlarmsSettings convertAlarmSettings(const proto::AlarmsSettings& alarmsSettings) {
        AlarmObserver::AlarmsSettings convertedSettings;
        if (alarmsSettings.has_max_volume_level()) {
            convertedSettings.finishVolume = alarmsSettings.max_volume_level();
        }
        if (alarmsSettings.has_min_volume_level()) {
            convertedSettings.startVolume = alarmsSettings.min_volume_level();
        }
        if (alarmsSettings.has_volume_raise_step_ms()) {
            convertedSettings.volumeRaiseStepMs = alarmsSettings.volume_raise_step_ms();
        }
        return convertedSettings;
    }

    MediaObserver::ContentType convertContentType(proto::MediaContentType contentType) {
        static_assert(proto::MediaContentType_MAX == 1, "Unexpected MediContentType enum values amount");
        switch (contentType) {
            case proto::MediaContentType::MUSIC: {
                return MediaObserver::ContentType::MUSIC;
            }
            case proto::MediaContentType::VIDEO: {
                return MediaObserver::ContentType::VIDEO;
            }
        }
        return MediaObserver::ContentType::MUSIC;
    }

    SDKState::IotState convertIotState(const proto::IotState& iotState) {
        static_assert(proto::IotState::State_MAX == 2, "There must be 3 IOT states");
        switch (iotState.current_state()) {
            case proto::IotState::IDLE:
                return SDKState::IotState::IDLE;
            case proto::IotState::STARTING_DISCOVERY:
                return SDKState::IotState::STARTING_DISCOVERY;
            case proto::IotState::DISCOVERY_IN_PROGRESS:
                return SDKState::IotState::DISCOVERY_IN_PROGRESS;
        }
        return SDKState::IotState::UNKNOWN;
    }

    SDKState::MultiroomState convertMultiroomState(const proto::MultiroomState& multiroomState) {
        SDKState::MultiroomState state;

        switch (multiroomState.mode()) {
            default:
            case proto::MultiroomState::NONE:
                state.mode = SDKState::MultiroomState::Mode::NONE;
                break;
            case proto::MultiroomState::MASTER:
                state.mode = SDKState::MultiroomState::Mode::MASTER;
                break;
            case proto::MultiroomState::SLAVE:
                state.mode = SDKState::MultiroomState::Mode::SLAVE;
                break;
        }
        state.playing = multiroomState.has_multiroom_broadcast() && multiroomState.multiroom_broadcast().state() == proto::MultiroomBroadcast::PLAYING;
        state.slaveClockSyncing = multiroomState.slave_clock_syncing();
        if (multiroomState.has_slave_sync_level()) {
            switch (multiroomState.slave_sync_level()) {
                case proto::MultiroomState::NOSYNC:
                    state.slaveSyncLevel = SDKState::MultiroomState::SyncLevel::NONE;
                    break;
                case proto::MultiroomState::WEAK:
                    state.slaveSyncLevel = SDKState::MultiroomState::SyncLevel::WEAK;
                    break;
                case proto::MultiroomState::STRONG:
                    state.slaveSyncLevel = SDKState::MultiroomState::SyncLevel::STRONG;
                    break;
                default:
                    break;
            }
        }
        return state;
    }

    SDKState::AllStartupSettings convertAllStartupSettings(const proto::AllStartupSettings& allStartupSettings) {
        SDKState::AllStartupSettings result;
        result.allStartupInfo = allStartupSettings.all_startup_info();

        if (allStartupSettings.has_auth_token()) {
            result.authToken = allStartupSettings.auth_token();
        }

        if (allStartupSettings.has_passport_uid()) {
            result.passportUid = allStartupSettings.passport_uid();
        }
        return result;
    }

    MusicStateObserver::TrackType convertTrackType(const proto::AppState::MusicState::TrackType type) {
        switch (type) {
            case proto::AppState::MusicState::TrackType::AppState_MusicState_TrackType_MUSIC:
                return MusicStateObserver::TrackType::MUSIC;
            case proto::AppState::MusicState::TrackType::AppState_MusicState_TrackType_FAIRY_TALE:
                return MusicStateObserver::TrackType::FAIRY_TALE;
            case proto::AppState::MusicState::TrackType::AppState_MusicState_TrackType_PODCAST:
                return MusicStateObserver::TrackType::PODCAST;
            case proto::AppState::MusicState::TrackType::AppState_MusicState_TrackType_AUDIOBOOK:
                return MusicStateObserver::TrackType::AUDIOBOOK;
            case proto::AppState::MusicState::TrackType::AppState_MusicState_TrackType_UNKNOWN:
                return MusicStateObserver::TrackType::UNKNOWN;
        }
        return MusicStateObserver::TrackType::UNKNOWN;
    }

    BrickStatus convertBrickStatus(const proto::BrickStatus& brickStatus) {
        switch (brickStatus) {
            case proto::BrickStatus::BRICK:
                return BrickStatus::BRICK;
            case proto::BrickStatus::NOT_BRICK:
                return BrickStatus::NOT_BRICK;
            case proto::BrickStatus::UNKNOWN_BRICK_STATUS:
                return BrickStatus::UNKNOWN_BRICK_STATUS;
            case proto::BrickStatus::BRICK_BY_TTL:
                return BrickStatus::BRICK_BY_TTL;
        }
    }

} /* anonymous namespace */

namespace YandexIO {
    class IOSDKStateHolder {
    public:
        bool getAllowInitConfigurationStateFired() const {
            std::lock_guard<std::mutex> guard(mutex_);
            return allowInitConfigurationStateFired_;
        }
        void setAllowInitConfigurationStateFired(bool allowInitConfigurationStateFired) {
            std::lock_guard<std::mutex> guard(mutex_);
            allowInitConfigurationStateFired_ = allowInitConfigurationStateFired;
        }

        void addDeviceConfigSubscription(const std::string& configName) {
            std::lock_guard<std::mutex> guard(mutex_);
            deviceConfigSubscriptions_.insert(configName);
        }
        void deleteDeviceConfigSubscription(const std::string& configName) {
            std::lock_guard<std::mutex> guard(mutex_);
            deviceConfigSubscriptions_.erase(configName);
        }
        std::set<std::string> getDeviceConfigSubscription() const {
            std::lock_guard<std::mutex> guard(mutex_);
            return deviceConfigSubscriptions_;
        }

        void addSystemConfigSubscription(const std::string& configName) {
            std::lock_guard<std::mutex> guard(mutex_);
            systemConfigSubscriptions_.insert(configName);
        }
        void deleteSystemConfigSubscription(const std::string& configName) {
            std::lock_guard<std::mutex> guard(mutex_);
            systemConfigSubscriptions_.erase(configName);
        }
        std::set<std::string> getSystemConfigSubscription() const {
            std::lock_guard<std::mutex> guard(mutex_);
            return systemConfigSubscriptions_;
        }

        void addAccountConfigSubscription(const std::string& configName) {
            std::lock_guard<std::mutex> guard(mutex_);
            accountConfigSubscriptions_.insert(configName);
        }
        void deleteAccountConfigSubscription(const std::string& configName) {
            std::lock_guard<std::mutex> guard(mutex_);
            accountConfigSubscriptions_.erase(configName);
        }
        std::set<std::string> getAccountConfigSubscription() const {
            std::lock_guard<std::mutex> guard(mutex_);
            return accountConfigSubscriptions_;
        }

        enum class UpdateAllow {
            ALLOWED,
            NOT_ALLOWED,
            UNKNOWN
        };
        void setAllowUpdate(bool allowUpdateAll, bool allowCritical) {
            std::lock_guard<std::mutex> guard(mutex_);
            allowUpdateAll_ = allowUpdateAll ? UpdateAllow::ALLOWED : UpdateAllow::NOT_ALLOWED;
            allowCritical_ = allowCritical ? UpdateAllow::ALLOWED : UpdateAllow::NOT_ALLOWED;
        }
        std::pair<UpdateAllow, UpdateAllow> getAllowUpdate() const {
            std::lock_guard<std::mutex> guard(mutex_);
            return {allowUpdateAll_, allowCritical_};
        }

        void setUserAccountInfo(std::string oauthToken, std::string uid) {
            UserAccountInfo info;
            info.oauthToken = std::move(oauthToken);
            info.uid = std::move(uid);
            std::scoped_lock guard(mutex_);
            userAccountInfo_ = std::move(info);
        }

        void removeUserAccountInfo() {
            std::scoped_lock guard(mutex_);
            userAccountInfo_.reset();
        }

        struct UserAccountInfo {
            std::string oauthToken;
            std::string uid;
        };
        std::optional<UserAccountInfo> getUserAccountInfo() const {
            std::scoped_lock guard(mutex_);
            return userAccountInfo_;
        }

        void blockAlice(const std::string& source, const std::optional<std::string>& errorSound) {
            std::lock_guard<std::mutex> guard(mutex_);
            aliceBlockers_[source] = errorSound;
        }

        void unblockAlice(const std::string& source) {
            std::lock_guard<std::mutex> guard(mutex_);
            aliceBlockers_.erase(source);
        }

        std::map<std::string, std::optional<std::string>> getAliceBlocks() const {
            std::lock_guard<std::mutex> guard(mutex_);
            return aliceBlockers_;
        }

        void setScreenActive(bool isScreenActive) {
            std::lock_guard<std::mutex> guard(mutex_);
            isScreenActive_ = isScreenActive;
        }

        bool isScreenActive() const {
            std::lock_guard<std::mutex> guard(mutex_);
            return isScreenActive_;
        }

        void setLocation(proto::Location location) {
            std::lock_guard<std::mutex> guard(mutex_);
            location_ = std::move(location);
        }

        std::optional<proto::Location> getLocation() const {
            std::lock_guard<std::mutex> guard(mutex_);
            return location_;
        }

        void setTimezone(proto::Timezone timezone) {
            std::lock_guard<std::mutex> guard(mutex_);
            timezone_ = std::move(timezone);
        }

        std::optional<proto::Timezone> getTimezone() const {
            std::lock_guard<std::mutex> guard(mutex_);
            return timezone_;
        }

        void setWifiList(proto::WifiList wifiList) {
            std::lock_guard<std::mutex> guard(mutex_);
            wifiList_ = std::move(wifiList);
        }

        proto::WifiList getWifiList() const {
            std::lock_guard<std::mutex> guard(mutex_);
            return wifiList_;
        }

        void setClockDisplayState(proto::ClockDisplayState state) {
            std::scoped_lock<std::mutex> lock(mutex_);
            clockDisplayState_ = state;
        }

        std::optional<proto::ClockDisplayState> getClockDisplayState() const {
            std::scoped_lock<std::mutex> lock(mutex_);
            return clockDisplayState_;
        }

        void setEqualizerConfig(proto::EqualizerConfig equalizerConfig) {
            std::lock_guard<std::mutex> guard(mutex_);
            equalizerConfig_ = std::move(equalizerConfig);
        }

        std::optional<proto::EqualizerConfig> getEqualizerConfig() const {
            std::lock_guard<std::mutex> guard(mutex_);
            return equalizerConfig_;
        }

    private:
        mutable std::mutex mutex_;

        bool allowInitConfigurationStateFired_{false};
        std::set<std::string> deviceConfigSubscriptions_;
        std::set<std::string> systemConfigSubscriptions_;
        std::set<std::string> accountConfigSubscriptions_;
        // If we don't allow or prohibit updates, we'll never send these messages to quasar
        UpdateAllow allowUpdateAll_{UpdateAllow::UNKNOWN};
        UpdateAllow allowCritical_{UpdateAllow::UNKNOWN};

        std::optional<UserAccountInfo> userAccountInfo_;

        std::optional<proto::ClockDisplayState> clockDisplayState_;

        std::optional<proto::Location> location_;
        std::optional<proto::Timezone> timezone_;
        proto::WifiList wifiList_;

        std::map<std::string, std::optional<std::string>> aliceBlockers_;
        bool isScreenActive_{false};

        std::optional<proto::EqualizerConfig> equalizerConfig_;
    };

    class IOConnector {
    public:
        using OnDone = std::function<void(const proto::QuasarMessage&)>;
        using OnError = std::function<void(const std::string&)>;

        explicit IOConnector(const std::shared_ptr<ipc::IIpcFactory>& ipcFactory, YandexIOSDK* sdk);
        void start();
        void waitUntilConnected();
        void setVinsResponsePreprocessorHandler(SDKInterface::VinsResponsePreprocessorHandler value);

        void sendMessage(proto::QuasarMessage&& message);
        quasar::ipc::SharedMessage sendRequestSync(proto::QuasarMessage&& message, std::chrono::milliseconds timeout);

        IOSDKStateHolder stateHolder;

    private:
        void handleQuasarMessage(const ipc::SharedMessage& message);
        void handleClientConnection();

        std::shared_ptr<ipc::IConnector> connector_;
        YandexIOSDK* sdk_;
        SDKInterface::VinsResponsePreprocessorHandler vinsResponsePreprocessorHandler_;
    };

} // namespace YandexIO

IOConnector::IOConnector(const std::shared_ptr<ipc::IIpcFactory>& ipcFactory, YandexIOSDK* sdk)
    : sdk_(sdk)
{
    connector_ = ipcFactory->createIpcConnector("iohub_clients");
    connector_->setMessageHandler(std::bind(&IOConnector::handleQuasarMessage, this, std::placeholders::_1));
    connector_->setConnectHandler(std::bind(&IOConnector::handleClientConnection, this));
}

void IOConnector::waitUntilConnected() {
    connector_->waitUntilConnected();
}

void IOConnector::setVinsResponsePreprocessorHandler(SDKInterface::VinsResponsePreprocessorHandler value) {
    vinsResponsePreprocessorHandler_ = std::move(value);
}

void IOConnector::start() {
    connector_->connectToService();
}

void IOConnector::handleClientConnection() {
    YIO_LOG_DEBUG("Handle client connection");
    /* Make sure that in case if Quasar services crash -> start quasar will be sent on reconnect */
    if (stateHolder.getAllowInitConfigurationStateFired()) {
        proto::QuasarMessage message;
        message.mutable_io_control()->mutable_allow_init_configuration_state();

        connector_->sendMessage(std::move(message));
    }

    const auto subscriptionsToDeviceConfig = stateHolder.getDeviceConfigSubscription();
    for (auto it = subscriptionsToDeviceConfig.begin(); it != subscriptionsToDeviceConfig.end(); ++it) {
        proto::QuasarMessage message;
        message.mutable_io_control()->set_subscribe_to_device_config(TString(*it));

        connector_->sendMessage(std::move(message));
    }
    const auto subscriptionsToSystemConfig = stateHolder.getSystemConfigSubscription();
    for (auto it = subscriptionsToSystemConfig.begin(); it != subscriptionsToSystemConfig.end(); ++it) {
        proto::QuasarMessage message;
        message.mutable_io_control()->set_subscribe_to_system_config(TString(*it));

        connector_->sendMessage(std::move(message));
    }
    const auto subscriptionsToAccountConfig = stateHolder.getAccountConfigSubscription();
    for (auto it = subscriptionsToAccountConfig.begin(); it != subscriptionsToAccountConfig.end(); ++it) {
        proto::QuasarMessage message;
        message.mutable_io_control()->set_subscribe_to_account_config(TString(*it));

        connector_->sendMessage(std::move(message));
    }

    {
        auto allowUpdates = stateHolder.getAllowUpdate();
        if (allowUpdates.first != IOSDKStateHolder::UpdateAllow::UNKNOWN && allowUpdates.second != IOSDKStateHolder::UpdateAllow::UNKNOWN) {
            YIO_LOG_DEBUG("Send Allow state on Client Connected: " << int(allowUpdates.first) << " " << int(allowUpdates.second));
            proto::QuasarMessage message;

            message.mutable_io_control()->mutable_allow_update()->set_for_all(allowUpdates.first == IOSDKStateHolder::UpdateAllow::ALLOWED);
            message.mutable_io_control()->mutable_allow_update()->set_for_critical(allowUpdates.second == IOSDKStateHolder::UpdateAllow::ALLOWED);

            connector_->sendMessage(std::move(message));
        }
    }
    {
        auto info = stateHolder.getUserAccountInfo();
        if (info.has_value()) {
            proto::QuasarMessage message;
            auto pinfo = message.mutable_io_control()->mutable_provide_user_account_info();
            pinfo->set_oauth_token(std::move(info->oauthToken));
            pinfo->set_uid(std::move(info->uid));
            connector_->sendMessage(std::move(message));
        }
    }
    for (const auto& [blockSource, errorSound] : stateHolder.getAliceBlocks()) {
        proto::QuasarMessage message;
        message.mutable_io_control()->mutable_assistant_blocking()->set_source(TString(blockSource));
        if (errorSound) {
            message.mutable_io_control()->mutable_assistant_blocking()->set_error_sound(TString(*errorSound));
        }
        connector_->sendMessage(std::move(message));
    }

    {
        auto isScreenActive = stateHolder.isScreenActive();
        proto::QuasarMessage message;
        message.mutable_io_control()->set_is_screen_active(isScreenActive);
        connector_->sendMessage(std::move(message));
    }

    {
        auto location = stateHolder.getLocation();
        if (location) {
            proto::QuasarMessage message;
            message.mutable_io_control()->mutable_location()->CopyFrom(*location);
            connector_->sendMessage(std::move(message));
        }
    }

    {
        auto timezone = stateHolder.getTimezone();
        if (timezone) {
            proto::QuasarMessage message;
            message.mutable_io_control()->mutable_timezone()->CopyFrom(*timezone);
            connector_->sendMessage(std::move(message));
        }
    }

    {
        auto wifiList = stateHolder.getWifiList();
        if (wifiList.hotspots_size()) {
            proto::QuasarMessage message;
            message.mutable_io_control()->mutable_wifi_list()->CopyFrom(wifiList);
            connector_->sendMessage(std::move(message));
        }
    }

    {
        auto clockDisplayState = stateHolder.getClockDisplayState();
        if (clockDisplayState) {
            proto::QuasarMessage message;
            message.mutable_io_control()->mutable_clock_display_state()->CopyFrom(*clockDisplayState);
            connector_->sendMessage(std::move(message));
        }
    }

    {
        auto equalizerConfig = stateHolder.getEqualizerConfig();
        if (equalizerConfig) {
            proto::QuasarMessage message;
            message.mutable_io_control()->mutable_equalizer_config()->CopyFrom(*equalizerConfig);
            connector_->sendMessage(std::move(message));
        }
    }
}

void IOConnector::handleQuasarMessage(const ipc::SharedMessage& message) {
    __PROTOTRACE("IOConnector", *message);
    if (message->has_io_hub_message() && message->io_hub_message().has_retrieve_state()) {
        // Hub requested to return saved state
        handleClientConnection();
    }
    if (message->has_io_event()) {
        const auto& ioEvent = message->io_event();

        if (ioEvent.has_on_notification_pending()) {
            for (const auto& wlistener : sdk_->getNotificationListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onNotificationPending();
                }
            }
        }

        if (ioEvent.has_on_notification_started()) {
            for (const auto& wlistener : sdk_->getNotificationListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onNotificationStarted();
                }
            }
        }

        if (ioEvent.has_on_notification_end()) {
            for (const auto& wlistener : sdk_->getNotificationListeners()) {
                auto listener = wlistener.lock();
                if (auto listener = wlistener.lock()) {
                    listener->onNotificationEnd();
                }
            }
        }

        if (ioEvent.has_on_start_setup()) {
            for (const auto& wlistener : sdk_->getDeviceModeListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onStartSetup(ioEvent.on_start_setup());
                }
            }
        }
        if (ioEvent.has_on_sound_data_transfer_start()) {
            for (const auto& wlistener : sdk_->getDeviceModeListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onSoundDataTransferStart();
                }
            }
        }
        if (ioEvent.has_on_connecting_to_network()) {
            for (const auto& wlistener : sdk_->getDeviceModeListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onConnectingToNetwork();
                }
            }
        }
        if (ioEvent.has_on_setup_error()) {
            for (const auto& wlistener : sdk_->getDeviceModeListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onSetupError();
                }
            }
        }
        if (ioEvent.has_on_finish_setup()) {
            for (const auto& wlistener : sdk_->getDeviceModeListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onFinishSetup();
                }
            }
        }

        if (ioEvent.has_starting_configure_success()) {
            auto state = ioEvent.starting_configure_success().should_update() ? DeviceModeObserver::ConfigurationSuccessState::HAS_UPDATE
                                                                              : DeviceModeObserver::ConfigurationSuccessState::NO_UPDATE;
            for (const auto& wlistener : sdk_->getDeviceModeListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onConfigureSuccessUpdate(state);
                }
            }
        }

        if (ioEvent.has_on_alarm_enqueued()) {
            auto alarmType = convertAlarmType(ioEvent.on_alarm_enqueued().type());
            auto alarmId = ioEvent.on_alarm_enqueued().id();
            for (const auto& wlistener : sdk_->getAlarmListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onAlarmEnqueued(alarmType, alarmId);
                }
            }
        }
        if (ioEvent.has_on_alarm_started()) {
            auto alarmType = convertAlarmType(ioEvent.on_alarm_started());
            for (const auto& wlistener : sdk_->getAlarmListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onAlarmStarted(alarmType);
                }
            }
        }
        if (ioEvent.has_on_alarm_stopped()) {
            auto alarmType = convertAlarmType(ioEvent.on_alarm_stopped().type());
            auto hasRemainingMedia = ioEvent.on_alarm_stopped().has_remaining_media();
            for (const auto& wlistener : sdk_->getAlarmListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onAlarmStopped(alarmType, hasRemainingMedia);
                }
            }
        }
        if (ioEvent.has_on_alarm_stop_remaining_media()) {
            for (const auto& wlistener : sdk_->getAlarmListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onAlarmStopRemainingMedia();
                }
            }
        }
        if (ioEvent.has_on_alarms_settings_changed()) {
            auto alarmSettings = convertAlarmSettings(ioEvent.on_alarms_settings_changed());
            for (const auto& wlistener : sdk_->getAlarmListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onAlarmsSettingsChanged(alarmSettings);
                }
            }
        }
        if (ioEvent.has_on_conversation_error()) {
            for (const auto& wlistener : sdk_->getSDKStateListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onConversationError();
                }
            }
        }

        if (ioEvent.has_on_media_request()) {
            for (const auto& wlistener : sdk_->getMediaListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onMediaRequest(MediaObserver::ContentType::MUSIC);
                }
            }
        }
        if (ioEvent.has_on_media_paused()) {
            for (const auto& wlistener : sdk_->getMediaListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onMediaPaused(convertContentType(ioEvent.on_media_paused()));
                }
            }
        }
        if (ioEvent.has_on_media_started()) {
            for (const auto& wlistener : sdk_->getMediaListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onMediaStarted(convertContentType(ioEvent.on_media_started()));
                }
            }
        }
        if (ioEvent.has_on_media_resumed()) {
            for (const auto& wlistener : sdk_->getMediaListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onMediaResumed(convertContentType(ioEvent.on_media_resumed()));
                }
            }
        }
        if (ioEvent.has_on_media_error()) {
            for (const auto& wlistener : sdk_->getMediaListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onMediaError(convertContentType(ioEvent.on_media_error()));
                }
            }
        }

        if (ioEvent.has_bluetooth_request()) {
            const auto& bluetoothRequest = ioEvent.bluetooth_request();
            const auto listeners = sdk_->getBluetoothMediaListeners();
            if (bluetoothRequest.has_sink_request()) {
                const auto& sinkRequest = bluetoothRequest.sink_request();
                if (sinkRequest.has_pause()) {
                    for (const auto& wlistener : listeners) {
                        if (auto listener = wlistener.lock()) {
                            listener->onBtPause();
                        }
                    }
                }
                if (sinkRequest.has_play()) {
                    for (const auto& wlistener : listeners) {
                        if (auto listener = wlistener.lock()) {
                            listener->onBtResume();
                        }
                    }
                }
                if (sinkRequest.has_next()) {
                    for (const auto& wlistener : listeners) {
                        if (auto listener = wlistener.lock()) {
                            listener->onBtNext();
                        }
                    }
                }
                if (sinkRequest.has_prev()) {
                    for (const auto& wlistener : listeners) {
                        if (auto listener = wlistener.lock()) {
                            listener->onBtPrev(sinkRequest.prev().forced());
                        }
                    }
                }
            }
            if (bluetoothRequest.has_disconnect_all()) {
                for (const auto& wlistener : listeners) {
                    if (auto listener = wlistener.lock()) {
                        listener->onBtDisconnectAll();
                    }
                }
            }
            if (bluetoothRequest.has_take_audio_focus()) {
                for (const auto& wlistener : listeners) {
                    if (auto listener = wlistener.lock()) {
                        listener->onBtTakeAudioFocus();
                    }
                }
            }
            if (bluetoothRequest.has_free_audio_focus()) {
                for (const auto& wlistener : listeners) {
                    if (auto listener = wlistener.lock()) {
                        listener->onBtFreeAudioFocus();
                    }
                }
            }
        }
        if (ioEvent.has_last_update_info()) {
            const auto& lastUpdateInfo = ioEvent.last_update_info();
            for (const auto& wlistener : sdk_->getUpdateStateListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onLastUpdateInfo(lastUpdateInfo.update_was_attempted(), lastUpdateInfo.update_succeeded());
                }
            }
        }
        if (ioEvent.has_external_command()) {
            for (const auto& wlistener : sdk_->getDirectiveListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onDirective(
                        ioEvent.external_command().name(),
                        ioEvent.external_command().vins_request_id(),
                        ioEvent.external_command().payload());
                }
            }
        }
        if (ioEvent.has_device_config()) {
            for (const auto& wlistener : sdk_->getBackendConfigListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onDeviceConfig(ioEvent.device_config().name(), ioEvent.device_config().value());
                }
            }
        }
        if (ioEvent.has_backend_config()) {
            for (const auto& wlistener : sdk_->getBackendConfigListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onSystemConfig(ioEvent.backend_config().name(), ioEvent.backend_config().value());
                }
            }
        }
        if (ioEvent.has_account_config()) {
            for (const auto& wlistener : sdk_->getBackendConfigListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onAccountConfig(ioEvent.account_config().name(), ioEvent.account_config().value());
                }
            }
        }
        if (ioEvent.has_get_sync_info_status()) {
            const auto protoStatus = ioEvent.get_sync_info_status().status();
            const auto status = (protoStatus == proto::IOEvent::GetSyncInfoRequestStatus::OK) ? BackendConfigObserver::BackendConfigHandleStatus::OK : BackendConfigObserver::BackendConfigHandleStatus::NETWORK_ERROR;
            for (const auto& wlistener : sdk_->getBackendConfigListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onBackendConfigHandleStatus(status);
                }
            }
        }
        if (ioEvent.has_push_notification()) {
            for (const auto& wlistener : sdk_->getPushNotificationListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onPushNotification(ioEvent.push_notification().operation(), ioEvent.push_notification().message());
                }
            }
        }
        if (ioEvent.has_authentication_status()) {
            for (const auto& wlistener : sdk_->getAuthListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onAuthenticationStatus(ioEvent.authentication_status().oauth_code(),
                                                     ioEvent.authentication_status().is_ok(),
                                                     ioEvent.authentication_status().response());
                }
            }
        }
        if (ioEvent.has_on_invalid_oauth_token()) {
            for (const auto& wlistener : sdk_->getAuthListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onInvalidOAuthToken(ioEvent.on_invalid_oauth_token());
                }
            }
        }

        if (ioEvent.has_on_invalid_authentication()) {
            for (const auto& wlistener : sdk_->getAuthListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onInvalidAuthentication(ioEvent.on_invalid_authentication());
                }
            }
        }

        if (ioEvent.has_sdk_state()) {
            SDKState st;
            st.aliceState = convertAliceState(ioEvent.sdk_state().alice_state());
            st.isAlarmPlaying = ioEvent.sdk_state().alarms_state().alarm_playing();
            st.isTimerPlaying = ioEvent.sdk_state().alarms_state().timer_playing();
            st.setAlarmsTimersCount = ioEvent.sdk_state().alarms_state().set_alarms_timers_count();
            st.iCalendarState = ioEvent.sdk_state().icalendar_state();
            st.timers = convertTimersState(ioEvent.sdk_state().timers_timings());
            st.isReminderPlaying = ioEvent.sdk_state().reminder_state().reminder_playing();
            st.isDoNotDisturbMode = ioEvent.sdk_state().do_not_disturb_state().is_dnd_mode();
            st.configurationState = convertConfigurationState(ioEvent.sdk_state().configuration_state());
            st.updateState = convertUpdateState(ioEvent.sdk_state().update_state());
            st.wifiState = convertWifiState(ioEvent.sdk_state().wifi_status());
            st.wifiList = YandexIO::convertWifiListFromProto(ioEvent.sdk_state().wifi_list());
            st.notificationState = convertNotificationState(ioEvent.sdk_state().notification_state());
            st.playerState = convertPlayerState(ioEvent.sdk_state().app_state());
            st.screenState = convertScreenState(ioEvent.sdk_state().app_state());
            st.callState = convertCallState(ioEvent.sdk_state().calld_state());
            st.ntpdState = convertNtpdState(ioEvent.sdk_state().ntpd_state());
            st.iotState = convertIotState(ioEvent.sdk_state().iot_state());
            st.multiroomState = convertMultiroomState(ioEvent.sdk_state().multiroom_state());
            st.allStartupSettings = convertAllStartupSettings(ioEvent.sdk_state().all_startup_settings());
            for (const auto& wlistener : sdk_->getSDKStateListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onSDKState(st);
                }
            }
        }

        if (ioEvent.has_spectrum()) {
            const auto spectrumListeners = sdk_->getSpectrumListeners();

            if (!spectrumListeners.empty()) {
                const auto& spectrum = ioEvent.spectrum();

                SpectrumObserver::SpectrumType type;
                switch (spectrum.type()) {
                    case proto::IOEvent::Spectrum::MUSIC: {
                        type = SpectrumObserver::SpectrumType::MUSIC;
                        break;
                    }
                    case proto::IOEvent::Spectrum::VOICE: {
                        type = SpectrumObserver::SpectrumType::VOICE;
                        break;
                    }
                }

                if (spectrum.event() == proto::IOEvent::Spectrum::SPECTRUM) {
                    SpectrumObserver::SpectrumFrame frame;
                    frame.runningTime = std::chrono::nanoseconds(spectrum.running_time_ns());
                    frame.duration = std::chrono::nanoseconds(spectrum.duration_ns());
                    frame.rate = spectrum.rate();
                    frame.magnitudes.reserve(spectrum.magnitudes_size());
                    for (int i = 0; i < spectrum.magnitudes_size(); i++) {
                        frame.magnitudes.push_back(spectrum.magnitudes(i));
                    }
                    frame.threshold = spectrum.threshold();

                    for (const auto& wlistener : spectrumListeners) {
                        if (auto listener = wlistener.lock()) {
                            listener->onSpectrum(spectrum.player_instance_id(), frame, type);
                        }
                    }
                } else {
                    for (const auto& wlistener : spectrumListeners) {
                        if (auto listener = wlistener.lock()) {
                            switch (spectrum.event()) {
                                case proto::IOEvent::Spectrum::START:
                                    listener->onSpectrumStart(spectrum.player_instance_id(), type);
                                    break;
                                case proto::IOEvent::Spectrum::PAUSE:
                                    listener->onSpectrumPause(spectrum.player_instance_id(), type);
                                    break;
                                case proto::IOEvent::Spectrum::RESUME:
                                    listener->onSpectrumResume(spectrum.player_instance_id(), type);
                                    break;
                                default:
                                    break;
                            }
                        }
                    }
                }
            }
        }

        if (ioEvent.has_on_media_switched_forward()) {
            for (const auto& wlistener : sdk_->getMediaListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onMediaSwitchedForward(convertContentType(ioEvent.on_media_switched_forward()));
                }
            }
        }

        if (ioEvent.has_on_media_switched_backward()) {
            for (const auto& wlistener : sdk_->getMediaListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onMediaSwitchedBackward(convertContentType(ioEvent.on_media_switched_backward()));
                }
            }
        }

        if (ioEvent.has_on_media_liked()) {
            for (const auto& wlistener : sdk_->getMediaListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onMediaLiked(convertContentType(ioEvent.on_media_liked()));
                }
            }
        }

        if (ioEvent.has_on_media_disliked()) {
            for (const auto& wlistener : sdk_->getMediaListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onMediaDisliked(convertContentType(ioEvent.on_media_disliked()));
                }
            }
        }

        if (ioEvent.has_sound_command()) {
            for (const auto& wlistener : sdk_->getSoundCommandListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onCommand(ioEvent.sound_command());
                }
            }
        }

        if (ioEvent.has_on_music_state_changed() && ioEvent.on_music_state_changed().has_music_state()) {
            auto protoState = ioEvent.on_music_state_changed().music_state();
            MusicStateObserver::MusicState musicState{
                .isPaused = protoState.is_paused(),
                .title = protoState.title(),
                .artists = protoState.artists(),
                .coverUri = protoState.cover_uri(),
                .durationMs = protoState.duration_ms(),
                .progress = protoState.progress(),
                .nextTrackType = convertTrackType(protoState.next_track_type()),
            };
            for (const auto& wlistener : sdk_->getMusicStateListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onMusicStateChanged(musicState);
                }
            }
        }

        if (ioEvent.has_on_audio_client_event()) {
            const auto audioClientEvent = convertAudioClientEventFromProto(ioEvent.on_audio_client_event());
            for (const auto& wlistener : sdk_->getAudioClientEventListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onAudioClientEvent(audioClientEvent);
                }
            }
        }

        if (ioEvent.has_brick_info()) {
            const auto& brickInfo = ioEvent.brick_info();
            for (const auto& wlistener : sdk_->getBrickStatusListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onBrickStatusChanged(
                        convertBrickStatus(brickInfo.brick_status()),
                        brickInfo.status_url(),
                        brickInfo.subscription_mode());
                }
            }
        }

        if (ioEvent.has_vins_response_preprocess_request()) {
            if (vinsResponsePreprocessorHandler_ != nullptr) {
                std::string result = vinsResponsePreprocessorHandler_(ioEvent.vins_response_preprocess_request());

                proto::QuasarMessage response;
                response.set_request_id(message->request_id());
                response.mutable_io_control()->set_vins_response_preprocess_response(std::move(result));
                connector_->sendMessage(std::move(response));
            }
        }

        if (ioEvent.has_device_group_state()) {
            const auto deviceGroupState = convertDeviceGroupStateFromProto(ioEvent.device_group_state());
            for (const auto& wlistener : sdk_->getDeviceGroupStateListeners()) {
                if (auto listener = wlistener.lock()) {
                    listener->onDeviceGroupState(deviceGroupState);
                }
            }
        }

        if (ioEvent.has_glagol_discovery_result()) {
            for (const auto& wlistener : sdk_->getDiscoveryListeners()) {
                const auto result = convertDiscoveryResultFromProto(ioEvent.glagol_discovery_result());
                if (auto listener = wlistener.lock()) {
                    listener->onDiscoveryResult(result);
                }
            }
        }
    }
}

void IOConnector::sendMessage(proto::QuasarMessage&& message) {
    connector_->sendMessage(std::move(message));
}

quasar::ipc::SharedMessage IOConnector::sendRequestSync(proto::QuasarMessage&& message, std::chrono::milliseconds timeout) {
    return connector_->sendRequestSync(std::move(message), timeout);
}

YandexIOSDK::YandexIOSDK()
{
    // No operations.
}

YandexIOSDK::~YandexIOSDK() {
    if (worker_) {
        worker_->destroy();
    }
}

void YandexIOSDK::init(const std::shared_ptr<quasar::ipc::IIpcFactory>& ipcFactory,
                       const std::string& processName,
                       const std::string& deviceId) {
    connector_ = std::make_shared<IOConnector>(ipcFactory, this);
    worker_ = std::make_shared<NamedCallbackQueue>("YandexIOSDK-" + processName);
    remotingWrapper_ = std::make_unique<RemotingConnectorWrapper>(*ipcFactory, "aliced", worker_, "SdkRouter-" + processName);

    initCapabilities();
    initEndpointStorage(processName, deviceId);

    connector_->start();
    remotingWrapper_->start();
}

void YandexIOSDK::initCapabilities() {
    auto aliceCapability = std::make_shared<AliceCapabilityProxy>(remotingWrapper_->getRemotingRegistry(), worker_);
    aliceCapability->init();
    aliceCapability_ = std::move(aliceCapability);

    auto filePlayerCapability = std::make_shared<FilePlayerCapabilityProxy>(remotingWrapper_->getRemotingRegistry(), worker_);
    filePlayerCapability->init();
    filePlayerCapability_ = std::move(filePlayerCapability);

    auto alarmCapability = std::make_shared<AlarmCapabilityProxy>(remotingWrapper_->getRemotingRegistry(), worker_);
    alarmCapability->init();
    alarmCapability_ = std::move(alarmCapability);

    auto playbackControlCapability = std::make_shared<PlaybackControlCapabilityProxy>(remotingWrapper_->getRemotingRegistry(), worker_);
    playbackControlCapability->init();
    playbackControlCapability_ = std::move(playbackControlCapability);

    auto activationSpotterCapability = SpotterCapabilityProxy::createActivation(remotingWrapper_->getRemotingRegistry(), worker_);
    activationSpotterCapability->init();
    activationSpotterCapability_ = std::move(activationSpotterCapability);
    auto commandSpotterCapability = SpotterCapabilityProxy::createCommand(remotingWrapper_->getRemotingRegistry(), worker_);
    commandSpotterCapability->init();
    commandSpotterCapability_ = std::move(commandSpotterCapability);
    auto naviOldSpotterCapability = SpotterCapabilityProxy::createNaviOld(remotingWrapper_->getRemotingRegistry(), worker_);
    naviOldSpotterCapability->init();
    naviOldSpotterCapability_ = std::move(naviOldSpotterCapability);

    auto deviceStateCapability = std::make_shared<DeviceStateCapabilityProxy>(remotingWrapper_->getRemotingRegistry(), worker_);
    deviceStateCapability->init();
    deviceStateCapability_ = std::move(deviceStateCapability);
}

void YandexIOSDK::initEndpointStorage(const std::string& storageName, const std::string& deviceId)
{
    auto storage = std::make_shared<EndpointStorageProxy>(storageName, deviceId, worker_, remotingWrapper_->getRemotingRegistry());
    storage->start();
    endpointStorage_ = std::move(storage);
}

void YandexIOSDK::ensureInitialized() const {
    if (Y_UNLIKELY(!connector_)) {
        YIO_LOG_ERROR_EVENT("YandexIOSDK.NotInitialized", "YandexIOSDK is not initialized yet");
        std::abort();
    }
}

void YandexIOSDK::waitUntilConnected() {
    ensureInitialized();

    connector_->waitUntilConnected();
}

void YandexIOSDK::allowInitConfigurationState() {
    ensureInitialized();

    proto::QuasarMessage message;
    message.mutable_io_control()->mutable_allow_init_configuration_state();

    connector_->sendMessage(std::move(message));
    /* Connector will send start_quasar message to DeviceContexts on connect/reconnect */
    connector_->stateHolder.setAllowInitConfigurationStateFired(true);
}

void YandexIOSDK::registerDeviceInBackend(std::string oauthToken, std::string uid) {
    ensureInitialized();

    proto::QuasarMessage message;
    auto info = message.mutable_io_control()->mutable_register_device_in_backend();
    info->set_oauth_token(std::move(oauthToken));
    info->set_uid(std::move(uid));
    connector_->sendMessage(std::move(message));
}

RegistrationResult YandexIOSDK::registerDeviceInBackendSync(std::string oauthToken, std::string uid) {
    ensureInitialized();

    proto::QuasarMessage message;
    auto info = message.mutable_io_control()->mutable_register_device_in_backend();
    info->set_oauth_token(std::move(oauthToken));
    info->set_uid(std::move(uid));

    try {
        const auto response = connector_->sendRequestSync(std::move(message), REGISTRATION_TIMEOUT);

        if (response->has_io_event() && response->io_event().has_registration_result()) {
            const auto registrationResult = response->io_event().registration_result();
            return RegistrationResult(registrationResult.is_success(),
                                      registrationResult.response_code(),
                                      registrationResult.error_message());
        } else {
            YIO_LOG_WARN("Got unexpected message instead of registration result");
            return RegistrationResult(false, -1, "Got Unexpected message instead of registration result");
        }
    } catch (const std::exception& ex) {
        return RegistrationResult(false, -1, ex.what());
    }
}

void YandexIOSDK::provideUserAccountInfo(std::string oauthToken, std::string uid) {
    ensureInitialized();

    connector_->stateHolder.setUserAccountInfo(oauthToken, uid);

    proto::QuasarMessage message;
    auto info = message.mutable_io_control()->mutable_provide_user_account_info();
    info->set_oauth_token(std::move(oauthToken));
    info->set_uid(std::move(uid));
    connector_->sendMessage(std::move(message));
}

void YandexIOSDK::revokeUserAccountInfo() {
    ensureInitialized();

    connector_->stateHolder.removeUserAccountInfo();

    proto::QuasarMessage message;
    message.mutable_io_control()->mutable_revoke_user_account_info();
    connector_->sendMessage(std::move(message));
}

void YandexIOSDK::approveAlarm(const std::string& alarmId) {
    ensureInitialized();

    proto::QuasarMessage message;
    message.mutable_io_control()->set_approve_alarm(TString(alarmId));

    connector_->sendMessage(std::move(message));
}

void YandexIOSDK::acceptIncomingCall() {
    ensureInitialized();

    proto::QuasarMessage message;
    message.mutable_io_control()->mutable_call_message()->mutable_accept_incoming();

    connector_->sendMessage(std::move(message));
}

void YandexIOSDK::declineIncomingCall() {
    ensureInitialized();

    proto::QuasarMessage message;
    message.mutable_io_control()->mutable_call_message()->mutable_decline_incoming();

    connector_->sendMessage(std::move(message));
}

void YandexIOSDK::declineCurrentCall() {
    ensureInitialized();

    proto::QuasarMessage message;
    message.mutable_io_control()->mutable_call_message()->mutable_decline_current();

    connector_->sendMessage(std::move(message));
}

void YandexIOSDK::addDeviceModeObserver(std::weak_ptr<DeviceModeObserver> observer) {
    std::scoped_lock lock(mutex_);
    deviceModeListeners_.push_back(std::move(observer));

    requestLatestState();
}

void YandexIOSDK::addAlarmObserver(std::weak_ptr<AlarmObserver> observer) {
    std::lock_guard lock(mutex_);
    alarmListeners_.push_back(std::move(observer));

    requestLatestState();
}

void YandexIOSDK::addMediaObserver(std::weak_ptr<MediaObserver> observer) {
    std::lock_guard lock(mutex_);
    mediaListeners_.push_back(std::move(observer));

    requestLatestState();
}

void YandexIOSDK::addUpdateObserver(std::weak_ptr<UpdateObserver> observer) {
    std::lock_guard lock(mutex_);
    updateStateListeners_.push_back(std::move(observer));

    requestLatestState();
}

void YandexIOSDK::addDirectiveObserver(std::weak_ptr<DirectiveObserver> observer) {
    std::lock_guard lock(mutex_);
    directiveListeners_.push_back(std::move(observer));

    requestLatestState();
}

void YandexIOSDK::addBackendConfigObserver(std::weak_ptr<BackendConfigObserver> observer) {
    std::lock_guard lock(mutex_);
    backendConfigListeners_.push_back(std::move(observer));

    requestLatestState();
}

void YandexIOSDK::addSDKStateObserver(std::weak_ptr<SDKStateObserver> observer) {
    std::lock_guard lock(mutex_);
    sdkStateListeners_.push_back(std::move(observer));

    requestLatestState();
}

void YandexIOSDK::addAuthObserver(std::weak_ptr<AuthObserver> observer) {
    std::lock_guard lock(mutex_);
    authListeners_.push_back(std::move(observer));

    requestLatestState();
}

void YandexIOSDK::setSetupMode() {
    proto::QuasarMessage message;
    message.mutable_io_control()->mutable_set_setup_mode();

    connector_->sendMessage(std::move(message));
}

void YandexIOSDK::stopSetupMode() {
    proto::QuasarMessage message;
    message.mutable_io_control()->mutable_stop_setup_mode();

    connector_->sendMessage(std::move(message));
}

void YandexIOSDK::toggleSetupMode() {
    proto::QuasarMessage message;
    message.mutable_io_control()->mutable_toggle_setup_mode();

    connector_->sendMessage(std::move(message));
}

void YandexIOSDK::subscribeToDeviceConfig(const std::string& configName) {
    proto::QuasarMessage message;
    message.mutable_io_control()->set_subscribe_to_device_config(TString(configName));

    connector_->sendMessage(std::move(message));

    connector_->stateHolder.addDeviceConfigSubscription(configName);
}

void YandexIOSDK::unsubscribeFromDeviceConfig(const std::string& configName) {
    proto::QuasarMessage message;
    message.mutable_io_control()->set_unsubscribe_from_device_config(TString(configName));

    connector_->sendMessage(std::move(message));

    connector_->stateHolder.deleteDeviceConfigSubscription(configName);
}

void YandexIOSDK::subscribeToSystemConfig(const std::string& configName) {
    proto::QuasarMessage message;
    message.mutable_io_control()->set_subscribe_to_system_config(TString(configName));

    connector_->sendMessage(std::move(message));

    connector_->stateHolder.addSystemConfigSubscription(configName);
}

void YandexIOSDK::unsubscribeFromSystemConfig(const std::string& configName) {
    proto::QuasarMessage message;
    message.mutable_io_control()->set_unsubscribe_from_system_config(TString(configName));

    connector_->sendMessage(std::move(message));

    connector_->stateHolder.deleteSystemConfigSubscription(configName);
}

void YandexIOSDK::subscribeToAccountConfig(const std::string& configName) {
    proto::QuasarMessage message;
    message.mutable_io_control()->set_subscribe_to_account_config(TString(configName));

    connector_->sendMessage(std::move(message));

    connector_->stateHolder.addAccountConfigSubscription(configName);
}

void YandexIOSDK::unsubscribeFromAccountConfig(const std::string& configName) {
    proto::QuasarMessage message;
    message.mutable_io_control()->set_unsubscribe_from_account_config(TString(configName));

    connector_->sendMessage(std::move(message));

    connector_->stateHolder.deleteAccountConfigSubscription(configName);
}

void YandexIOSDK::authenticate(const std::string& oauthCode) {
    proto::QuasarMessage message;
    message.mutable_io_control()->set_authenticate(TString(oauthCode));

    connector_->sendMessage(std::move(message));
}

void YandexIOSDK::reportTvPolicyInfo(const TvPolicyInfo& tvPolicyInfo) {
    proto::QuasarMessage message;
    if (tvPolicyInfo.contentSettings) {
        message.mutable_io_control()->mutable_tv_policy_info()->set_content_settings(TString(*tvPolicyInfo.contentSettings));
    }

    if (tvPolicyInfo.ageLimit) {
        message.mutable_io_control()->mutable_tv_policy_info()->set_age_limit(*tvPolicyInfo.ageLimit);
    }

    connector_->sendMessage(std::move(message));
}

void YandexIOSDK::reportClockDisplayState(ClockDisplayState clockDisplayState) {
    proto::ClockDisplayState state;
    state.set_is_clock_on(clockDisplayState.isClockOn);
    state.set_clock_enabled(clockDisplayState.clockEnabled);
    state.set_show_clock_during_dnd(clockDisplayState.showClockDuringDnd);
    state.set_brightness(clockDisplayState.brightness);

    proto::QuasarMessage message;
    message.mutable_io_control()->mutable_clock_display_state()->CopyFrom(state);

    connector_->stateHolder.setClockDisplayState(state);
    connector_->sendMessage(std::move(message));
}

void YandexIOSDK::setActiveActions(const NAlice::TDeviceState::TActiveActions& activeActions) {
    proto::QuasarMessage message;
    message.mutable_io_control()->mutable_active_actions()->CopyFrom(activeActions);
    connector_->sendMessage(std::move(message));
}

void YandexIOSDK::setActiveActionsSemanticFrames(const std::optional<std::string>& jsonPayload) {
    proto::QuasarMessage message;
    auto activeActionMsg = message.mutable_io_control()->mutable_active_action();
    if (jsonPayload) {
        activeActionMsg->set_semantic_frame_payload(TString(*jsonPayload));
    }
    connector_->sendMessage(std::move(message));
}

void YandexIOSDK::setAllowUpdate(bool allowUpdateAll, bool allowCritical) {
    proto::QuasarMessage message;
    message.mutable_io_control()->mutable_allow_update()->set_for_all(allowUpdateAll);
    message.mutable_io_control()->mutable_allow_update()->set_for_critical(allowCritical);

    connector_->stateHolder.setAllowUpdate(allowUpdateAll, allowCritical);
    connector_->sendMessage(std::move(message));
}

void YandexIOSDK::bluetoothMediaSinkPause() {
    proto::QuasarMessage message;
    message.mutable_io_control()->mutable_bluetooth_sink_event()->set_audio_event(proto::BluetoothSinkEvent::BtSinkAudioEvent::BluetoothSinkEvent_BtSinkAudioEvent_NOT_PLAYING);
    connector_->sendMessage(std::move(message));
}

void YandexIOSDK::bluetoothMediaSinkStart() {
    proto::QuasarMessage message;
    message.mutable_io_control()->mutable_bluetooth_sink_event()->set_audio_event(proto::BluetoothSinkEvent::BtSinkAudioEvent::BluetoothSinkEvent_BtSinkAudioEvent_PLAYING);
    connector_->sendMessage(std::move(message));
}

void YandexIOSDK::bluetoothSinkConnected(const std::string& networkAddr, const std::string& networkName) {
    proto::QuasarMessage message;
    auto connectionEvent = message.mutable_io_control()->mutable_bluetooth_sink_event()->mutable_connection_event();
    auto messageNetwork = connectionEvent->mutable_network();
    messageNetwork->set_address(TString(networkAddr));
    messageNetwork->set_name(TString(networkName));
    messageNetwork->set_role(proto::BluetoothNetwork::SINK);
    connectionEvent->set_connection_event(proto::BtConnection::CONNECTED);
    connector_->sendMessage(std::move(message));
}

void YandexIOSDK::bluetoothSinkDisconnected(const std::string& networkAddr, const std::string& networkName) {
    proto::QuasarMessage message;
    auto connectionEvent = message.mutable_io_control()->mutable_bluetooth_sink_event()->mutable_connection_event();
    auto messageNetwork = connectionEvent->mutable_network();
    messageNetwork->set_address(TString(networkAddr));
    messageNetwork->set_name(TString(networkName));
    messageNetwork->set_role(proto::BluetoothNetwork::SINK);
    connectionEvent->set_connection_event(proto::BtConnection::DISCONNECTED);
    connector_->sendMessage(std::move(message));
}

void YandexIOSDK::bluetoothMediaSinkTrackInfo(const std::string& title, const std::string& artist, const std::string& album, const std::string& genre, int songLenMs, int currPosMs)
{
    proto::QuasarMessage message;
    auto trackMeta = message.mutable_io_control()->mutable_bluetooth_sink_event()->mutable_bluetooth_track_meta_info();
    if (!title.empty()) {
        trackMeta->set_title(TString(title));
    }
    if (!artist.empty()) {
        trackMeta->set_artist(TString(artist));
    }
    if (!album.empty()) {
        trackMeta->set_album(TString(album));
    }
    if (!genre.empty()) {
        trackMeta->set_title(TString(genre));
    }
    if (songLenMs > 0 && currPosMs >= 0) {
        trackMeta->set_song_len_ms(songLenMs);
        trackMeta->set_curr_pos_ms(currPosMs);
    }
    connector_->sendMessage(std::move(message));
}

void YandexIOSDK::sendNumericMetrics(const std::string& key, double value) {
    proto::QuasarMessage message;
    message.mutable_io_control()->mutable_numeric_metrics()->set_key(TString(key));
    message.mutable_io_control()->mutable_numeric_metrics()->set_value(value);
    connector_->sendMessage(std::move(message));
}

void YandexIOSDK::sendCategoricalMetrics(const std::string& key, const std::string& value) {
    proto::QuasarMessage message;
    message.mutable_io_control()->mutable_categorical_metrics()->set_key(TString(key));
    message.mutable_io_control()->mutable_categorical_metrics()->set_value(TString(value));
    connector_->sendMessage(std::move(message));
}

void YandexIOSDK::notifyPreparedForNotification() {
    proto::QuasarMessage message;
    message.mutable_io_control()->mutable_prepared_for_notification();
    connector_->sendMessage(std::move(message));
}

void YandexIOSDK::blockVoiceAssistant(const std::string& sourceId, const std::optional<std::string>& errorSound) {
    proto::QuasarMessage message;
    message.mutable_io_control()->mutable_assistant_blocking()->set_source(TString(sourceId));
    if (errorSound) {
        message.mutable_io_control()->mutable_assistant_blocking()->set_error_sound(TString(*errorSound));
    }

    connector_->stateHolder.blockAlice(sourceId, errorSound);
    connector_->sendMessage(std::move(message));
}

void YandexIOSDK::unblockVoiceAssistant(const std::string& sourceId) {
    proto::QuasarMessage message;
    message.mutable_io_control()->mutable_assistant_unblocking()->set_source(TString(sourceId));

    connector_->stateHolder.unblockAlice(sourceId);
    connector_->sendMessage(std::move(message));
}

void YandexIOSDK::reportScreenActive(bool isScreenActive) {
    proto::QuasarMessage message;
    message.mutable_io_control()->set_is_screen_active(isScreenActive);
    connector_->stateHolder.setScreenActive(isScreenActive);
    connector_->sendMessage(std::move(message));
}

void YandexIOSDK::setLocation(double latitude, double longitude, std::optional<double> accuracy) {
    proto::QuasarMessage message;
    proto::Location location;
    location.set_latitude(latitude);
    location.set_longitude(longitude);
    if (accuracy) {
        location.set_precision(*accuracy);
    }
    message.mutable_io_control()->mutable_location()->CopyFrom(location);
    connector_->stateHolder.setLocation(location);
    connector_->sendMessage(std::move(message));
}

void YandexIOSDK::setTimezone(const std::string& timezone, int32_t offsetSec) {
    proto::QuasarMessage message;
    proto::Timezone tz;
    tz.set_timezone_name(TString(timezone));
    tz.set_timezone_offset_sec(offsetSec);
    message.mutable_io_control()->mutable_timezone()->CopyFrom(tz);
    connector_->stateHolder.setTimezone(tz);
    connector_->sendMessage(std::move(message));
}

void YandexIOSDK::setWifiList(const std::vector<WifiInfo>& wifiList) {
    proto::QuasarMessage message;
    auto list = YandexIO::convertWifiListToProto(wifiList);
    message.mutable_io_control()->mutable_wifi_list()->CopyFrom(list);
    connector_->stateHolder.setWifiList(list);
    connector_->sendMessage(std::move(message));
}

void YandexIOSDK::setVinsResponsePreprocessorHandler(VinsResponsePreprocessorHandler value) {
    connector_->setVinsResponsePreprocessorHandler(value);
}

void YandexIOSDK::sendVinsResponse(const std::string& responseJson) {
    ensureInitialized();

    proto::QuasarMessage message;
    message.mutable_io_control()->set_vins_response(TString(responseJson));

    connector_->sendMessage(std::move(message));
}

const std::shared_ptr<IEndpointStorage>& YandexIOSDK::getEndpointStorage() const {
    Y_VERIFY(endpointStorage_ != nullptr);
    return endpointStorage_;
}

void YandexIOSDK::setEqualizerConfig(const EqualizerInfo& equalizerInfo) {
    ensureInitialized();

    proto::EqualizerConfig equalizerConfigProto;
    for (const auto& band : equalizerInfo.bands) {
        auto bandProto = equalizerConfigProto.add_bands();
        bandProto->set_freq(band.freq);
        bandProto->set_width(band.width);
        bandProto->set_gain(band.gain);
    }
    equalizerConfigProto.set_prevent_clipping(equalizerInfo.preventClipping);

    proto::QuasarMessage message;
    message.mutable_io_control()->mutable_equalizer_config()->CopyFrom(equalizerConfigProto);
    connector_->stateHolder.setEqualizerConfig(equalizerConfigProto);
    connector_->sendMessage(std::move(message));
}

std::shared_ptr<IAlarmCapability> YandexIOSDK::getAlarmCapability() const {
    return alarmCapability_;
}

std::shared_ptr<IAliceCapability> YandexIOSDK::getAliceCapability() const {
    return aliceCapability_;
}

std::shared_ptr<IFilePlayerCapability> YandexIOSDK::getFilePlayerCapability() const {
    return filePlayerCapability_;
}

std::shared_ptr<IPlaybackControlCapability> YandexIOSDK::getPlaybackControlCapability() const {
    return playbackControlCapability_;
}

std::shared_ptr<ISpotterCapability> YandexIOSDK::getActivationSpotterCapability() const {
    return activationSpotterCapability_;
}

std::shared_ptr<ISpotterCapability> YandexIOSDK::getCommandSpotterCapability() const {
    return commandSpotterCapability_;
}

std::shared_ptr<ISpotterCapability> YandexIOSDK::getNaviOldSpotterCapability() const {
    return naviOldSpotterCapability_;
}

std::shared_ptr<IDeviceStateCapability> YandexIOSDK::getDeviceStateCapability() const {
    return deviceStateCapability_;
}

std::list<std::weak_ptr<DeviceModeObserver>> YandexIOSDK::getDeviceModeListeners() const {
    std::lock_guard lock(mutex_);
    return deviceModeListeners_;
}

std::list<std::weak_ptr<AlarmObserver>> YandexIOSDK::getAlarmListeners() const {
    std::lock_guard lock(mutex_);
    return alarmListeners_;
}

std::list<std::weak_ptr<MediaObserver>> YandexIOSDK::getMediaListeners() const {
    std::lock_guard lock(mutex_);
    return mediaListeners_;
}

std::list<std::weak_ptr<UpdateObserver>> YandexIOSDK::getUpdateStateListeners() const {
    std::lock_guard lock(mutex_);
    return updateStateListeners_;
}

std::list<std::weak_ptr<DirectiveObserver>> YandexIOSDK::getDirectiveListeners() const {
    std::lock_guard lock(mutex_);
    return directiveListeners_;
}

std::list<std::weak_ptr<BackendConfigObserver>> YandexIOSDK::getBackendConfigListeners() const {
    std::lock_guard lock(mutex_);
    return backendConfigListeners_;
}

std::list<std::weak_ptr<SDKStateObserver>> YandexIOSDK::getSDKStateListeners() const {
    std::lock_guard lock(mutex_);
    return sdkStateListeners_;
}

std::list<std::weak_ptr<AuthObserver>> YandexIOSDK::getAuthListeners() const {
    std::lock_guard lock(mutex_);
    return authListeners_;
}

std::list<std::weak_ptr<NotificationObserver>> YandexIOSDK::getNotificationListeners() const {
    std::lock_guard lock(mutex_);
    return notificationListeners_;
}

void YandexIOSDK::addNotificationObserver(std::weak_ptr<NotificationObserver> observer) {
    std::lock_guard lock(mutex_);
    notificationListeners_.push_back(std::move(observer));
}

std::list<std::weak_ptr<SpectrumObserver>> YandexIOSDK::getSpectrumListeners() const {
    std::lock_guard lock(mutex_);
    return spectrumListeners_;
}

void YandexIOSDK::addSpectrumObserver(std::weak_ptr<SpectrumObserver> observer) {
    std::lock_guard lock(mutex_);
    spectrumListeners_.push_back(std::move(observer));

    requestLatestState();
}

std::list<std::weak_ptr<SoundCommandObserver>> YandexIOSDK::getSoundCommandListeners() const {
    std::lock_guard lock(mutex_);
    return soundCommandListeners_;
}

void YandexIOSDK::addSoundCommandObserver(std::weak_ptr<SoundCommandObserver> observer) {
    std::lock_guard lock(mutex_);
    soundCommandListeners_.push_back(std::move(observer));

    requestLatestState();
}

std::list<std::weak_ptr<MusicStateObserver>> YandexIOSDK::getMusicStateListeners() const {
    std::lock_guard lock(mutex_);
    return musicStateListeners_;
}

void YandexIOSDK::addMusicStateObserver(std::weak_ptr<MusicStateObserver> observer) {
    std::lock_guard lock(mutex_);
    musicStateListeners_.push_back(std::move(observer));

    requestLatestState();
}

std::list<std::weak_ptr<AudioClientEventObserver>> YandexIOSDK::getAudioClientEventListeners() const {
    std::lock_guard lock(mutex_);
    return audioClientEventListeners_;
}

void YandexIOSDK::addAudioClientEventObserver(std::weak_ptr<AudioClientEventObserver> observer) {
    std::lock_guard lock(mutex_);
    audioClientEventListeners_.push_back(std::move(observer));

    requestLatestState();
}

std::list<std::weak_ptr<BluetoothMediaObserver>> YandexIOSDK::getBluetoothMediaListeners() const {
    std::scoped_lock guard(mutex_);
    return bluetoothMediaListeners_;
}

void YandexIOSDK::addBluetoothMediaObserver(std::weak_ptr<BluetoothMediaObserver> observer) {
    std::scoped_lock guard(mutex_);
    bluetoothMediaListeners_.push_back(std::move(observer));

    requestLatestState();
}

std::list<std::weak_ptr<BrickStatusObserver>> YandexIOSDK::getBrickStatusListeners() const {
    std::scoped_lock guard(mutex_);
    return brickStatusListeners_;
}

void YandexIOSDK::addBrickStatusObserver(std::weak_ptr<BrickStatusObserver> observer) {
    std::scoped_lock guard(mutex_);
    brickStatusListeners_.push_back(std::move(observer));

    requestLatestState();
}

std::list<std::weak_ptr<PushNotificationObserver>> YandexIOSDK::getPushNotificationListeners() const {
    std::scoped_lock guard(mutex_);
    return pushNotificationListeners_;
}

void YandexIOSDK::addPushNotificationObserver(std::weak_ptr<PushNotificationObserver> observer) {
    std::scoped_lock guard(mutex_);
    pushNotificationListeners_.push_back(std::move(observer));

    requestLatestState();
}

std::list<std::weak_ptr<DiscoveryObserver>> YandexIOSDK::getDiscoveryListeners() const {
    std::lock_guard guard(mutex_);
    return discoveryListeners_;
}

void YandexIOSDK::addDiscoveryObserver(std::weak_ptr<DiscoveryObserver> observer) {
    std::lock_guard guard(mutex_);
    discoveryListeners_.push_back(std::move(observer));

    requestLatestState();
}

void YandexIOSDK::provideState(const yandex_io::proto::TDeviceStatePart& statePart) {
    proto::QuasarMessage message;
    message.mutable_io_control()->mutable_provide_device_state()->CopyFrom(statePart);
    connector_->sendMessage(std::move(message));
}

void YandexIOSDK::provideMediaDeviceIdentifier(const NAlice::TClientInfoProto::TMediaDeviceIdentifier& identifier) {
    proto::QuasarMessage message;
    message.mutable_io_control()->mutable_media_device_identifier()->CopyFrom(identifier);
    connector_->sendMessage(std::move(message));
}

std::list<std::weak_ptr<DeviceGroupStateObserver>> YandexIOSDK::getDeviceGroupStateListeners() const {
    std::scoped_lock guard(mutex_);
    return deviceGroupStateListeners_;
}

void YandexIOSDK::addDeviceGroupStateObserver(std::weak_ptr<DeviceGroupStateObserver> observer) {
    std::scoped_lock guard(mutex_);
    deviceGroupStateListeners_.push_back(std::move(observer));

    requestLatestState();
}

void YandexIOSDK::requestLatestState() const {
    if (connector_) {
        proto::QuasarMessage message;
        message.mutable_io_hub_message()->mutable_retrieve_state();

        connector_->sendMessage(std::move(message));
    }
}
