#include "adb_service.h"

#include <yandex_io/libs/base/named_callback_queue.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

using namespace quasar;

namespace {

    struct DefaultApi: public AdbService::Api {
        int system(const std::string& commandLine) const override {
            return ::system(commandLine.c_str());
        }
    };

} // namespace

AdbService::AdbService(
    std::shared_ptr<YandexIO::IDevice> device,
    std::shared_ptr<IUserConfigProvider> userConfigProvider)
    : AdbService(
          std::move(device),
          std::move(userConfigProvider),
          std::make_shared<NamedCallbackQueue>("adbd"),
          std::make_shared<DefaultApi>())
{
}

AdbService::AdbService(
    std::shared_ptr<YandexIO::IDevice> device,
    std::shared_ptr<IUserConfigProvider> userConfigProvider,
    std::shared_ptr<ICallbackQueue> lifecycle,
    std::shared_ptr<Api> api)
    : lifecycle_(std::move(lifecycle))
    , api_(std::move(api))
    , quasmodromActivateCmd_(tryGetString(device->configuration()->getServiceConfig("adbd"), "quasmodromActivateCmd", ""))
    , quasmodromNetworkActivateCmd_(tryGetString(device->configuration()->getServiceConfig("adbd"), "quasmodromNetworkActivateCmd", ""))
    , quasmodromDeactivateCmd_(tryGetString(device->configuration()->getServiceConfig("adbd"), "quasmodromDeactivateCmd", ""))
    , allDeactivateCmd_(tryGetString(device->configuration()->getServiceConfig("adbd"), "allDeactivateCmd", ""))
    , userConfigProvider_(std::move(userConfigProvider))
{
}

AdbService::~AdbService()
{
    lifetime_.die();
}

void AdbService::start()
{
    YIO_LOG_INFO("Adb service started");
    userConfigProvider_->jsonChangedSignal(IUserConfigProvider::ConfigScope::SYSTEM, "adbd").connect([this](auto adbdConfig) {
        YIO_LOG_INFO("Read quasmodrom config section \"adbd\": "
                     << "forceDisable=" << (adbdConfig->isMember("forceDisable") ? (getBool(*adbdConfig, "forceDisable") ? "true" : "false") : "undefined") << ", "
                     << "remoteActivation=" << (adbdConfig->isMember("remoteActivation") ? (getBool(*adbdConfig, "remoteActivation") ? "true" : "false") : "undefined") << ", "
                     << "network=" << (adbdConfig->isMember("network") ? (getBool(*adbdConfig, "network") ? "true" : "false") : "undefined"));

        bool forceDisable = tryGetBool(*adbdConfig, "forceDisable", false);
        bool remoteActivation = tryGetBool(*adbdConfig, "remoteActivation", false);
        bool network = tryGetBool(*adbdConfig, "network", false);

        if (forceDisable) {
            if (!config_ || !config_->forceDisable) {
                YIO_LOG_INFO("Force disable adb service. Running command: " << allDeactivateCmd_);
                std::ignore = api_->system(allDeactivateCmd_);
            }
        } else if (remoteActivation) {
            if (!config_ || config_->forceDisable || !config_->remoteActivation || config_->network != network) {
                const auto& command = network ? quasmodromNetworkActivateCmd_ : quasmodromActivateCmd_;
                YIO_LOG_INFO("Activate adb service. Running command: " << command);
                std::ignore = api_->system(command);
            }
        } else {
            if (!config_ || config_->remoteActivation) {
                YIO_LOG_INFO("Deactivate adb service. Running command: " << quasmodromDeactivateCmd_);
                std::ignore = api_->system(quasmodromDeactivateCmd_);
            }
        }
        if (!config_) {
            config_.emplace(Config{});
        }
        config_->forceDisable = forceDisable;
        config_->remoteActivation = remoteActivation;
        config_->network = network;
    }, lifetime_, lifecycle_);
}

void AdbService::stop() {
    lifetime_.die();
}
