#include "alarm_endpoint.h"

#include "alarm_utils.h"

#include <yandex_io/libs/base/persistent_file.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/debug.h>
#include <yandex_io/libs/telemetry/telemetry.h>
#include <yandex_io/protos/quasar_proto.pb.h>

#include <ctime>
#include <memory>

YIO_DEFINE_LOG_MODULE("alarm");

using namespace quasar;

namespace {
    constexpr std::chrono::milliseconds CHECK_TIMEOUT_MS{500};
    constexpr std::chrono::milliseconds MAX_DELAY_MS{5000};
} // namespace

/**
 * Сервис будильников.
 *
 * Выстреливанием будильников занимается поток cronThread_. Он каждые CHECK_TIMEOUT_MS микросекунд берет из хранилища
 * событий eventStorage_ список будильников, которые должны были сработать за последние MAX_DELAY_MS микросекунд,
 * ставит их в очередь на проигрывание и рассылает другим сервисам сообщения alarm_fired по каждому из них.
 */
AlarmEndpoint::AlarmEndpoint(
    std::shared_ptr<YandexIO::IDevice> device,
    std::shared_ptr<ipc::IIpcFactory> ipcFactory,
    std::shared_ptr<IAuthProvider> authProvider,
    std::shared_ptr<IStereoPairProvider> stereoPairProvider)
    : device_(std::move(device))
    , authProvider_(std::move(authProvider))
    , stereoPairProvider_(std::move(stereoPairProvider))
    , deviceContext_(std::make_shared<YandexIO::DeviceContext>(ipcFactory, [this] { handleSDKConnect(); }, false))
    , server_(ipcFactory->createIpcServer("alarmd"))
    , syncdConnector_(ipcFactory->createIpcConnector("syncd"))
    , dbFileName_(getString(device_->configuration()->getServiceConfig("alarmd"), "dbFileName"))
    , mediaAlarmSettingJsonFile_(getString(device_->configuration()->getServiceConfig("alarmd"), "mediaAlarmSettingJsonFile"))
    , alarmsSettingsFile_(getString(device_->configuration()->getServiceConfig("alarmd"), "alarmsSettingsFile"))
    , alarmPlayer_(device_, ipcFactory, deviceContext_, server_)
    , randomGenerator_(std::random_device{}())
    , isRunning_(true)
{
    server_->setMessageHandler(std::bind(&AlarmEndpoint::processQuasarMessage, this, std::placeholders::_1, std::placeholders::_2));
    server_->setClientConnectedHandler([this](auto& connection) {
        proto::QuasarMessage message;
        message.mutable_alarms_state()->set_icalendar_state(eventStorage_.getICalendarState());

        if (const auto alarm = alarmPlayer_.getPlayingAlarm(); !alarm.id().empty()) {
            auto msg = ipc::buildMessage([&alarm](auto& msg) {
                auto* event = msg.mutable_alarm_event();
                event->mutable_alarm_fired()->CopyFrom(alarm);
            });
            connection.send(std::move(msg));
        }

        std::string mediaAlarmSettingInfo = mediaAlarmSetting_.getInfo();
        if (!mediaAlarmSettingInfo.empty()) {
            message.mutable_alarms_state()->set_media_alarm_setting(std::move(mediaAlarmSettingInfo));
        }
        {
            std::lock_guard<std::mutex> guard(alarmsSettingsMutex_);
            message.mutable_alarms_state()->mutable_alarms_settings()->CopyFrom(alarmsSettings_);
        }
        message.mutable_timers_state()->CopyFrom(eventStorage_.getTimersState());
        message.mutable_set_alarms_state()->CopyFrom(eventStorage_.getAlarmsState());

        connection.send(std::move(message));
    });
    syncdConnector_->setMessageHandler([this](const auto& message) {
        if (message->has_user_config_update()) {
            handleSystemConfigUpdate(message->user_config_update().config());
        }
    });

    const auto config = device_->configuration()->getServiceConfig("alarmd");

    defaultSettings_.maxAlarmDelaySec = tryGetInt64(config, "maxAlarmDelaySec", 20);
    defaultSettings_.alwaysStopsMedia = tryGetBool(config, "alwaysStopsMedia", false);
    // required for tests, 0 for production

    currentSettings_ = defaultSettings_;
    if (currentSettings_.maxAlarmDelaySec) {
        changeAlarmsDelay(currentSettings_.maxAlarmDelaySec);
    }

    if (fileExists(mediaAlarmSettingJsonFile_)) {
        try {
            const std::string fileContent = getFileContent(mediaAlarmSettingJsonFile_);
            YIO_LOG_INFO("Use MediaAlarmSetting from file cache:" << fileContent);
            const Json::Value mediaAlarmSettingJson = parseJson(fileContent);
            mediaAlarmSetting_.setInfo(jsonToString(mediaAlarmSettingJson));
        } catch (const Json::Exception& e) {
            YIO_LOG_ERROR_EVENT("AlarmEndpoint.BadJson.MediaAlarmSetting", "Can't parse Sound Alarm Json file: " << mediaAlarmSettingJsonFile_ << ". Error: " << e.what());
            /* json file was corrupted. It's better to remove it. User should reset setting */
            std::remove(mediaAlarmSettingJsonFile_.c_str());
        }
    }

    alarmsSettings_.set_max_volume_level(config["finishAlarmUserVolume"].asInt());
    alarmsSettings_.set_min_volume_level(config["startAlarmUserVolume"].asInt());
    alarmsSettings_.set_volume_raise_step_ms(config["alarmVolumeStepMs"].asInt());
    if (fileExists(alarmsSettingsFile_)) {
        proto::AlarmsSettings alarmsSettings;
        Y_PROTOBUF_SUPPRESS_NODISCARD alarmsSettings.ParseFromString(getFileContent(alarmsSettingsFile_));
        /* Currently only Max Volume can be set up by user */
        if (alarmsSettings.has_max_volume_level()) {
            alarmsSettings_.set_max_volume_level(alarmsSettings.max_volume_level());
        }
    }

    deviceContext_->connectToSDK();

    eventStorage_.loadEvents(dbFileName_);
    parseIcalendar(); // we need to parse icalendar once to send init state. Then parse with periodic executor
    if (eventStorage_.removeExpiredEvents(std::chrono::seconds(time(nullptr)), MAX_DELAY_MS)) {
        eventStorage_.saveEvents(dbFileName_);
    }
    reportState("init_alarm_state");

    maxICalendarParseInterval_ = std::chrono::milliseconds(tryGetInt(config, "iCalendarParseIntervalMs", maxICalendarParseInterval_.count()));
    currentIcalendarParseInterval_ = std::chrono::milliseconds(tryGetInt(config, "startIcalendarParseIntervalMs", currentIcalendarParseInterval_.count()));
    auto periodicParseIcalendar = [this](PeriodicExecutor* executor) {
        parseIcalendar();
        // After start we need to parse more often to be sure that we reparse calendar after NTP time sync
        currentIcalendarParseInterval_ = std::min(currentIcalendarParseInterval_ * 2, maxICalendarParseInterval_);
        executor->setPeriodTime(currentIcalendarParseInterval_);
    };
    iCalendarParseExecutor_ = std::make_unique<PeriodicExecutor>(
        PeriodicExecutor::PECallback(periodicParseIcalendar),
        currentIcalendarParseInterval_,
        PeriodicExecutor::PeriodicType::SLEEP_FIRST);

    server_->listenService();
    syncdConnector_->connectToService();

    authProvider_->ownerAuthInfo().connect(
        [this](const auto& authInfo) {
            handleAccountChange(authInfo->passportUid);
        }, lifetime_);

    if (stereoPairProvider_) {
        stereoPairProvider_->stereoPairState().connect(
            [this](const auto& state) {
                bool newValue = state->isFollower();
                bool oldValue = isStereoPairFollower_.exchange(newValue);
                if (newValue && !oldValue) {
                    YIO_LOG_INFO("AlarmEndpoint switch to stereo pair follower mode and disable any alarms");
                    alarmPlayer_.cancelAlarm(true);
                } else if (oldValue && !newValue) {
                    YIO_LOG_INFO("AlarmEndpoint switch to stereo pair non follower mode and enable all alarms");
                }
            }, lifetime_);
    }
    cronThread_ = std::thread(&AlarmEndpoint::checkAlarms, this);
}

AlarmEndpoint::~AlarmEndpoint()
{
    lifetime_.die();
    isRunning_ = false;
    cronThread_.join();

    server_->shutdown();
    syncdConnector_->shutdown();

    // device context should be destroyed before players
    deviceContext_->shutdown();
}

void AlarmEndpoint::handleSDKConnect()
{
    std::lock_guard<std::mutex> guard(alarmsSettingsMutex_);
    deviceContext_->fireAlarmsSettingsChanged(alarmsSettings_);
}

void AlarmEndpoint::processQuasarMessage(const ipc::SharedMessage& sharedMessage, ipc::IServer::IClientConnection& /* connection */)
{
    const auto& message = *sharedMessage;
    if (message.has_alarm_message()) {
        processAlarmMessage(message);
    }

    if (message.has_alarms_state() && message.alarms_state().has_icalendar_state()) {
        processNewIcalendarState(message);
    }

    if (message.has_alarm_set_sound()) {
        processAlarmSetSound(message);
    } else if (message.has_alarm_reset_sound()) {
        processAlarmResetSound();
    }

    if (message.has_alarms_settings()) {
        processAlarmsSettings(message);
    }

    if (message.has_alarm_event()) {
        alarmPlayer_.onAlarmEvent(message.alarm_event());
    }

    if (onQuasarMessageReceivedCallback) {
        onQuasarMessageReceivedCallback(message); // Testing purposes only!
    }
}

void AlarmEndpoint::processAlarmMessage(const proto::QuasarMessage& message)
{
    std::lock_guard<std::mutex> lockGuard(mutex_);
    if (message.alarm_message().has_add_alarm()) {
        processAddAlarm(message.alarm_message().add_alarm());
    } else if (message.alarm_message().has_remove_alarm_id()) {
        processRemoveAlarm(message.alarm_message().remove_alarm_id());
    } else if (message.alarm_message().has_pause_alarm_id()) {
        processPauseAlarm(message.alarm_message().pause_alarm_id());
    } else if (message.alarm_message().has_resume_alarm_id()) {
        processResumeAlarm(message.alarm_message().resume_alarm_id());
    } else if (message.alarm_message().has_stop_alarm()) {
        stopAlarm(message.alarm_message().stop_alarm().stop_media());
    } else if (message.alarm_message().has_alarm_stop_directive()) {
        alarmPlayer_.stopAnyRemainingMedia();
    }

    sendState();
}

void AlarmEndpoint::processNewIcalendarState(const proto::QuasarMessage& message)
{
    YIO_LOG_INFO("Setting new icalendar state" << message.alarms_state().icalendar_state());
    eventStorage_.setICalendarState(message.alarms_state().icalendar_state());
    parseIcalendar();
    eventStorage_.saveEvents(dbFileName_);

    auto msg = ipc::buildMessage([](auto& msg) {
        auto* event = msg.mutable_alarm_event();
        event->mutable_alarms_updated();
    });
    server_->sendToAll(msg);

    sendState();
}

void AlarmEndpoint::processAlarmSetSound(const proto::QuasarMessage& message)
{
    Json::Value setAlarmSoundJson;
    const std::string payload = message.alarm_set_sound().payload();
    try {
        setAlarmSoundJson = parseJson(payload);
    } catch (const Json::Exception& e) {
        YIO_LOG_ERROR_EVENT("AlarmEndpoint.BadJson.AlarmSound", "Incorrect setAlarmSound json: " << payload);
        return;
    }
    if (setAlarmSoundJson["server_action"].isNull() || setAlarmSoundJson["sound_alarm_setting"].isNull()) {
        YIO_LOG_WARN("SetAlarmSound json is incorrect! One of fields is missed: " << payload);
        return;
    }

    /* Save correct sound alarm setting JSON! */
    PersistentFile file(mediaAlarmSettingJsonFile_, PersistentFile::Mode::TRUNCATE);
    file.write(payload);

    std::lock_guard<std::mutex> lockGuard(mutex_);
    eventStorage_.saveEvents(dbFileName_);

    mediaAlarmSetting_.setInfo(jsonToString(setAlarmSoundJson));
    sendState();
    device_->telemetry()->reportEvent("setAlarmSoundSetting");

    YIO_LOG_INFO("User Set Sound for Alarm: " << mediaAlarmSetting_.getInfo());
}

void AlarmEndpoint::processAlarmResetSound()
{
    /* remove file cache */
    std::remove(mediaAlarmSettingJsonFile_.c_str());
    std::lock_guard<std::mutex> lockGuard(mutex_);
    mediaAlarmSetting_.clearInfo();
    eventStorage_.saveEvents(dbFileName_);
    sendState();
    device_->telemetry()->reportEvent("resetAlarmSoundSetting");
    YIO_LOG_INFO("User reset Sound Alarm. Use default alarm sound");
}

void AlarmEndpoint::processAlarmsSettings(const proto::QuasarMessage& message)
{
    std::unique_lock<std::mutex> lock(alarmsSettingsMutex_);
    const AlarmsSettings& newSettings = message.alarms_settings();
    /* Currently only Max volume can be changed */
    if (newSettings.has_max_volume_level()) {
        alarmsSettings_.set_max_volume_level(newSettings.max_volume_level());
    }

    PersistentFile file(alarmsSettingsFile_, PersistentFile::Mode::TRUNCATE);
    if (!file.write(alarmsSettings_.SerializeAsString())) {
        throw std::runtime_error("Cannot write to the file " + alarmsSettingsFile_);
    }
    /* Notify SDK User that new settings for Alarms should be applied */
    deviceContext_->fireAlarmsSettingsChanged(alarmsSettings_);
    lock.unlock();
    sendState();
}

void AlarmEndpoint::handleAccountChange(const std::string& passportUid)
{
    if (passportUid_ == passportUid) {
        return;
    }
    const bool wasEmpty = passportUid_.empty();
    passportUid_ = passportUid;
    if (wasEmpty) {
        // handle only actual account change
        return;
    }

    YIO_LOG_INFO("User account changed: clearing alarms and timers");
    eventStorage_.clear();
    eventStorage_.saveEvents(dbFileName_);
    sendState();
    reportState("changeUserEventState");
}

void AlarmEndpoint::handleSystemConfigUpdate(const std::string& receivedConfig)
{
    auto configOptional = tryParseJson(receivedConfig);
    if (!receivedConfig.empty() && configOptional && configOptional->isMember("system_config")) {
        const auto& systemConfig = (*configOptional)["system_config"];
        if (systemConfig.isMember("alarmd") && systemConfig["alarmd"].isObject()) {
            if (auto maxAlarmDelay = tryGetUInt32(systemConfig["alarmd"], "maxAlarmDelaySec", defaultSettings_.maxAlarmDelaySec);
                maxAlarmDelay != currentSettings_.maxAlarmDelaySec)
            {
                currentSettings_.maxAlarmDelaySec = maxAlarmDelay;
                changeAlarmsDelay(currentSettings_.maxAlarmDelaySec);
            }
            if (auto alwaysStopMedia = tryGetBool(systemConfig["alarmd"], "alwaysStopsMedia", defaultSettings_.alwaysStopsMedia);
                currentSettings_.alwaysStopsMedia != alwaysStopMedia)
            {
                std::lock_guard<std::mutex> lockGuard(mutex_);
                currentSettings_.alwaysStopsMedia = alwaysStopMedia;
                YIO_LOG_INFO("New alwaysStopsMedia value = " << currentSettings_.alwaysStopsMedia);
            }
        }
    }
}

int AlarmEndpoint::port() const {
    return server_->port();
}

AlarmEndpoint::AlarmsSettings AlarmEndpoint::getAlarmsSettings() const {
    std::lock_guard<std::mutex> guard(alarmsSettingsMutex_);
    return alarmsSettings_;
}

EventStorage& AlarmEndpoint::getEventStorage()
{
    return eventStorage_;
}

void AlarmEndpoint::setOnClientConnected(ipc::IServer::ClientHandler onClientConnected)
{
    server_->setClientConnectedHandler(std::move(onClientConnected));
}

void AlarmEndpoint::processAddAlarm(const proto::Alarm& addAlarm)
{
    YIO_LOG_INFO("Adding alarm: " << convertMessageToJsonString(addAlarm));
    eventStorage_.addEvent(
        addAlarm.id(),
        addAlarm);
    eventStorage_.saveEvents(dbFileName_);
    reportState("addAlarmState");
    device_->telemetry()->reportEvent("addAlarm", alarmToJSON(addAlarm));
}

void AlarmEndpoint::processRemoveAlarm(const ::std::string& alarmId)
{
    YIO_LOG_INFO("Removing alarm " << alarmId << " from storage");
    eventStorage_.deleteEvent(alarmId);
    eventStorage_.saveEvents(dbFileName_);
    reportState("removeAlarmState");
    device_->telemetry()->reportKeyValues("removeAlarm", {{"id", alarmId}});
}

void AlarmEndpoint::processPauseAlarm(const std::string& alarmId)
{
    YIO_LOG_INFO("Pausing alarm " << alarmId << " in storage");
    if (!eventStorage_.pauseTimerEvent(alarmId)) {
        YIO_LOG_WARN("Alarm " << alarmId << " not found in storage or already paused");
        eventStorage_.saveEvents(dbFileName_);
    }
    reportState("pauseTimerState");
    device_->telemetry()->reportKeyValues("pauseTimer", {{"id", alarmId}});
}

void AlarmEndpoint::processResumeAlarm(const std::string& alarmId)
{
    YIO_LOG_INFO("Resuming alarm " << alarmId << " in storage");
    if (!eventStorage_.resumeTimerEvent(alarmId)) {
        YIO_LOG_WARN("Alarm " << alarmId << " not found in storage or not paused");
        eventStorage_.saveEvents(dbFileName_);
    }
    reportState("resumeTimerState");
    device_->telemetry()->reportKeyValues("resumeTimer", {{"id", alarmId}});
}

void AlarmEndpoint::stopAlarm(bool stopMedia)
{
    YIO_LOG_INFO("Stopping alarm");
    // stopMedia matters only if alwaysStopsMedia == false
    alarmPlayer_.cancelAlarm(currentSettings_.alwaysStopsMedia || stopMedia);
}

void AlarmEndpoint::checkAlarms()
{
    while (isRunning_) {
        std::this_thread::sleep_for(CHECK_TIMEOUT_MS);
        std::lock_guard<std::mutex> lockGuard(mutex_);
        checkAlarmsIteration(time(nullptr));
        if (!eventStorage_.isEmpty()) {
            constexpr bool sendMetrica = false;
            sendState(sendMetrica);
        }
    }
}

void AlarmEndpoint::checkAlarmsIteration(time_t curtime)
{
    auto alarmsToEnqueue = eventStorage_.fireEvents(std::chrono::seconds(curtime), MAX_DELAY_MS);

    for (auto& alarm : alarmsToEnqueue.actualEvents) {
        if (!isStereoPairFollower_) {
            if (alarm.alarm_type() == proto::Alarm::ALARM && !mediaAlarmSetting_.getInfo().empty()) {
                alarm.set_alarm_type(proto::Alarm::MEDIA_ALARM);
            }
            YIO_LOG_INFO("Adding alarm to playing queue. " << convertMessageToJsonString(alarm));
            alarmPlayer_.enqueueAlarm(alarm);
            device_->telemetry()->reportEvent("alarmSchedule", alarmToJSON(alarm));
        } else {
            YIO_LOG_INFO("Alarm ignored because stereo pair follower mode enabled");
        }
    }

    if (!alarmsToEnqueue.actualEvents.empty() || !alarmsToEnqueue.expiredEvents.empty()) {
        eventStorage_.saveEvents(dbFileName_);
        reportState("checkAlarmsState");
        sendState();
    }
}

void AlarmEndpoint::parseIcalendar()
{
    YIO_LOG_DEBUG("Reparsing icalendar");
    std::lock_guard<std::mutex> lockGuard(mutex_);
    time_t curtime = std::time(nullptr);
    checkAlarmsIteration(curtime);
    eventStorage_.reParseICalendar(curtime);
    reportState("parseIcalendarState");
}

void AlarmEndpoint::sendState(bool sendMetrica)
{
    proto::QuasarMessage message;
    message.mutable_alarms_state()->set_icalendar_state(eventStorage_.getICalendarState());

    std::string mediaAlarmSettingInfo = mediaAlarmSetting_.getInfo();
    if (!mediaAlarmSettingInfo.empty()) {
        message.mutable_alarms_state()->set_media_alarm_setting(std::move(mediaAlarmSettingInfo));
    }
    {
        std::lock_guard<std::mutex> guard(alarmsSettingsMutex_);
        message.mutable_alarms_state()->mutable_alarms_settings()->CopyFrom(alarmsSettings_);
    }
    message.mutable_timers_state()->CopyFrom(eventStorage_.getTimersState());
    message.mutable_set_alarms_state()->CopyFrom(eventStorage_.getAlarmsState());
    server_->sendToAll(std::move(message));

    if (sendMetrica) {
        reportState("sending_alarm_state");
    }
}

void AlarmEndpoint::reportState(const std::string& event)
{
    std::unordered_map<std::string, std::string> alarm_state{{"icalendar", eventStorage_.getICalendarState()},
                                                             {"timers", alarmsToString(eventStorage_.getTimersState())},
                                                             {"alarms", alarmsToString(eventStorage_.getAlarmsState())}};
    std::string mediaAlarmSettingInfo = mediaAlarmSetting_.getInfo();
    if (!mediaAlarmSettingInfo.empty()) {
        alarm_state["sound_alarm_setting"] = std::move(mediaAlarmSettingInfo);
    }
    {
        std::lock_guard<std::mutex> guard(alarmsSettingsMutex_);
        alarm_state["max_sound_level"] = std::to_string(alarmsSettings_.max_volume_level());
    }
    device_->telemetry()->reportKeyValues(event, alarm_state);
}

void AlarmEndpoint::changeAlarmsDelay(int64_t maxDelaySec)
{
    std::lock_guard<std::mutex> lockGuard(mutex_);
    std::uniform_int_distribution<int64_t> distribution{0, maxDelaySec};
    int64_t randomDelaySec = distribution(randomGenerator_);
    YIO_LOG_INFO("New alarms delay in seconds: " << randomDelaySec << ", maxDelaySec was " << maxDelaySec);
    eventStorage_.setAlarmDelay(randomDelaySec);
}
