#pragma once

#include "alarm_player.h"
#include "event_storage.h"

#include <yandex_io/interfaces/auth/i_auth_provider.h>
#include <yandex_io/interfaces/stereo_pair/i_stereo_pair_provider.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/libs/threading/periodic_executor.h>
#include <yandex_io/protos/quasar_proto.pb.h>
#include <yandex_io/sdk/private/device_context.h>

#include <atomic>
#include <memory>
#include <mutex>
#include <random>
#include <string>

namespace quasar {

    class AlarmEndpoint {
        using AlarmsSettings = quasar::proto::AlarmsSettings;

    public:
        AlarmEndpoint(
            std::shared_ptr<YandexIO::IDevice> device,
            std::shared_ptr<ipc::IIpcFactory> ipcFactory,
            std::shared_ptr<IAuthProvider> authProvider,
            std::shared_ptr<IStereoPairProvider> stereoPairProvider);
        ~AlarmEndpoint();

        int port() const;

        AlarmsSettings getAlarmsSettings() const;

        EventStorage& getEventStorage();

        void setOnClientConnected(ipc::IServer::ClientHandler onClientConnected);

    private:
        void processQuasarMessage(const ipc::SharedMessage& message, ipc::IServer::IClientConnection& connection);
        void processAlarmMessage(const quasar::proto::QuasarMessage& message);
        void processNewIcalendarState(const quasar::proto::QuasarMessage& message);
        void processAlarmSetSound(const quasar::proto::QuasarMessage& message);
        void processAlarmResetSound();
        void processAlarmsSettings(const quasar::proto::QuasarMessage& message);
        void processMediaAlarmRequest(const quasar::proto::QuasarMessage& message);

        void handleSystemConfigUpdate(const std::string& receivedConfig);
        void handleAccountChange(const std::string& uid);

        void handleSDKConnect();

        void processAddAlarm(const quasar::proto::Alarm& addAlarm);
        void processRemoveAlarm(const std::string& alarmId);
        void processPauseAlarm(const std::string& alarmId);
        void processResumeAlarm(const std::string& alarmId);
        void stopAlarm(bool stopMedia);

        void checkAlarms();
        void checkAlarmsIteration(time_t curtime);
        void sendState(bool sendMetrica = true);
        void parseIcalendar();

        void changeAlarmsDelay(int64_t maxDelaySec);

        void reportState(const std::string& event);

    private:
        Lifetime lifetime_;
        const std::shared_ptr<YandexIO::IDevice> device_;
        const std::shared_ptr<IAuthProvider> authProvider_;
        const std::shared_ptr<IStereoPairProvider> stereoPairProvider_;
        const std::shared_ptr<YandexIO::DeviceContext> deviceContext_;
        const std::shared_ptr<ipc::IServer> server_;
        const std::shared_ptr<ipc::IConnector> syncdConnector_;
        const std::string dbFileName_;
        const std::string mediaAlarmSettingJsonFile_;
        const std::string alarmsSettingsFile_;
        AlarmPlayer alarmPlayer_;
        std::atomic<bool> isStereoPairFollower_{false};

        struct Settings {
            int64_t maxAlarmDelaySec = 0;
            bool alwaysStopsMedia = true;
        };
        Settings defaultSettings_;
        Settings currentSettings_;

        std::mt19937 randomGenerator_;

        EventStorage eventStorage_;
        std::string passportUid_;

        std::atomic<bool> isRunning_;
        std::thread cronThread_;

        std::chrono::milliseconds maxICalendarParseInterval_ = std::chrono::hours(1);
        std::chrono::milliseconds currentIcalendarParseInterval_ = std::chrono::seconds(1);

        mutable std::mutex mutex_;
        mutable std::mutex alarmsSettingsMutex_;
        AlarmsSettings alarmsSettings_;

        class MediaAlarmSetting final {
        public:
            std::string getInfo() const {
                std::lock_guard<std::mutex> guard(mediaAlarmSettingMutex_);
                return mediaAlarmSettingInfo_;
            }

            void clearInfo() {
                std::lock_guard<std::mutex> guard(mediaAlarmSettingMutex_);
                mediaAlarmSettingInfo_.clear();
            }

            void setInfo(const std::string& info) {
                std::lock_guard<std::mutex> guard(mediaAlarmSettingMutex_);
                mediaAlarmSettingInfo_ = info;
            }

        private:
            mutable std::mutex mediaAlarmSettingMutex_;
            std::string mediaAlarmSettingInfo_;
        };
        MediaAlarmSetting mediaAlarmSetting_;

        std::unique_ptr<PeriodicExecutor> iCalendarParseExecutor_;

    public:
        std::function<void(const quasar::proto::QuasarMessage&)> onQuasarMessageReceivedCallback; // Testing purposes only!
    };

} // namespace quasar
