#include "alarm_player.h"

#include "alarm_utils.h"

#include <yandex_io/libs/base/directives.h>
#include <yandex_io/libs/base/persistent_file.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/sdk/interfaces/directive.h>

YIO_DEFINE_LOG_MODULE("alarm");

using namespace quasar;

AlarmPlayer::AlarmPlayer(std::shared_ptr<YandexIO::IDevice> device,
                         std::shared_ptr<ipc::IIpcFactory> ipcFactory,
                         std::weak_ptr<YandexIO::DeviceContext> inDeviceContext,
                         std::shared_ptr<ipc::IServer> server)
    : device_(std::move(device))
    , deviceContext_(std::move(inDeviceContext))
    , server_(std::move(server))
    , alicedConnector_(ipcFactory->createIpcConnector("aliced"))
    , currentState_(AlarmPlayerState::IDLE)
    , asyncQueue_(std::make_shared<NamedCallbackQueue>("AlarmPlayer"))
    , timeoutStopCallback_(asyncQueue_)
{
    auto deviceContext = deviceContext_.lock();
    if (!deviceContext) {
        throw std::invalid_argument("Device Context is not inited");
    }
    if (!server_) {
        throw std::invalid_argument("ipc server is not inited");
    }

    const auto config = device_->configuration()->getServiceConfig("alarmd");
    needAlarmApproval_ = tryGetBool(config, "needAlarmApproval", true);
    alarmApprovalTimeout_ = std::chrono::milliseconds(tryGetUInt64(config, "approvalTimeoutMs", 1000));
    persistentFileName_ = config["alarmPlayerFile"].asString();
    alarmTimerTimeout_ = std::chrono::seconds(tryGetInt(config, "alarmTimerTimeoutSec", 10 * 60));

    deviceContext->onApproveAlarm = [this](const std::string& alarmId) {
        asyncQueue_->add([this, alarmId]() {
            if (currentState_ != AlarmPlayerState::WAITING_FOR_APPROVAL || alarmId != processedAlarm_.id()) {
                // alarm was probably cancelled or started without approval
                YIO_LOG_WARN("Alarm was approved, but AlarmPlayer is not waiting. alarm_id: " << alarmId);
                return;
            }
            YIO_LOG_INFO("Alarm approved. alarm_id: " << alarmId);
            startAlarm();
        });
    };

    alicedConnector_->connectToService();

    loadAlarmPlayerState();
    if (currentState_ != AlarmPlayerState::IDLE) {
        currentState_ = AlarmPlayerState::IDLE;
        enqueueAlarm(processedAlarm_);
    }
}

AlarmPlayer::~AlarmPlayer()
{
    lifetime_.die();
    asyncQueue_->destroy();

    alicedConnector_->shutdown();
}

void AlarmPlayer::enqueueAlarm(proto::Alarm alarm)
{
    asyncQueue_->add([this, alarm = std::move(alarm)]() {
        device_->telemetry()->reportEvent("playAlarm", alarmToJSON(alarm));

        if (alarm.alarm_type() == proto::Alarm::COMMAND_TIMER) {
            YIO_LOG_INFO("This is directive alarm/timer");
            for (const auto& command : alarm.command_list()) {
                auto directive = YandexIO::Directive::createFromExternalCommandMessage(command);
                auto message = ipc::buildMessage([&](auto& msg) {
                    msg.mutable_directive()->CopyFrom(YandexIO::Directive::convertToDirectiveProtobuf(directive));
                });

                alicedConnector_->sendMessage(message);
            }
            return;
        }

        if (currentState_ != AlarmPlayerState::IDLE) {
            YIO_LOG_INFO("Alarm or timer is already playing. Increase timeout for current alarm_id " << processedAlarm_.id() << " and skip alarm_id: " << alarm.id());
            setAlarmTimerTimeoutHandler(processedAlarm_.id());
            return;
        }

        YIO_LOG_INFO("Alarm enqueued, type: " << alarmTypeToString(alarm.alarm_type()) << ". alarm_id: " << alarm.id());
        {
            // notify other services about upcoming alarm sound
            auto message = ipc::buildMessage([&alarm](auto& msg) {
                auto* event = msg.mutable_alarm_event();
                event->mutable_alarm_fired()->CopyFrom(alarm);
            });
            server_->sendToAll(message);
        }

        hasAnyRemainingMedia_ = false;
        {
            std::scoped_lock<std::mutex> lock(processedAlarmMutex_);
            processedAlarm_ = alarm;
            setAlarmTimerTimeoutHandler(processedAlarm_.id());
        }
        if (needAlarmApproval_) {
            requestAlarmApproval();
        } else {
            startAlarm();
        }
    });
}

void AlarmPlayer::requestAlarmApproval()
{
    YIO_LOG_INFO("requestAlarmApproval. alarm_id: " << processedAlarm_.id());
    currentState_ = AlarmPlayerState::WAITING_FOR_APPROVAL;
    saveAlarmPlayerState();

    try {
        if (const auto deviceContext = deviceContext_.lock()) {
            deviceContext->fireEnqueueAlarm(alarmTypeToIOEvent(processedAlarm_.alarm_type()), processedAlarm_.id());
        }
        setApprovalTimeoutHandler(processedAlarm_.id());

    } catch (const std::runtime_error& e) {
        YIO_LOG_ERROR_EVENT("AlarmPlayer.FailedEnqueueAlarm", "Exception while fireEnqueueAlarm: " << e.what());
        device_->telemetry()->reportEvent("wrongTypeFireEnqueueAlarm");
        // fireEnqueueAlarm failed, so just start processedAlarm
        startAlarm();
    }
}

void AlarmPlayer::setApprovalTimeoutHandler(std::string alarmId)
{
    asyncQueue_->addDelayed([this, alarmId = std::move(alarmId)]() {
        if (currentState_ == AlarmPlayerState::WAITING_FOR_APPROVAL && alarmId == processedAlarm_.id()) {
            YIO_LOG_INFO("Alarm was not approved, fire it anyway. alarm_id: " << alarmId);
            startAlarm();
        }
    }, alarmApprovalTimeout_);
}

void AlarmPlayer::startAlarm()
{
    currentState_ = AlarmPlayerState::WAITING_FOR_ALARM_START_CONFIRM;
    saveAlarmPlayerState();

    {
        auto message = ipc::buildMessage([alarm = processedAlarm_](auto& msg) {
            auto* event = msg.mutable_alarm_event();
            event->mutable_alarm_approved()->CopyFrom(alarm);
        });
        server_->sendToAll(message);
    }
}

void AlarmPlayer::setAlarmTimerTimeoutHandler(std::string alarmId)
{
    timeoutStopCallback_.executeDelayed(
        [this, alarmId = std::move(alarmId)]() {
            if (currentState_ != AlarmPlayerState::IDLE && alarmId == processedAlarm_.id()) {
                YIO_LOG_INFO("Alarm is still playing after timeout. Stop it. alarm_id: " << alarmId);
                device_->telemetry()->reportEvent("AlarmTimeout", alarmToJSON(processedAlarm_));
                stopAlarm(true);
            } else {
                YIO_LOG_DEBUG("Alarm was already stopped manually. alarm_id: " << alarmId);
            }
        },
        alarmTimerTimeout_, lifetime_);
}

void AlarmPlayer::stopAnyRemainingMedia()
{
    asyncQueue_->add([this]() {
        if (currentState_ != AlarmPlayerState::IDLE) {
            YIO_LOG_WARN("Requested to stop media, but player is not idle. Abort");
            return;
        }

        if (!hasAnyRemainingMedia_) {
            YIO_LOG_INFO("No remaining media, ignore it");
            return;
        }

        YIO_LOG_INFO("Fire stop remaining media");
        hasAnyRemainingMedia_ = false;
        if (const auto deviceContext = deviceContext_.lock()) {
            deviceContext->fireAlarmStopRemainingMedia();
        }
    });
}

void AlarmPlayer::onAlarmEvent(const proto::AlarmEvent& event)
{
    if (event.has_alarm_confirmed()) {
        asyncQueue_->add([this, confirmedAlarm = event.alarm_confirmed()]() {
            if (confirmedAlarm.id() != processedAlarm_.id()) {
                YIO_LOG_WARN("Alarm was confirmed, but id doesn't match");
                return;
            }

            bool isFallback = false;
            if (currentState_ != AlarmPlayerState::WAITING_FOR_ALARM_START_CONFIRM) {
                YIO_LOG_INFO("Alarm fallback, new type: " << alarmTypeToString(confirmedAlarm.alarm_type()) << ". alarm_id: " << confirmedAlarm.id())
                isFallback = true;
            } else {
                YIO_LOG_INFO("Confirmed alarm type: " << alarmTypeToString(confirmedAlarm.alarm_type()) << ". alarm_id: " << confirmedAlarm.id());
            }

            auto alarmType = confirmedAlarm.alarm_type();
            {
                std::scoped_lock<std::mutex> lock(processedAlarmMutex_);
                processedAlarm_.set_alarm_type(alarmType);
            }

            if (alarmType == proto::Alarm::MEDIA_ALARM) {
                currentState_ = AlarmPlayerState::MEDIA_ALARM;
            } else if (alarmType == proto::Alarm::ALARM) {
                currentState_ = AlarmPlayerState::CLASSIC_ALARM;
            } else {
                currentState_ = AlarmPlayerState::TIMER;
            }

            if (isFallback) {
                return;
            }

            auto message = ipc::buildMessage([alarm = confirmedAlarm](auto& msg) {
                auto* event = msg.mutable_alarm_event();
                event->mutable_alarm_started()->CopyFrom(alarm);
            });
            server_->sendToAll(message);

            try {
                if (const auto deviceContext = deviceContext_.lock()) {
                    deviceContext->fireStartAlarm(alarmTypeToIOEvent(processedAlarm_.alarm_type()));
                }
            } catch (const std::runtime_error& e) {
                YIO_LOG_ERROR_EVENT("AlarmPlayer.FailedStartAlarm", "Exception while fireStartAlarm: " << e.what());
                device_->telemetry()->reportEvent("wrongTypeFireStartAlarm");
            }
        });
    }
}

void AlarmPlayer::cancelAlarm(bool stopMedia)
{
    asyncQueue_->add([this, stopMedia]() {
        if (currentState_ == AlarmPlayerState::IDLE) {
            YIO_LOG_WARN("Cancel alarm, but AlarmPlayer is idle. Abort");
            return;
        }
        stopAlarm(stopMedia);
    });
}

void AlarmPlayer::stopAlarm(bool stopMedia)
{
    YIO_LOG_INFO("Stopping alarm type: " << alarmTypeToString(processedAlarm_.alarm_type()) << ". alarm_id: " << processedAlarm_.id());

    hasAnyRemainingMedia_ = false;
    if (!stopMedia && currentState_ == AlarmPlayerState::MEDIA_ALARM) {
        YIO_LOG_INFO("Don't pause media for alarm_id: " << processedAlarm_.id());
        hasAnyRemainingMedia_ = true;
    }
    currentState_ = AlarmPlayerState::IDLE;

    auto processedAlarmCopy = processedAlarm_;
    {
        std::scoped_lock<std::mutex> lock(processedAlarmMutex_);
        processedAlarm_.set_id("");
        if (timeoutStopCallback_.isScheduled()) {
            timeoutStopCallback_.reset();
        }
    }
    saveAlarmPlayerState();

    {
        auto message = ipc::buildMessage([alarm = processedAlarmCopy, stopMedia = !hasAnyRemainingMedia_](auto& msg) {
            auto* event = msg.mutable_alarm_event();
            event->mutable_alarm_stopped()->mutable_alarm()->CopyFrom(alarm);
            event->mutable_alarm_stopped()->set_stop_media(stopMedia);
        });
        server_->sendToAll(message);
    }

    try {
        if (const auto deviceContext = deviceContext_.lock()) {
            deviceContext->fireStopAlarm(alarmTypeToIOEvent(processedAlarmCopy.alarm_type()), hasAnyRemainingMedia_);
        }
    } catch (const std::runtime_error& e) {
        YIO_LOG_ERROR_EVENT("AlarmPlayer.FailedStopAlarm", "Exception while fireStopAlarm" << e.what());
        device_->telemetry()->reportEvent("wrongTypeFireStopAlarm");
    }

    device_->telemetry()->reportEvent("stopAlarm", alarmToJSON(processedAlarmCopy));
}

proto::IOEvent_AlarmType AlarmPlayer::alarmTypeToIOEvent(proto::Alarm_AlarmType type)
{
    switch (type) {
        case proto::Alarm::TIMER:
            return proto::IOEvent::TIMER;
        case proto::Alarm::ALARM:
            return proto::IOEvent::CLASSIC_ALARM;
        case proto::Alarm::MEDIA_ALARM:
            return proto::IOEvent::MEDIA_ALARM;
        default:
            throw std::runtime_error("Unexpected type");
    }
}

void AlarmPlayer::loadAlarmPlayerState()
{
    if (!fileExists(persistentFileName_)) {
        return;
    }

    proto::CbAlarmPlayerState fromFile;
    try {
        Y_PROTOBUF_SUPPRESS_NODISCARD fromFile.ParseFromString(getFileContent(persistentFileName_));
    } catch (std::runtime_error& error) {
        YIO_LOG_ERROR_EVENT("AlarmPlayer.BadProto.AlarmPlayerState", "Error while reading QueuedAlarmPlayer persistent file: " << error.what());
        return;
    }

    switch (fromFile.current_state()) {
        case proto::CbAlarmPlayerState::IDLE:
            currentState_ = AlarmPlayerState::IDLE;
            break;
        case proto::CbAlarmPlayerState::WAITING_FOR_APPROVAL:
            currentState_ = AlarmPlayerState::WAITING_FOR_APPROVAL;
            break;
        case proto::CbAlarmPlayerState::TIMER:
            currentState_ = AlarmPlayerState::TIMER;
            break;
        case proto::CbAlarmPlayerState::CLASSIC_ALARM:
            currentState_ = AlarmPlayerState::CLASSIC_ALARM;
            break;
        case proto::CbAlarmPlayerState::WAITING_FOR_ALARM_START_CONFIRM:
            currentState_ = AlarmPlayerState::WAITING_FOR_ALARM_START_CONFIRM;
            break;
        case proto::CbAlarmPlayerState::MEDIA_ALARM:
            currentState_ = AlarmPlayerState::MEDIA_ALARM;
            break;
    }
    processedAlarm_ = fromFile.processed_alarm();
    hasAnyRemainingMedia_ = fromFile.has_remaining_media();
}

void AlarmPlayer::saveAlarmPlayerState() const {
    proto::CbAlarmPlayerState toFile;
    switch (currentState_) {
        case AlarmPlayerState::IDLE:
            toFile.set_current_state(proto::CbAlarmPlayerState::IDLE);
            break;
        case AlarmPlayerState::WAITING_FOR_APPROVAL:
            toFile.set_current_state(proto::CbAlarmPlayerState::WAITING_FOR_APPROVAL);
            break;
        case AlarmPlayerState::TIMER:
            toFile.set_current_state(proto::CbAlarmPlayerState::TIMER);
            break;
        case AlarmPlayerState::CLASSIC_ALARM:
            toFile.set_current_state(proto::CbAlarmPlayerState::CLASSIC_ALARM);
            break;
        case AlarmPlayerState::WAITING_FOR_ALARM_START_CONFIRM:
            toFile.set_current_state(proto::CbAlarmPlayerState::WAITING_FOR_ALARM_START_CONFIRM);
            break;
        case AlarmPlayerState::MEDIA_ALARM:
            toFile.set_current_state(proto::CbAlarmPlayerState::MEDIA_ALARM);
            break;
    }
    *(toFile.mutable_processed_alarm()) = processedAlarm_;
    toFile.set_has_remaining_media(hasAnyRemainingMedia_);

    PersistentFile fileStream(persistentFileName_, PersistentFile::Mode::TRUNCATE);
    if (!fileStream.write(toFile.SerializeAsString())) {
        throw std::runtime_error("Cannot write to the file " + persistentFileName_);
    }
}

proto::Alarm AlarmPlayer::getPlayingAlarm() const {
    std::scoped_lock<std::mutex> lock(processedAlarmMutex_);
    return processedAlarm_;
}

void AlarmPlayer::waitUntilConnected()
{
    alicedConnector_->waitUntilConnected();
}
