#pragma once

#include <yandex_io/libs/base/named_callback_queue.h>
#include <yandex_io/libs/threading/unique_callback.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/protos/quasar_proto.pb.h>
#include <yandex_io/protos/storage.pb.h>
#include <yandex_io/sdk/private/device_context.h>

#include <atomic>

namespace quasar {

    class AlarmPlayer {
    public:
        AlarmPlayer(
            std::shared_ptr<YandexIO::IDevice> device,
            std::shared_ptr<ipc::IIpcFactory> ipcFactory,
            std::weak_ptr<YandexIO::DeviceContext> deviceContext,
            std::shared_ptr<ipc::IServer> server);
        ~AlarmPlayer();

        void enqueueAlarm(quasar::proto::Alarm alarm);
        void cancelAlarm(bool stopMedia = true);

        void stopAnyRemainingMedia();

        proto::Alarm getPlayingAlarm() const;

        void onAlarmEvent(const proto::AlarmEvent& event);

    private:
        void requestAlarmApproval();
        void setApprovalTimeoutHandler(std::string alarmId);
        void startAlarm();
        void stopAlarm(bool stopMedia = true);
        void setAlarmTimerTimeoutHandler(std::string alarmId);

        /**
         * if yiod crashes while alarm is playing, we still will be able to stop it
         * However, persistent file should be in /tmp folder, so it will be deleted on device reboot
         */
        void loadAlarmPlayerState();
        void saveAlarmPlayerState() const;

        static quasar::proto::IOEvent_AlarmType alarmTypeToIOEvent(quasar::proto::Alarm_AlarmType type);

    public:
        /// For testing purposes only
        void waitUntilConnected();

    private:
        const std::shared_ptr<YandexIO::IDevice> device_;
        const std::weak_ptr<YandexIO::DeviceContext> deviceContext_;
        const std::shared_ptr<ipc::IServer> server_;
        const std::shared_ptr<ipc::IConnector> alicedConnector_;

        bool hasAnyRemainingMedia_ = false;

        bool needAlarmApproval_;
        std::chrono::milliseconds alarmApprovalTimeout_;

        std::chrono::seconds alarmTimerTimeout_;
        std::string persistentFileName_;

        Json::Value mediaAlarmSettings_;

        enum class AlarmPlayerState {
            IDLE,
            WAITING_FOR_APPROVAL,
            WAITING_FOR_ALARM_START_CONFIRM,
            TIMER,
            CLASSIC_ALARM,
            MEDIA_ALARM
        };
        AlarmPlayerState currentState_{AlarmPlayerState::IDLE};

        mutable std::mutex processedAlarmMutex_;
        quasar::proto::Alarm processedAlarm_;

        std::shared_ptr<NamedCallbackQueue> asyncQueue_;

        quasar::Lifetime lifetime_;
        quasar::UniqueCallback timeoutStopCallback_;

        AlarmPlayer(const AlarmPlayer&) = delete;
        AlarmPlayer& operator=(const AlarmPlayer&) = delete;
    };

} // namespace quasar
