#include "event_storage.h"

#include <yandex_io/libs/base/persistent_file.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/protos/storage.pb.h>

#include <contrib/libs/libical/src/libical/ical.h>

#include <util/generic/scope.h>

#include <fstream>
#include <iostream>

using namespace quasar;

EventStorage::EventStorage()
{
}

void EventStorage::addEvent(const std::string& eventId, const proto::Alarm& event) {
    std::lock_guard<std::mutex> lockGuard(mutex_);

    storage_[eventId] = event;
    if (event.alarm_type() == proto::Alarm::ALARM || event.alarm_type() == proto::Alarm::MEDIA_ALARM) {
        storage_[eventId].set_delay_seconds(alarmDelaySec_);
    }
}

bool EventStorage::hasEventWithId(const std::string& eventId) const {
    return storage_.count(eventId);
}

bool EventStorage::tryGetEventById(const std::string& eventId, proto::Alarm& event) const {
    auto result = storage_.find(eventId);

    if (result == storage_.end()) {
        return false;
    }

    event = result->second;
    return true;
}

bool EventStorage::deleteEvent(const std::string& eventId) {
    std::lock_guard<std::mutex> lockGuard(mutex_);

    return static_cast<bool>(storage_.erase(eventId));
}

bool EventStorage::pauseTimerEvent(const std::string& eventId) {
    std::lock_guard<std::mutex> lockGuard(mutex_);

    if (!storage_.count(eventId)) {
        return false;
    }

    auto& event = storage_[eventId];
    if (!event.has_pause_timestamp_sec()) {
        event.set_pause_timestamp_sec(time(nullptr));
        return true;
    }

    return false;
}

bool EventStorage::resumeTimerEvent(const std::string& eventId) {
    std::lock_guard<std::mutex> lockGuard(mutex_);

    if (!storage_.count(eventId)) {
        return false;
    }
    proto::Alarm& timer = storage_[eventId];

    if (timer.has_pause_timestamp_sec()) {
        auto pausedSeconds = timer.paused_seconds();
        const auto currentTime = time(nullptr);

        pausedSeconds += std::max(currentTime - timer.pause_timestamp_sec(), (int64_t)0);

        timer.set_paused_seconds(pausedSeconds);
        timer.clear_pause_timestamp_sec();
        return true;
    }

    return false;
}

int EventStorage::getEventCount() const {
    std::lock_guard<std::mutex> guard(mutex_);
    return storage_.size();
}

void EventStorage::clear() {
    std::lock_guard<std::mutex> lockGuard(mutex_);
    storage_.clear();
    setICalendarStateUnlocked("");
}

void EventStorage::clearAlarmsUnlocked()
{
    for (auto i = storage_.begin(), last = storage_.end(); i != last;) {
        if (i->second.alarm_type() == proto::Alarm::ALARM || i->second.alarm_type() == proto::Alarm::MEDIA_ALARM) {
            i = storage_.erase(i);
        } else {
            ++i;
        }
    }
}

void EventStorage::loadEvents(const std::string& dbFileName) {
    std::lock_guard<std::mutex> lockGuard(mutex_);

    if (!fileExists(dbFileName)) { // Storage is not created yet
        return;
    }

    proto::AlarmStorage storage;

    Y_PROTOBUF_SUPPRESS_NODISCARD storage.ParseFromString(getFileContent(dbFileName));

    std::map<std::string, quasar::proto::Alarm> newStorage;

    for (auto const& alarm : storage.alarms()) {
        newStorage[alarm.id()] = alarm;
        if (alarm.alarm_type() == proto::Alarm::ALARM || alarm.alarm_type() == proto::Alarm::MEDIA_ALARM) {
            newStorage[alarm.id()].set_delay_seconds(alarmDelaySec_);
        }
    }
    std::swap(storage_, newStorage);

    iCalendarState_ = storage.icalendar_state();
}

void EventStorage::reParseICalendar(time_t currentTime) {
    std::lock_guard<std::mutex> lockGuard(mutex_);

    parseIcalendar(currentTime);
}

void EventStorage::saveEvents(const std::string& dbFileName) const {
    std::unique_lock<std::mutex> lockGuard(mutex_);

    proto::AlarmStorage storage;

    for (auto const& alarm : storage_) {
        if (alarm.second.alarm_type() == proto::Alarm::TIMER || alarm.second.alarm_type() == proto::Alarm::COMMAND_TIMER) {
            *(storage.add_alarms()) = alarm.second;
        }
    }

    storage.set_icalendar_state(TString(iCalendarState_));

    PersistentFile alarmStream(dbFileName, PersistentFile::Mode::TRUNCATE);

    if (!alarmStream.write(storage.SerializeAsString())) {
        throw std::runtime_error("Cannot write to the file " + dbFileName);
    }

    lockGuard.unlock();
}

EventStorage::FireEvents EventStorage::fireEvents(std::chrono::seconds timestamp,
                                                  std::chrono::milliseconds maxDelay)
{
    std::lock_guard<std::mutex> lockGuard(mutex_);

    EventStorage::FireEvents alarmsToFire;

    const auto currentTimestampMs = std::chrono::milliseconds(timestamp);
    for (auto it = storage_.begin(); it != storage_.end();) {
        proto::Alarm alarm = it->second;
        if (alarm.has_pause_timestamp_sec()) { // timer is paused
            ++it;
            continue;
        }

        const auto alarmTimestampMs = getAlarmTimestampMs(alarm);
        if (alarmTimestampMs <= currentTimestampMs) {
            if (currentTimestampMs - alarmTimestampMs < maxDelay) {
                alarmsToFire.actualEvents.emplace_back(std::move(alarm));
            } else {
                alarmsToFire.expiredEvents.emplace_back(std::move(alarm));
            }
            it = storage_.erase(it);
        } else {
            ++it;
        }
    }

    return alarmsToFire;
}

size_t EventStorage::removeExpiredEvents(std::chrono::seconds timestamp, std::chrono::milliseconds maxDelay)
{
    const auto currentTimestampMs = std::chrono::milliseconds(timestamp);
    size_t expiredEventCount = 0;
    std::lock_guard<std::mutex> lockGuard(mutex_);

    for (auto it = storage_.begin(); it != storage_.end();) {
        proto::Alarm alarm = it->second;
        if (alarm.has_pause_timestamp_sec()) { // timer is paused
            ++it;
            continue;
        }

        const auto alarmTimestampMs = getAlarmTimestampMs(alarm);
        if (alarmTimestampMs <= currentTimestampMs) {
            if (currentTimestampMs - alarmTimestampMs >= maxDelay) {
                ++expiredEventCount;
                it = storage_.erase(it);
                continue;
            }
        }
        ++it;
    }
    return expiredEventCount;
}

std::chrono::milliseconds EventStorage::getAlarmTimestampMs(const proto::Alarm& alarm)
{
    return std::chrono::milliseconds(alarm.start_timestamp_ms()) +
           std::chrono::milliseconds(alarm.duration_seconds() * 1000ll) +
           std::chrono::milliseconds(alarm.delay_seconds() * 1000ll) +
           std::chrono::milliseconds(alarm.paused_seconds() * 1000ll); // how long timer was paused
}

void EventStorage::parseIcalendar(time_t currentTime) {
    clearAlarmsUnlocked();
    if (currentTime == 0) {
        currentTime = time(nullptr);
    }

    const time_t SECONDS_IN_DAY = 24 * 60 * 60;

    time_t currTimeWithoutDelay = currentTime - alarmDelaySec_;

    // Calculate current time in libical format
    const size_t CURTIME_MAXSIZE = 16;
    char curTimeStr[CURTIME_MAXSIZE + 1];
    struct tm resultStruct;
    memset(&resultStruct, 0x00, sizeof(resultStruct));

    const auto result = gmtime_r(&currTimeWithoutDelay, &resultStruct);
    if (result == nullptr) {
        /* Skip this parse stage */
        return;
    }
    strftime(curTimeStr, sizeof(curTimeStr), "%Y%m%dT%H%M%SZ", &resultStruct);
    icaltimetype currentTimeIcal = icaltime_from_string(curTimeStr);

    icalcomponent* iCalendarRoot = icalparser_parse_string(
        iCalendarState_.c_str());
    Y_DEFER {
        icalcomponent_free(iCalendarRoot);
    };

    for (icalcomponent* c = icalcomponent_get_first_component(iCalendarRoot, ICAL_ANY_COMPONENT);
         c != nullptr;
         c = icalcomponent_get_next_component(iCalendarRoot, ICAL_ANY_COMPONENT))
    {
        // Add events today with time after current
        icalproperty* dtStartProperty = icalcomponent_get_first_property(c, ICAL_DTSTART_PROPERTY);
        icalvalue* dtStartValue = icalproperty_get_value(dtStartProperty);
        icaltimetype eventIcalTime = icalvalue_get_datetime(dtStartValue);
        time_t eventTime = icaltime_as_timet(eventIcalTime);
        icalproperty* rRuleProp = icalcomponent_get_first_property(c, ICAL_RRULE_PROPERTY);
        if (!rRuleProp)
        {
            if (eventTime > currTimeWithoutDelay && eventTime < (currTimeWithoutDelay + SECONDS_IN_DAY))
            {
                proto::Alarm alarm;
                alarm.set_alarm_type(proto::Alarm::ALARM);
                alarm.set_id(makeUUID());
                alarm.set_start_timestamp_ms(currentTime * 1000ll);
                alarm.set_duration_seconds(eventTime - currentTime);
                alarm.set_delay_seconds(alarmDelaySec_);
                storage_[alarm.id()] = alarm;
            }
        } else {
            icalrecurrencetype repeatRule = icalproperty_get_rrule(rRuleProp);
            icalrecur_iterator* rItr = icalrecur_iterator_new(repeatRule, eventIcalTime);
            Y_DEFER {
                icalrecur_iterator_free(rItr);
            };
            icalrecur_iterator_set_start(rItr, currentTimeIcal);
            for (struct icaltimetype next = icalrecur_iterator_next(rItr); !icaltime_is_null_time(next);
                 next = icalrecur_iterator_next(rItr))
            {
                time_t nextTime = icaltime_as_timet(next);
                Y_VERIFY(nextTime >= currTimeWithoutDelay);
                if (nextTime - currTimeWithoutDelay >= SECONDS_IN_DAY) {
                    break;
                }
                proto::Alarm alarm;
                alarm.set_alarm_type(proto::Alarm::ALARM);
                alarm.set_id(makeUUID());
                alarm.set_start_timestamp_ms(currentTime * 1000ll);
                alarm.set_duration_seconds(nextTime - currentTime);
                alarm.set_delay_seconds(alarmDelaySec_);
                storage_[alarm.id()] = alarm;
            }
        }
    }
}

void EventStorage::setICalendarState(const std::string& iCalendar) {
    std::lock_guard<std::mutex> lockGuard(mutex_);
    setICalendarStateUnlocked(iCalendar);
}

void EventStorage::setICalendarStateUnlocked(const std::string& iCalendar) {
    iCalendarState_ = iCalendar;
}

std::string EventStorage::getICalendarState() const {
    std::lock_guard<std::mutex> lockGuard(mutex_);
    return iCalendarState_;
}

quasar::proto::TimersState EventStorage::getTimersState() const {
    std::lock_guard<std::mutex> lockGuard(mutex_);
    proto::TimersState timersState;
    for (const auto& alarm : storage_) {
        if (alarm.second.alarm_type() == proto::Alarm::TIMER || alarm.second.alarm_type() == proto::Alarm::COMMAND_TIMER) {
            *(timersState.add_timers()) = alarm.second;
        }
    }
    return timersState;
}

quasar::proto::TimersState EventStorage::getAlarmsState() const {
    std::lock_guard<std::mutex> lockGuard(mutex_);
    proto::TimersState alarmsState;
    for (const auto& alarm : storage_) {
        if (alarm.second.alarm_type() == proto::Alarm::ALARM || alarm.second.alarm_type() == proto::Alarm::MEDIA_ALARM) {
            *(alarmsState.add_timers()) = alarm.second;
        }
    }
    return alarmsState;
}

bool EventStorage::isEmpty() const {
    return storage_.empty();
}

void EventStorage::setAlarmDelay(int64_t alarmDelaySec) {
    std::lock_guard<std::mutex> lockGuard(mutex_);
    alarmDelaySec_ = alarmDelaySec;
    YIO_LOG_INFO("New alarm delay: " << alarmDelaySec_ << " sec");

    for (auto& [eventId, alarm] : storage_) {
        if (alarm.alarm_type() == proto::Alarm::ALARM || alarm.alarm_type() == proto::Alarm::MEDIA_ALARM) {
            alarm.set_delay_seconds(alarmDelaySec_);
        }
    }
}
