#pragma once

#include <yandex_io/protos/model_objects.pb.h>

#include <chrono>
#include <map>
#include <mutex>

namespace quasar {

    class EventStorage {
    public:
        EventStorage();
        virtual ~EventStorage() = default;

        void addEvent(const std::string& eventId, const quasar::proto::Alarm& event);
        bool hasEventWithId(const std::string& eventId) const;
        bool tryGetEventById(const std::string& eventId,
                             quasar::proto::Alarm& event) const;
        bool deleteEvent(const std::string& eventId);
        void clear();

        bool pauseTimerEvent(const std::string& eventId);
        bool resumeTimerEvent(const std::string& eventId);
        int getEventCount() const;
        void reParseICalendar(time_t currentTime = 0);
        void setICalendarState(const std::string& iCalendar);
        std::string getICalendarState() const;
        quasar::proto::TimersState getTimersState() const;
        quasar::proto::TimersState getAlarmsState() const;

        void loadEvents(const std::string& dbFileName);
        void saveEvents(const std::string& dbFileName) const;

        /**
         * Возвращает список будильников, которые должны сработать в указанный промежуток времени, а так же
         * список просроченных.
         *
         * Попутно очищает текущий список активных будильников, удаляя из него все будильники, которые должны были
         * сработать до старшей границы интервала timestamp. Таким образом, каждый будильник сработает не более 1 раза.
         *
         * @param timestamp Временная отметка старшей границы интервала в unix-time.
         * @param maxDelay  Продолжительность интервала.
         * @return Список активных и просроченных будильников в интервале [timestamp - maxDelay, timestamp].
         */
        struct FireEvents {
            std::vector<quasar::proto::Alarm> actualEvents;
            std::vector<quasar::proto::Alarm> expiredEvents;
        };
        FireEvents fireEvents(std::chrono::seconds timestamp, std::chrono::milliseconds maxDelay);
        size_t removeExpiredEvents(std::chrono::seconds timestamp, std::chrono::milliseconds maxDelay);

        size_t size() const {
            return storage_.size();
        }
        bool isEmpty() const;

        void setAlarmDelay(int64_t alarmDelaySec);

    private:
        std::map<std::string, quasar::proto::Alarm> storage_;
        mutable std::mutex mutex_;
        std::string iCalendarState_;
        int64_t alarmDelaySec_ = 0;

        void parseIcalendar(time_t currentTime = 0);
        void clearAlarmsUnlocked();
        void setICalendarStateUnlocked(const std::string& iCalendar);
        static std::chrono::milliseconds getAlarmTimestampMs(const quasar::proto::Alarm& alarm);

        EventStorage(const EventStorage&) = delete;
        EventStorage& operator=(const EventStorage&) = delete;
    };

} // namespace quasar
