#include "alice_config.h"

#include <yandex_io/libs/cryptography/digest.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/spotter_types/spotter_types.h>

YIO_DEFINE_LOG_MODULE("aliceconfig");

using namespace quasar;

::SpeechKit::VoiceServiceSettings AliceConfig::defaultSettings_{SpeechKit::Language::russian};

AliceConfig::AliceConfig(std::shared_ptr<YandexIO::IDevice> device, Json::Value fileConfig)
    : device_(std::move(device))
    , fileConfig_(std::move(fileConfig))
    , fileVoiceDialogSettings_(tryGetJson(fileConfig_, "voiceDialogSettings"))
    , voiceDialogSettings_(fileVoiceDialogSettings_)
{
}

bool AliceConfig::setReceivedConfig(std::string_view jsonConfigStr)
{
    configReceived_ = true;

    if (jsonConfigStr.empty()) {
        return false;
    }

    const auto parsedConfig = tryParseJson(jsonConfigStr);
    if (parsedConfig.has_value()) {
        const auto& config = *parsedConfig;
        backendAccountConfig_ = tryGetJson(config, "account_config");
        backendDeviceConfig_ = tryGetJson(config, "device_config");
        backendSystemConfig_ = tryGetJson(config, "system_config");
        auxiliaryDeviceConfig_ = tryGetJson(config, "auxiliary_device_config");

        voiceDialogSettings_ = fileVoiceDialogSettings_;
        jsonMerge(readBackendJson("voiceDialogSettings", Json::Value()), voiceDialogSettings_);

        const auto directiveNames = jsonToVec<std::string>(getBlockShowCardDirectiveJson());
        blockShowCardDirectiveNames_ = std::set<std::string>(directiveNames.begin(), directiveNames.end());

        return true;
    }

    return false;
}

bool AliceConfig::hasSystemConfig() const {
    return !backendSystemConfig_.isNull();
}

const Json::Value& AliceConfig::getSystemConfig() const {
    return backendSystemConfig_;
}

bool AliceConfig::hasReceivedConfig() const {
    return configReceived_;
}

bool AliceConfig::hasAuxiliaryDeviceConfig() const {
    return !auxiliaryDeviceConfig_.isNull();
}

const Json::Value& AliceConfig::getAuxiliaryDeviceConfig() const {
    return auxiliaryDeviceConfig_;
}

std::string AliceConfig::getContentAccess() const {
    return tryGetString(backendAccountConfig_, "contentAccess");
}

std::string AliceConfig::getChildContentAccess() const {
    return tryGetString(backendAccountConfig_, "childContentAccess");
}

Json::Value AliceConfig::getSpotter() const {
    return tryGetJson(backendSystemConfig_, "spotter");
}

Json::Value AliceConfig::getDisabledCommandPhrases() const {
    return tryGetArray(backendSystemConfig_, "disabledCommandPhrases", Json::nullValue);
}

Json::Value AliceConfig::getDisabledCommandSpotterTypes() const {
    return tryGetArray(backendSystemConfig_, "disabledCommandSpotterTypes", Json::nullValue);
}

bool AliceConfig::getAudioSenderMode() const {
    return tryGetBool(backendSystemConfig_, "audiosenderMode", false);
}

std::string AliceConfig::getSpotterCollisionMode() const {
    return tryGetString(backendSystemConfig_, "spotterCollisionMode");
}

Json::Value AliceConfig::getBlockShowCardDirectiveJson() const {
    return tryGetArray(backendSystemConfig_, "blockShowCardDirectiveNames");
}

Json::Value AliceConfig::getRandomSoundLogger() const {
    auto allowedGroups = readFileStringSet("allowedForRandomLoggingQuasmodromGroups", {});
    if (allowedGroups.count(getQuasmodromGroup()) == 0) {
        return Json::nullValue;
    }

    return tryGetJson(backendSystemConfig_, "randomSoundLogger");
}

std::unordered_set<std::string> AliceConfig::getRandomSoundLoggerChannels() const {
    return tryGetEmplaceableStringSet<std::unordered_set<std::string>>(
        getRandomSoundLogger(), "channelNames", {});
}

Json::Value AliceConfig::getSmartActivation() const {
    constexpr const char* fieldName = "smartActivation";
    if (backendDeviceConfig_.isMember(fieldName)) {
        return backendDeviceConfig_[fieldName];
    }

    if (backendAccountConfig_.isMember(fieldName)) {
        return backendAccountConfig_[fieldName];
    }
    return Json::nullValue;
}

bool AliceConfig::getWithLocalVins() const {
    return readBool("withLocalVins", false);
}

std::string AliceConfig::getSpeechkitDumpPath() const {
    return readFileString("speechkitDumpPath", "");
}

std::string AliceConfig::getQuasmodromGroup() const {
    return readBackendString("quasmodrom_group", "");
}

std::string AliceConfig::getQuasmodromSubgroup() const {
    return readBackendString("quasmodrom_subgroup", "");
}

std::string AliceConfig::getApiKey() const {
    return readFileString("apiKey", "");
}

std::string AliceConfig::getRecognizerModel() const {
    return readFileString("shortRecognizerModel", defaultSettings_.recognizer.model.getValue());
}

bool AliceConfig::getUseCustomJinglePlayer() const {
    return readBool("useCustomJinglePlayer", false);
}

bool AliceConfig::getRequireAuthorization() const {
    return readFileBool("requireAuthorization", true);
}

std::string AliceConfig::getBassUrl() const {
    return readString("bassUrl", "");
}

std::string AliceConfig::getVinsUrl() const {
    return readString("vinsUrl", "");
}

std::string AliceConfig::getUniProxyUrl() const {
    return readString("uniProxyUrl", defaultSettings_.uniProxyUrl);
}

bool AliceConfig::getUseAudioClientTtsPlayer() const {
    return readBool("useAudioClientTtsPlayer", false);
}

std::chrono::milliseconds AliceConfig::getSilenceTimeAfterTts() const {
    return readMillis("silenceTimeAfterTtsInMs", std::chrono::milliseconds(0));
}

bool AliceConfig::getJingle() const {
    if (readBool("jingleAlwaysOn", false)) {
        return true;
    }
    return tryGetBool(backendAccountConfig_, "jingle", readFileBool("jingle", false));
}

std::string AliceConfig::getSingleDialogMode() const {
    return readString("singleDialogMode", "");
}

bool AliceConfig::getSpottersEnabled() const {
    return readBool("enableSpotters", true);
}

bool AliceConfig::getCommandSpottersEnabled() const {
    return readBool("enableCommandSpotters", false);
}

bool AliceConfig::getOnlineSpotterEnabled() const {
    return readBool("onlineSpotterEnabled", true);
}

bool AliceConfig::getLongListeningEnabled() const {
    return readBool("longListeningEnabled", false);
}

int AliceConfig::getSoundLogMaxParallelSendings() const {
    return readInt("soundLogMaxParallelSendings", 1);
}

bool AliceConfig::getPrefetchEnabled() const {
    return readBool("prefetchEnabled", false);
}

bool AliceConfig::getSendNavigationSpotterLog() const {
    return readBackendBool("sendNavigationSpotterLog", false);
}

bool AliceConfig::getSendCommandSpotterLog() const {
    return readBackendBool("sendCommandSpotterLog", false);
}

std::chrono::seconds AliceConfig::getStartDelay() const {
    return readFileSeconds("startDelaySec", std::chrono::seconds::zero());
}

std::chrono::milliseconds AliceConfig::getSpotterStatisticsLoggingIntevalMillis() const {
    return readFileMillis("spotterStatisticsLoggingIntevalMillis", std::chrono::milliseconds::zero());
}

bool AliceConfig::getVadEnabled() const {
    return readFileBool("vadEnabled", defaultSettings_.recognizer.vadEnabled);
}

std::chrono::milliseconds AliceConfig::getSpotterLoggingHead() const {
    return getVoiceDialogTimingWithFileDefault("spotterLoggingHeadMillis", std::chrono::milliseconds(1500));
}

std::chrono::milliseconds AliceConfig::getSpotterLoggingTail() const {
    return getVoiceDialogTimingWithFileDefault("spotterLoggingTailMillis", std::chrono::milliseconds(500));
}

bool AliceConfig::getEnableRealtimeStreamer() const {
    return readFileBool("enable_realtime_streamer", false);
}

std::chrono::milliseconds AliceConfig::getSubThresholdSendRate() const {
    return tryGetMillis(voiceDialogSettings_, "subThresholdSendRate", std::chrono::milliseconds(60 * 60 * 1000));
}

bool AliceConfig::environmentStateEnabled() const {
    return tryGetBool(backendSystemConfig_, "environmentStateEnabled", true);
}

bool AliceConfig::getDnsCacheEnabled() const {
    return tryGetBool(voiceDialogSettings_, "dnsCacheEnabled", false);
}

double AliceConfig::getRMSCorrection() const {
    return readDouble("RMSCorrection", 1.0);
}

bool AliceConfig::getRMSOverVqe() const {
    return tryGetBool(backendSystemConfig_, "RMSOverVqe", false);
}

std::unordered_set<std::string> AliceConfig::getSpotterRMSChannels() const {
    return readStringSet("spotterRMSChannels", {});
}

std::unordered_set<std::string> AliceConfig::getSpeechkitLogChannels() const {
    auto logChannels = readStringSet("speechkitLogChannels", {"vqe"});
    if (logChannels.count("*")) {
        logChannels.clear();
    }

    return logChannels;
}

std::string AliceConfig::getDeviceStateSenderCachePath() const {
    return readFileString("deviceStateSenderCachePath", "");
}

Json::Value AliceConfig::getDeviceStateSenderSettings() const {
    return readBackendJson("deviceStateSender", Json::nullValue);
}

::SpeechKit::VoiceServiceSettings AliceConfig::getVoiceServiceSettings(
    const std::string& authToken,
    const std::string& passportUid,
    const std::string& activationSpotterPath,
    const std::string& interruptionSpotterPath,
    const std::string& additionalSpotterPath,
    const proto::WifiList& wifiList,
    const Json::Value& experiments) const {
    ::SpeechKit::VoiceServiceSettings settings{getVoiceDialogLanguage()};

    settings.uniProxyUrl = getUniProxyUrl();
    settings.vinsRequestTimeout = std::chrono::seconds(15);
    settings.keepAliveTimeout = std::chrono::seconds(365ll * 24 * 60 * 60);
    settings.processingMode = SpeechKit::AudioProcessingMode::PASS;
    settings.useOnlineSpotterValidation = getOnlineSpotterEnabled();
    settings.synchronizeStatePayload = buildSynchronizeStatePayload(wifiList, experiments);

    settings.vocalizer.voice = SpeechKit::Voice{"shitova.us"};
    settings.vocalizer.speed = 1.0;

    settings.recognizer.model = getRecognizerModel();
    settings.recognizer.vadEnabled = getVadEnabled();
    settings.recognizer.resetStartingSilenceTimeoutOnLocalVad = false;

    settings.oauthToken = authToken;
    if (!passportUid.empty()) {
        settings.biometryGroup = calcMD5Digest(passportUid);
    }

    settings.diagUrl = tryGetString(voiceDialogSettings_, "diagUrl", settings.diagUrl);
    settings.diagPeriod = getVoiceDialogTiming("diagPeriod", settings.diagPeriod);
    settings.lastMsgTimeout = getVoiceDialogTiming("lastMsgTimeout", settings.lastMsgTimeout);

    settings.useUniProxyProtobufProtocol = tryGetBool(voiceDialogSettings_, "useUniProxyProtobuf", false);
    settings.requestStatAckTimeout = getVoiceDialogTiming("requestStatAckTimeoutMs", settings.requestStatAckTimeout);
    settings.sendPingPongTime = tryGetInt(voiceDialogSettings_, "sendPingPongTime", settings.sendPingPongTime);
    settings.pingInterval = getVoiceDialogTiming("pingInterval", settings.pingInterval);
    settings.pongTimeout = getVoiceDialogTiming("pongTimeout", settings.pongTimeout);
    settings.synthesisChunkTimeout = getVoiceDialogTiming("synthesisChunkTimeout", settings.synthesisChunkTimeout);
    settings.connectionTimeout = getVoiceDialogTiming("connectionTimeout", settings.connectionTimeout);
    settings.socketConnectionTimeout = getVoiceDialogTiming("socketConnectionTimeout", settings.socketConnectionTimeout);
    settings.echoPingInterval = getVoiceDialogTiming("echoPingIntervalMs", settings.echoPingInterval);
    settings.vinsRequestTimeout = getVoiceDialogTiming("vinsRequestTimeout", settings.vinsRequestTimeout);
    settings.echoPayloadBytes = tryGetInt64(voiceDialogSettings_, "echoPayloadBytes", settings.echoPayloadBytes);
    settings.logSoundUntilEndOfUtterance = tryGetBool(voiceDialogSettings_, "logSoundUntilEndOfUtterance", settings.logSoundUntilEndOfUtterance);

    settings.backoffMinInterval = tryGetSeconds(voiceDialogSettings_, "backoffMinIntervalSec", settings.backoffMinInterval);
    settings.backoffMaxInterval = tryGetSeconds(voiceDialogSettings_, "backoffMaxIntervalSec", settings.backoffMaxInterval);

    auto& recognizerSettings = settings.recognizer;
    const Json::Value& recognizer = voiceDialogSettings_["recognizer"];
    recognizerSettings.packSoundBuffer = tryGetBool(recognizer, "packSoundBuffer", recognizerSettings.packSoundBuffer);
    recognizerSettings.enablePunctuation = tryGetBool(recognizer, "enablePunctuation", recognizerSettings.enablePunctuation);
    recognizerSettings.enableCapitalization = tryGetBool(recognizer, "enableCapitalization", recognizerSettings.enableCapitalization);
    recognizerSettings.forceReconnectTimeout = tryGetMillis(recognizer, "forceReconnectTimeout", recognizerSettings.forceReconnectTimeout);
    recognizerSettings.startingSilenceTimeout = tryGetMillis(recognizer, "startingSilenceTimeout", recognizerSettings.startingSilenceTimeout);
    recognizerSettings.waitForResultTimeout = tryGetMillis(recognizer, "waitForResultTimeout", recognizerSettings.waitForResultTimeout);

    if (getSpottersEnabled()) {
        settings.activationPhraseSpotter = getActivationSpotterSettings(activationSpotterPath);
        settings.interruptionPhraseSpotter = getInterruptionSpotterSettings(interruptionSpotterPath, activationSpotterPath);
        settings.additionalPhraseSpotter = getAdditionalSpotterSettings(additionalSpotterPath);
    }

    return settings;
}

bool AliceConfig::readBool(const std::string& name, bool defaultValue) const {
    if (backendSystemConfig_.isMember(name)) {
        return readBackendBool(name, defaultValue);
    }

    return readFileBool(name, defaultValue);
}

bool AliceConfig::readFileBool(const std::string& name, bool defaultValue) const {
    return tryGetBool(fileConfig_, name, defaultValue);
}

bool AliceConfig::readBackendBool(const std::string& name, bool defaultValue) const {
    return tryGetBool(backendSystemConfig_, name, defaultValue);
}

int AliceConfig::readInt(const std::string& name, int defaultValue) const {
    if (backendSystemConfig_.isMember(name)) {
        return readBackendInt(name, defaultValue);
    }

    return readFileInt(name, defaultValue);
}

int AliceConfig::readFileInt(const std::string& name, int defaultValue) const {
    return tryGetInt(fileConfig_, name, defaultValue);
}

int AliceConfig::readBackendInt(const std::string& name, int defaultValue) const {
    return tryGetInt(backendSystemConfig_, name, defaultValue);
}

double AliceConfig::readDouble(const std::string& name, double defaultValue) const {
    if (backendSystemConfig_.isMember(name)) {
        return readBackendDouble(name, defaultValue);
    }

    return readFileDouble(name, defaultValue);
}

double AliceConfig::readFileDouble(const std::string& name, double defaultValue) const {
    return tryGetDouble(fileConfig_, name, defaultValue);
}

double AliceConfig::readBackendDouble(const std::string& name, double defaultValue) const {
    return tryGetDouble(backendSystemConfig_, name, defaultValue);
}

std::chrono::milliseconds AliceConfig::readMillis(const std::string& name, std::chrono::milliseconds defaultValue) const {
    if (backendSystemConfig_.isMember(name)) {
        return readBackendMillis(name, defaultValue);
    }

    return readFileMillis(name, defaultValue);
}

std::chrono::milliseconds AliceConfig::readFileMillis(const std::string& name, std::chrono::milliseconds defaultValue) const {
    return tryGetMillis(fileConfig_, name, defaultValue);
}

std::chrono::milliseconds AliceConfig::readBackendMillis(const std::string& name, std::chrono::milliseconds defaultValue) const {
    return tryGetMillis(backendSystemConfig_, name, defaultValue);
}

std::chrono::seconds AliceConfig::readSeconds(const std::string& name, std::chrono::seconds defaultValue) const {
    if (backendSystemConfig_.isMember(name)) {
        return readBackendSeconds(name, defaultValue);
    }

    return readFileSeconds(name, defaultValue);
}

std::chrono::seconds AliceConfig::readFileSeconds(const std::string& name, std::chrono::seconds defaultValue) const {
    return tryGetSeconds(fileConfig_, name, defaultValue);
}

std::chrono::seconds AliceConfig::readBackendSeconds(const std::string& name, std::chrono::seconds defaultValue) const {
    return tryGetSeconds(backendSystemConfig_, name, defaultValue);
}

std::string AliceConfig::readString(const std::string& name, const std::string& defaultValue) const {
    if (backendSystemConfig_.isMember(name)) {
        return readBackendString(name, defaultValue);
    }

    return readFileString(name, defaultValue);
}

std::string AliceConfig::readFileString(const std::string& name, const std::string& defaultValue) const {
    return tryGetString(fileConfig_, name, defaultValue);
}

std::string AliceConfig::readBackendString(const std::string& name, const std::string& defaultValue) const {
    return tryGetString(backendSystemConfig_, name, defaultValue);
}

std::unordered_set<std::string> AliceConfig::readStringSet(
    const std::string& name, const std::unordered_set<std::string>& defaultValue) const {
    if (backendSystemConfig_.isMember(name)) {
        return readBackendStringSet(name, defaultValue);
    }

    return readFileStringSet(name, defaultValue);
}

std::unordered_set<std::string> AliceConfig::readFileStringSet(
    const std::string& name, const std::unordered_set<std::string>& defaultValue) const {
    return tryGetEmplaceableStringSet<std::unordered_set<std::string>>(fileConfig_, name, defaultValue);
}

std::unordered_set<std::string> AliceConfig::readBackendStringSet(
    const std::string& name, const std::unordered_set<std::string>& defaultValue) const {
    return tryGetEmplaceableStringSet<std::unordered_set<std::string>>(backendSystemConfig_, name, defaultValue);
}

Json::Value AliceConfig::readJson(const std::string& name, const Json::Value& defaultValue) const {
    if (backendSystemConfig_.isMember(name)) {
        return readBackendJson(name, defaultValue);
    }

    return readFileJson(name, defaultValue);
}

Json::Value AliceConfig::readFileJson(const std::string& name, const Json::Value& defaultValue) const {
    return tryGetJson(fileConfig_, name, defaultValue);
}

Json::Value AliceConfig::readBackendJson(const std::string& name, const Json::Value& defaultValue) const {
    return tryGetJson(backendSystemConfig_, name, defaultValue);
}

SpeechKit::Language AliceConfig::getVoiceDialogLanguage() const {
    if (auto language = tryGetString(voiceDialogSettings_, "language"); !language.empty()) {
        return SpeechKit::Language{std::move(language)};
    }

    return SpeechKit::Language::russian;
}

std::chrono::milliseconds AliceConfig::getVoiceDialogTiming(
    const std::string& name, std::chrono::milliseconds defaultValue) const {
    return tryGetMillis(voiceDialogSettings_, name, defaultValue);
}

std::chrono::milliseconds AliceConfig::getVoiceDialogTimingWithFileDefault(
    const std::string& name, std::chrono::milliseconds defaultValue) const {
    if (voiceDialogSettings_.isMember(name)) {
        return getVoiceDialogTiming(name, defaultValue);
    }

    return readFileMillis(name, defaultValue);
}

void AliceConfig::setSpotterLoggingSettings(
    ::SpeechKit::PhraseSpotterSettings& settings) const {
    const auto logChannels = getSpeechkitLogChannels();
    Json::Value spotterLoggingPayloadExtra;
    if (!vqeInfo_.type.empty()) {
        spotterLoggingPayloadExtra["VQEType"] = vqeInfo_.type;
    }
    if (!vqeInfo_.preset.empty()) {
        spotterLoggingPayloadExtra["VQEPreset"] = vqeInfo_.preset;
    }

    settings.soundLoggerSettings.soundLengthBeforeTrigger = getSpotterLoggingHead();
    settings.soundLoggerSettings.soundLengthAfterTrigger = getSpotterLoggingTail();
    settings.soundLoggerSettings.soundFormat = getSoundLoggingFormat();
    settings.soundLoggerSettings.payloadExtra = jsonToString(spotterLoggingPayloadExtra);
    settings.soundLoggerSettings.channelNamesToSend = {logChannels.begin(), logChannels.end()};
    settings.soundLoggerSettings.streamingLength = getVoiceDialogTiming("spotterLoggingStreamingLengthMillis", settings.soundLoggerSettings.streamingLength);

    settings.rareEventPercent = tryGetInt(voiceDialogSettings_, "spotterLoggingRareEventPercent", 0);
    settings.rareEventSoundLoggerSettings.soundFormat = SpeechKit::SoundFormat::PCM;
    settings.rareEventSoundLoggerSettings.soundLengthBeforeTrigger = getVoiceDialogTiming("spotterLoggingRareEventHeadMillis", settings.rareEventSoundLoggerSettings.soundLengthBeforeTrigger);
    settings.rareEventSoundLoggerSettings.soundLengthAfterTrigger = getVoiceDialogTiming("spotterLoggingRareEventTailMillis", settings.rareEventSoundLoggerSettings.soundLengthAfterTrigger);
    settings.rareEventSoundLoggerSettings.payloadExtra = jsonToString(spotterLoggingPayloadExtra);
    settings.rareEventSoundLoggerSettings.channelNamesToSend = {logChannels.begin(), logChannels.end()};
    settings.rareEventSoundLoggerSettings.streamingLength = getVoiceDialogTiming("spotterLoggingRareStreamingLengthMillis", settings.rareEventSoundLoggerSettings.streamingLength);

    settings.logStatisticsInterval = getSpotterStatisticsLoggingIntevalMillis();
}

Json::Value AliceConfig::getUniProxyPingerConfig() const {
    return tryGetJson(backendSystemConfig_, "uniProxyPinger");
}

std::optional<std::string> AliceConfig::getSpeechkitLogLevel() const {
    if (backendSystemConfig_["logging"]["speechkit"]["level"].isString()) {
        return backendSystemConfig_["logging"]["speechkit"]["level"].asString();
    }
    return std::nullopt;
}

bool AliceConfig::getPreprocessVinsResponse() const {
    return readFileBool("preprocessVinsResponse", false);
}

Json::Value AliceConfig::getSystemConfigEnv() const {
    return tryGetJson(backendSystemConfig_, "env");
}

std::string AliceConfig::getTestBuckets() const {
    return tryGetString(getSystemConfigEnv(), "test_buckets");
}

std::string AliceConfig::getTestIds() const {
    return tryGetString(getSystemConfigEnv(), "testids");
}

bool AliceConfig::getIotCapabilityEnabled() const {
    return readBool("iotCapabilityEnabled", false);
}

bool AliceConfig::getUseParallelRequests() const {
    return readBool("useParallelRequests", true);
}

std::chrono::seconds AliceConfig::getEndpointsUpdateTimeout() const {
    return readSeconds("endpointsUpdateTimeoutSeconds", std::chrono::seconds(5));
}

std::string AliceConfig::getUserAgent() const {
    return device_ == nullptr ? "" : device_->getUserAgent();
}

bool AliceConfig::isLongListeningEnabled() const {
    if (getCommandSpottersEnabled() && isSpotterEnabled(SpotterTypes::NAVIGATION)) {
        return false;
    }

    return getSpottersEnabled() && getLongListeningEnabled();
}

bool AliceConfig::isSpotterEnabled(const std::string& name) const {
    const Json::Value disabledSpotters = getDisabledCommandSpotterTypes();
    for (const auto& disabledSpotter : disabledSpotters) {
        if (disabledSpotter.isString() && (disabledSpotter.asString() == name)) {
            return false;
        }
    }

    return true;
}

void AliceConfig::setVqeInfo(YandexIO::ChannelData::VqeInfo vqeInfo)
{
    vqeInfo_ = std::move(vqeInfo);
}

const YandexIO::ChannelData::VqeInfo& AliceConfig::getVqeInfo() const {
    return vqeInfo_;
}

const std::set<std::string>& AliceConfig::getBlockShowCardDirectiveNames() const {
    return blockShowCardDirectiveNames_;
}

std::string AliceConfig::getDirectiveSequencerStatePath() const {
    return readFileString("directiveSequencerStatePath", "");
}

SpeechKit::SoundFormat::Type AliceConfig::getSoundLoggingFormat() const {
    const auto formatString = tryGetString(voiceDialogSettings_, "soundLoggingFormat", "opus");
    if (formatString == "pcm") {
        return SpeechKit::SoundFormat::PCM;
    } else if (formatString == "opus") {
        return SpeechKit::SoundFormat::OPUS;
    }

    YIO_LOG_WARN("Unknown soundLoggingFormat: " << formatString << ". Log opus sound.");
    return SpeechKit::SoundFormat::OPUS;
}

std::string AliceConfig::buildSynchronizeStatePayload(const proto::WifiList& wifiList, const Json::Value& experiments) const {
    Json::Value synchronizeStatePayload;
    synchronizeStatePayload["request"]["experiments"] = experiments;
    synchronizeStatePayload["enable_realtime_streamer"] = Json::Value(getEnableRealtimeStreamer() ? "true" : "false");
    synchronizeStatePayload["vins"]["application"]["enable_realtime_streamer"] = synchronizeStatePayload["enable_realtime_streamer"];

    try {
        Json::Value networks(Json::arrayValue);
        for (const auto& item : wifiList.hotspots()) {
            if (!item.mac().empty()) {
                Json::Value jsonItem;
                jsonItem["mac"] = item.mac();
                jsonItem["signal_strength"] = item.rssi();

                networks.append(jsonItem);
            }
        }
        synchronizeStatePayload["wifi_networks"] = networks;
    } catch (const std::runtime_error& e) {
        YIO_LOG_ERROR_EVENT("AliceConfig.FailedGetWifiNetworks", "Can't get wifi_networks");
    }

    return jsonToString(synchronizeStatePayload);
}

SpeechKit::PhraseSpotterSettings AliceConfig::getActivationSpotterSettings(const std::string& path) const {
    SpeechKit::PhraseSpotterSettings settings{path};

    setSpotterLoggingSettings(settings);

    settings.resetAfterTrigger = tryGetBool(voiceDialogSettings_, "resetAfterTrigger", false);
    settings.soundLoggerSettings.subThresholdDelay = tryGetMillis(voiceDialogSettings_["activationSpotter"],
                                                                  "subThresholdDelayMs", settings.soundLoggerSettings.subThresholdDelay);
    settings.rareEventSoundLoggerSettings.subThresholdDelay = tryGetMillis(voiceDialogSettings_["activationSpotter"],
                                                                           "subThresholdDelayMs", settings.rareEventSoundLoggerSettings.subThresholdDelay);

    return settings;
}

SpeechKit::PhraseSpotterSettings AliceConfig::getInterruptionSpotterSettings(
    const std::string& interruptionPath,
    const std::string& activationPath) const {
    SpeechKit::PhraseSpotterSettings settings{interruptionPath.empty() ? activationPath : interruptionPath};

    setSpotterLoggingSettings(settings);

    settings.soundLoggerSettings.subThresholdDelay = tryGetMillis(voiceDialogSettings_["interruptionSpotter"],
                                                                  "subThresholdDelayMs", settings.soundLoggerSettings.subThresholdDelay);
    settings.rareEventSoundLoggerSettings.subThresholdDelay = tryGetMillis(voiceDialogSettings_["interruptionSpotter"],
                                                                           "subThresholdDelayMs", settings.rareEventSoundLoggerSettings.subThresholdDelay);

    return settings;
}

SpeechKit::PhraseSpotterSettings AliceConfig::getAdditionalSpotterSettings(const std::string& path) const {
    SpeechKit::PhraseSpotterSettings settings{""};

    if (path.empty()) {
        return settings;
    }

    const auto head = tryGetMillis(voiceDialogSettings_, "additionalSpotterLoggingHeadMillis",
                                   std::chrono::milliseconds::zero());
    const auto tail = tryGetMillis(voiceDialogSettings_, "additionalSpotterLoggingTailMillis",
                                   std::chrono::milliseconds::zero());

    if (tail + head <= std::chrono::milliseconds::zero()) {
        return settings;
    }

    setSpotterLoggingSettings(settings);

    settings.modelPath = path;
    settings.soundLoggerSettings.soundLengthBeforeTrigger = head;
    settings.soundLoggerSettings.soundLengthAfterTrigger = tail;
    settings.soundLoggerSettings.subThresholdDelay = tryGetMillis(voiceDialogSettings_["additionalSpotter"],
                                                                  "subThresholdDelayMs", settings.soundLoggerSettings.subThresholdDelay);
    settings.rareEventSoundLoggerSettings.subThresholdDelay = tryGetMillis(voiceDialogSettings_["additionalSpotter"],
                                                                           "subThresholdDelayMs", settings.rareEventSoundLoggerSettings.subThresholdDelay);

    return settings;
}
