#pragma once

#include <yandex_io/libs/configuration/features_config.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/protos/quasar_proto.pb.h>
#include <yandex_io/sdk/audio_source/defines.h>
#include <speechkit/core/include/speechkit/VoiceServiceSettings.h>

#include <json/value.h>

#include <chrono>
#include <memory>
#include <optional>
#include <string>
#include <string_view>
#include <unordered_set>

namespace quasar {

    class AliceConfig {
    public:
        AliceConfig(std::shared_ptr<YandexIO::IDevice> device, Json::Value fileConfig);

        bool setReceivedConfig(std::string_view jsonConfigStr);

        bool hasReceivedConfig() const;
        bool hasSystemConfig() const;
        const Json::Value& getSystemConfig() const;
        bool hasAuxiliaryDeviceConfig() const;
        const Json::Value& getAuxiliaryDeviceConfig() const;

        bool getWithLocalVins() const;
        bool getUseCustomJinglePlayer() const;
        bool getRequireAuthorization() const;
        bool getUseAudioClientTtsPlayer() const;
        std::chrono::milliseconds getSilenceTimeAfterTts() const;
        bool getJingle() const;
        bool getSpottersEnabled() const;
        bool getCommandSpottersEnabled() const;
        bool getOnlineSpotterEnabled() const;
        bool getLongListeningEnabled() const;
        int getSoundLogMaxParallelSendings() const;
        bool getPrefetchEnabled() const;
        bool getSendNavigationSpotterLog() const;
        bool getSendCommandSpotterLog() const;
        std::chrono::seconds getStartDelay() const;
        bool getVadEnabled() const;
        bool getEnableRealtimeStreamer() const;
        std::chrono::milliseconds getSubThresholdSendRate() const;
        bool getDnsCacheEnabled() const;
        bool environmentStateEnabled() const;

        std::string getBassUrl() const;
        std::string getVinsUrl() const;
        std::string getUniProxyUrl() const;
        std::string getSingleDialogMode() const;
        std::string getSpeechkitDumpPath() const;
        std::string getQuasmodromGroup() const;
        std::string getQuasmodromSubgroup() const;
        std::string getApiKey() const;
        std::string getRecognizerModel() const;

        double getRMSCorrection() const;
        bool getRMSOverVqe() const;
        std::unordered_set<std::string> getSpotterRMSChannels() const;
        std::unordered_set<std::string> getSpeechkitLogChannels() const;

        ::SpeechKit::VoiceServiceSettings getVoiceServiceSettings(
            const std::string& authToken,
            const std::string& passportUid,
            const std::string& activationSpotterPath,
            const std::string& interruptionSpotterPath,
            const std::string& additionalSpotterPath,
            const proto::WifiList& wifiList,
            const Json::Value& experiments) const;
        std::string buildSynchronizeStatePayload(const proto::WifiList& wifiList, const Json::Value& experiments) const;

        void setSpotterLoggingSettings(SpeechKit::PhraseSpotterSettings& settings) const;

        std::string getDeviceStateSenderCachePath() const;
        Json::Value getDeviceStateSenderSettings() const;

        std::string getContentAccess() const;
        std::string getChildContentAccess() const;
        Json::Value getSpotter() const;

        Json::Value getDisabledCommandPhrases() const;
        Json::Value getDisabledCommandSpotterTypes() const;

        bool getAudioSenderMode() const;

        std::string getSpotterCollisionMode() const;
        Json::Value getRandomSoundLogger() const;
        std::unordered_set<std::string> getRandomSoundLoggerChannels() const;

        Json::Value getSmartActivation() const;

        Json::Value getUniProxyPingerConfig() const;

        std::optional<std::string> getSpeechkitLogLevel() const;
        bool getPreprocessVinsResponse() const;

        Json::Value getSystemConfigEnv() const;
        std::string getTestBuckets() const;
        std::string getTestIds() const;

        bool getIotCapabilityEnabled() const;

        bool getUseParallelRequests() const;
        std::chrono::seconds getEndpointsUpdateTimeout() const;

        void setFeaturesConfig(FeaturesConfig value);
        const FeaturesConfig& getFeaturesConfig() const;

        std::string getUserAgent() const;

        bool isLongListeningEnabled() const;
        bool isSpotterEnabled(const std::string& name) const;

        void setVqeInfo(YandexIO::ChannelData::VqeInfo vqeInfo);
        const YandexIO::ChannelData::VqeInfo& getVqeInfo() const;

        const std::set<std::string>& getBlockShowCardDirectiveNames() const;

        std::string getDirectiveSequencerStatePath() const;

    private:
        bool readBool(const std::string& name, bool defaultValue) const;
        bool readFileBool(const std::string& name, bool defaultValue) const;
        bool readBackendBool(const std::string& name, bool defaultValue) const;
        int readInt(const std::string& name, int defaultValue) const;
        int readFileInt(const std::string& name, int defaultValue) const;
        int readBackendInt(const std::string& name, int defaultValue) const;
        double readDouble(const std::string& name, double defaultValue) const;
        double readFileDouble(const std::string& name, double defaultValue) const;
        double readBackendDouble(const std::string& name, double defaultValue) const;
        std::chrono::milliseconds readMillis(const std::string& name, std::chrono::milliseconds defaultValue) const;
        std::chrono::milliseconds readFileMillis(const std::string& name, std::chrono::milliseconds defaultValue) const;
        std::chrono::milliseconds readBackendMillis(const std::string& name, std::chrono::milliseconds defaultValue) const;
        std::chrono::seconds readSeconds(const std::string& name, std::chrono::seconds defaultValue) const;
        std::chrono::seconds readFileSeconds(const std::string& name, std::chrono::seconds defaultValue) const;
        std::chrono::seconds readBackendSeconds(const std::string& name, std::chrono::seconds defaultValue) const;
        std::string readString(const std::string& name, const std::string& defaultValue) const;
        std::string readFileString(const std::string& name, const std::string& defaultValue) const;
        std::string readBackendString(const std::string& name, const std::string& defaultValue) const;
        std::unordered_set<std::string> readStringSet(const std::string& name,
                                                      const std::unordered_set<std::string>& defaultValue) const;
        std::unordered_set<std::string> readFileStringSet(const std::string& name,
                                                          const std::unordered_set<std::string>& defaultValue) const;
        std::unordered_set<std::string> readBackendStringSet(const std::string& name,
                                                             const std::unordered_set<std::string>& defaultValue) const;

        Json::Value readJson(const std::string& name, const Json::Value& defaultValue) const;
        Json::Value readFileJson(const std::string& name, const Json::Value& defaultValue) const;
        Json::Value readBackendJson(const std::string& name, const Json::Value& defaultValue) const;

        SpeechKit::SoundFormat::Type getSoundLoggingFormat() const;

        SpeechKit::Language getVoiceDialogLanguage() const;

        std::chrono::milliseconds getVoiceDialogTiming(
            const std::string& name,
            std::chrono::milliseconds defaultValue) const;
        std::chrono::milliseconds getVoiceDialogTimingWithFileDefault(
            const std::string& name,
            std::chrono::milliseconds defaultValue) const;
        std::chrono::milliseconds getSpotterStatisticsLoggingIntevalMillis() const;

        std::chrono::milliseconds getSpotterLoggingHead() const;
        std::chrono::milliseconds getSpotterLoggingTail() const;

        SpeechKit::PhraseSpotterSettings getActivationSpotterSettings(const std::string& path) const;
        SpeechKit::PhraseSpotterSettings getInterruptionSpotterSettings(const std::string& interruptionPath, const std::string& activationPath) const;
        SpeechKit::PhraseSpotterSettings getAdditionalSpotterSettings(const std::string& path) const;

        Json::Value getBlockShowCardDirectiveJson() const;

    private:
        std::shared_ptr<YandexIO::IDevice> device_;
        const Json::Value fileConfig_;
        const Json::Value fileVoiceDialogSettings_;

        Json::Value backendSystemConfig_;
        Json::Value backendAccountConfig_;
        Json::Value backendDeviceConfig_;
        Json::Value auxiliaryDeviceConfig_;
        Json::Value voiceDialogSettings_;
        static ::SpeechKit::VoiceServiceSettings defaultSettings_;
        YandexIO::ChannelData::VqeInfo vqeInfo_;
        bool configReceived_ = false;
        std::set<std::string> blockShowCardDirectiveNames_;
    };

} // namespace quasar
