#include "alice_service.h"

#include "speechkit_facade/quasar_voice_dialog_linux_impl.h"

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/device/defines.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/delay_timings_policy/delay_timings_policy.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/telemetry/telemetry.h>
#include <yandex_io/libs/voice_stats/voice_stats.h>

#include <yandex_io/protos/model_objects.pb.h>

#include <memory>

YIO_DEFINE_LOG_MODULE("alice");

using namespace quasar;
using namespace quasar::proto;

AliceService::AliceService(
    std::shared_ptr<YandexIO::IDevice> device,
    NAlice::TEndpoint::EEndpointType endpointType,
    std::shared_ptr<YandexIO::SDKInterface> sdk,
    std::shared_ptr<ipc::IIpcFactory> ipcFactory,
    std::shared_ptr<quasar::IAuthProvider> authProvider,
    std::shared_ptr<IClockTowerProvider> clockTowerProvider,
    std::shared_ptr<IDeviceStateProvider> deviceStateProvider,
    std::shared_ptr<IGlagolClusterProvider> glagolClusterProvider,
    std::shared_ptr<IMultiroomProvider> multiroomProvider,
    std::shared_ptr<IStereoPairProvider> stereoPairProvider,
    std::shared_ptr<IUserConfigProvider> userConfigProvider,
    SpeechKit::AudioPlayer::SharedPtr audioPlayer,
    std::shared_ptr<SelfDestroyer> selfDestroyer)
    : callbackExecutor_(std::make_shared<NamedCallbackQueue>("AliceService"))
    , device_(std::move(device))
    , endpointType_(endpointType)
    , sdk_(std::move(sdk))
    , ipcFactory_(std::move(ipcFactory))
    , authProvider_(std::move(authProvider))
    , clockTowerProvider_(std::move(clockTowerProvider))
    , deviceStateProvider_(std::move(deviceStateProvider))
    , glagolClusterProvider_(std::move(glagolClusterProvider))
    , multiroomProvider_(std::move(multiroomProvider))
    , stereoPairProvider_(std::move(stereoPairProvider))
    , userConfigProvider_(std::move(userConfigProvider))
    , speechkitAudioPlayer_(std::move(audioPlayer))
    , selfDestroyer_(std::move(selfDestroyer))
    , aliceConfig_(device_, device_->configuration()->getServiceConfig(SpeechkitEndpoint::SERVICE_NAME))
    , aliceDeviceState_(std::make_shared<AliceDeviceState>(
          device_->deviceId(), clockTowerProvider_, deviceStateProvider_, EnvironmentStateHolder{device_->deviceId(), device_->telemetry()}))
{
}

std::string AliceService::getServiceName() const {
    return SpeechkitEndpoint::SERVICE_NAME;
}

void AliceService::start() {
    auto config = device_->configuration()->getServiceConfig(SpeechkitEndpoint::SERVICE_NAME);

    auto voiceStats = VoiceStats::create();

    randomSoundLogger_ = std::make_shared<RandomSoundLogger>(callbackExecutor_, device_->deviceId(), aliceConfig_, *aliceDeviceState_);

    aliceAudioSource_ = AliceAudioSource::create([this](YandexIO::ChannelData::VqeInfo vqeInfo) {
        if (dialogListener_) {
            dialogListener_->setVqeInfo(std::move(vqeInfo));
        }
    }, voiceStats);
    aliceAudioSource_->subscribe(randomSoundLogger_);

    ioAudioSourceClient_ = YandexIO::createAudioSourceClient(ipcFactory_);
    ioAudioSourceClient_->addListener(aliceAudioSource_);

    dialogListener_ = std::make_shared<SpeechkitEndpoint>(
        device_,
        endpointType_,
        sdk_,
        ipcFactory_,
        authProvider_,
        clockTowerProvider_,
        deviceStateProvider_,
        glagolClusterProvider_,
        multiroomProvider_,
        stereoPairProvider_,
        userConfigProvider_,
        speechkitAudioPlayer_,
        aliceAudioSource_,
        ioAudioSourceClient_,
        voiceStats,
        randomSoundLogger_,
        selfDestroyer_,
        callbackExecutor_,
        std::make_unique<quasar::BackoffRetriesWithRandomPolicy>(getRandomSeed(device_->deviceId())),
        aliceConfig_,
        aliceDeviceState_);

    quasarVoiceDialog_ = std::make_shared<QuasarVoiceDialogLinuxImpl>(
        device_, dialogListener_, aliceAudioSource_, speechkitAudioPlayer_);

    dialogListener_->start(quasarVoiceDialog_);

    /* Start receive data only after SpeechkitEndpoint initialization */
    ioAudioSourceClient_->start();
    /* Get all channels by default, raw channels are needed to calc RMSes */
    ioAudioSourceClient_->subscribeToChannels(YandexIO::RequestChannelType::ALL);
}

AliceService::~AliceService() {
    callbackExecutor_->destroy();
}
