#include "audio_focus_dispatcher.h"

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/protos/quasar_proto.pb.h>

YIO_DEFINE_LOG_MODULE("alice");

using namespace quasar;

AudioFocusDispatcher::AudioFocusDispatcher(std::shared_ptr<ipc::IConnector> toMediad,
                                           std::shared_ptr<ipc::IConnector> toInterfaced,
                                           std::shared_ptr<ipc::IConnector> toAudioClientd)
    : toMediad_(std::move(toMediad))
    , toInterfaced_(std::move(toInterfaced))
    , toAudioClientd_(std::move(toAudioClientd))
{
}

std::optional<proto::AudioChannel> AudioFocusDispatcher::getAudioFocus() const {
    return currentFocusedChannel_;
}

void AudioFocusDispatcher::setAudioFocus(proto::AudioChannel channel)
{
    YIO_LOG_INFO("setAudioFocus channel=" << proto::AudioChannel_Name(channel));

    currentFocusedChannel_ = channel;

    applyFocusToMediad();
    applyFocusToAudioClientd();
    applyFocusToInterfaced();
}

void AudioFocusDispatcher::onMediadConnected()
{
    if (!toMediadConnected_) {
        toMediadConnected_ = true;
        applyFocusToMediad();
    }
}

void AudioFocusDispatcher::onInterfacedConnected()
{
    if (!toInterfacedConnected_) {
        toInterfacedConnected_ = true;
        applyFocusToInterfaced();
    }
}

void AudioFocusDispatcher::onAudioClientdConnected()
{
    if (!toAudioClientdConnected_) {
        toAudioClientdConnected_ = true;
        applyFocusToAudioClientd();
    }
}

void AudioFocusDispatcher::applyFocusToMediad()
{
    if (!toMediadConnected_ || !currentFocusedChannel_.has_value()) {
        return;
    }

    switch (currentFocusedChannel_.value()) {
        case proto::CONTENT_CHANNEL: {
            takeMediadFocus();
            break;
        }
        case proto::DIALOG_CHANNEL: {
            freeMediadFocus();
            break;
        }
        default: {
            YIO_LOG_ERROR_EVENT("AudioFocusDispatcher.InvalidChannel", "Unsupported channel=" << formatCurrentFocusChannel());
        }
    }
}

void AudioFocusDispatcher::applyFocusToAudioClientd()
{
    if (!toAudioClientdConnected_ || !currentFocusedChannel_.has_value()) {
        return;
    }

    auto message = ipc::buildMessage([&](auto& msg) {
        msg.mutable_media_request()->set_current_focused_channel(currentFocusedChannel_.value());
    });

    toAudioClientd_->sendMessage(message);
}

void AudioFocusDispatcher::applyFocusToInterfaced()
{
    if (!toInterfacedConnected_ || !currentFocusedChannel_.has_value()) {
        return;
    }

    switch (currentFocusedChannel_.value()) {
        case proto::CONTENT_CHANNEL: {
            takeInterfacedFocus();
            break;
        }
        case proto::DIALOG_CHANNEL: {
            freeInterfacedFocus();
            break;
        }
        default: {
            YIO_LOG_ERROR_EVENT("AudioFocusDispatcher.InvalidChannel", "Unsupported channel=" << formatCurrentFocusChannel());
        }
    }
}

void AudioFocusDispatcher::takeMediadFocus()
{
    proto::QuasarMessage command;
    command.mutable_media_request()->mutable_take_audio_focus();
    toMediad_->sendMessage(std::move(command));
}

void AudioFocusDispatcher::freeMediadFocus()
{
    proto::QuasarMessage command;
    command.mutable_media_request()->mutable_free_audio_focus();
    toMediad_->sendMessage(std::move(command));
}

void AudioFocusDispatcher::takeInterfacedFocus()
{
    proto::QuasarMessage command;
    command.mutable_media_request()->mutable_take_audio_focus();
    toInterfaced_->sendMessage(std::move(command));
}

void AudioFocusDispatcher::freeInterfacedFocus()
{
    proto::QuasarMessage command;
    command.mutable_media_request()->mutable_free_audio_focus();
    toInterfaced_->sendMessage(std::move(command));
}

std::string AudioFocusDispatcher::formatCurrentFocusChannel() const {
    if (currentFocusedChannel_.has_value()) {
        return proto::AudioChannel_Name(currentFocusedChannel_.value());
    } else {
        return "null";
    }
}
