#include "alarm_capability.h"

#include <yandex_io/libs/base/directives.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/ete_metrics/ete_util.h>
#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/protobuf_utils/json.h>
#include <yandex_io/libs/logging/logging.h>

#include <yandex_io/capabilities/device_state/converters/converters.h>

#include <util/system/yassert.h>

YIO_DEFINE_LOG_MODULE("alarm_capability");

using namespace quasar;
using namespace YandexIO;

namespace {
    constexpr auto MEDIA_ALARM_TIMEOUT = std::chrono::seconds(15);
    const auto REMOTE_REGISTRY_NAME = "AlarmCapability";
} // namespace

AlarmCapability::AlarmCapability(
    std::shared_ptr<quasar::ICallbackQueue> worker,
    std::shared_ptr<IDevice> device,
    std::shared_ptr<ipc::IIpcFactory> ipcFactory,
    std::weak_ptr<YandexIO::IRemotingRegistry> remotingRegistry,
    YandexIO::IDirectiveProcessorWeakPtr directiveProcessor,
    std::weak_ptr<YandexIO::IAliceCapability> aliceCapability,
    std::weak_ptr<YandexIO::IFilePlayerCapability> filePlayerCapability,
    std::weak_ptr<YandexIO::IPlaybackControlCapability> playbackCapability,
    std::shared_ptr<YandexIO::IDeviceStateCapability> deviceStateCapability)
    : IRemoteObject(std::move(remotingRegistry))
    , worker_(std::move(worker))
    , device_(std::move(device))
    , directiveProcessor_(std::move(directiveProcessor))
    , alarmsConnector_(ipcFactory->createIpcConnector("alarmd"))
    , aliceCapability_(std::move(aliceCapability))
    , filePlayerCapability_(std::move(filePlayerCapability))
    , playbackCapability_(std::move(playbackCapability))
    , deviceStateCapability_(std::move(deviceStateCapability))
{
    Y_VERIFY(worker_);
    Y_VERIFY(deviceStateCapability_);

    alarmsConnector_->setMessageHandler([this](const auto& message) {
        worker_->add([this, message]() {
            handleAlarmdMessage(message);
        });
    });

    alarmsConnector_->connectToService();
}

AlarmCapability::~AlarmCapability()
{
    if (auto remotingRegistry = getRemotingRegistry().lock()) {
        remotingRegistry->removeRemoteObject(REMOTE_REGISTRY_NAME);
    }
    alarmsConnector_->shutdown();
}

void AlarmCapability::init() {
    if (auto remotingRegistry = getRemotingRegistry().lock()) {
        remotingRegistry->addRemoteObject(REMOTE_REGISTRY_NAME, weak_from_this());
    }
}

void AlarmCapability::addListener(std::weak_ptr<IAlarmCapabilityListener> listener)
{
    wlisteners_.push_back(listener);
}

void AlarmCapability::removeListener(std::weak_ptr<IAlarmCapabilityListener> listener)
{
    const auto iter = std::find_if(wlisteners_.begin(), wlisteners_.end(),
                                   [listener](const auto& wp) {
                                       return !(wp.owner_before(listener) || listener.owner_before(wp));
                                   });

    if (iter != wlisteners_.end()) {
        wlisteners_.erase(iter);
    }
}

void AlarmCapability::stopAlarm() {
    if (const auto dp = directiveProcessor_.lock()) {
        Json::Value payload;
        payload["stop_media"] = true;
        dp->addDirectives({Directive::createLocalAction(Directives::ALARM_STOP, payload)});
    }
}

const std::string& AlarmCapability::getHandlerName() const {
    static const std::string s_name = "AlarmCapability";
    return s_name;
}

const std::set<std::string>& AlarmCapability::getSupportedDirectiveNames() const {
    static const std::set<std::string> s_names = {
        Directives::SET_TIMER,
        Directives::CANCEL_TIMER,
        Directives::PAUSE_TIMER,
        Directives::RESUME_TIMER,
        Directives::ALARMS_UPDATE,
        Directives::ALARM_START,
        Directives::ALARM_STOP,
        Directives::ALARM_SET_SOUND,
        Directives::ALARM_RESET_SOUND,
        Directives::ALARM_SET_MAX_LEVEL,
    };

    return s_names;
}

void AlarmCapability::handleDirective(const std::shared_ptr<Directive>& directive)
{
    const auto& data = directive->getData();

    YIO_LOG_INFO("Alarm capability directive \"" << data.name << "\", request_id=" << data.requestId);
    if (!data.requestId.empty()) {
        Json::Value etePayload;
        appendETEHeader(data.requestId, etePayload);
        etePayload["command_name"] = data.name;

        device_->telemetry()->reportEvent("alarm_directive_received", jsonToString(etePayload));
    }

    if (directive->is(Directives::SET_TIMER)) {
        handleSetTimer(data.payload);
    } else if (directive->is(Directives::ALARMS_UPDATE)) {
        handleAlarmsUpdate(data.payload);
    } else if (directive->is(Directives::CANCEL_TIMER)) {
        handleCancelTimer(data.payload);
    } else if (directive->is(Directives::PAUSE_TIMER)) {
        handlePauseTimer(data.payload);
    } else if (directive->is(Directives::RESUME_TIMER)) {
        handleResumeTimer(data.payload);
    } else if (directive->is(Directives::ALARM_START)) {
        handleAlarmStartDirective(data.payload);
    } else if (directive->is(Directives::ALARM_STOP)) {
        handleAlarmStopDirective(data.payload);
    } else if (directive->is(Directives::ALARM_SET_SOUND)) {
        handleAlarmSetSound(data.payload);
    } else if (directive->is(Directives::ALARM_RESET_SOUND)) {
        handleAlarmResetSound();
    } else if (directive->is(Directives::ALARM_SET_MAX_LEVEL)) {
        handleAlarmSetMaxLevel(data.payload);
    }
}

void AlarmCapability::cancelDirective(const std::shared_ptr<Directive>& /*directive*/)
{
    // do nothing
}

void AlarmCapability::prefetchDirective(const std::shared_ptr<Directive>& /*directive*/)
{
    // do nothing
}

void AlarmCapability::handleSetTimer(const Json::Value& payload)
{
    YIO_LOG_INFO("Setting timer: " << jsonToString(payload, true));

    if (payload["directives"].isNull()) {
        auto fireDelay = std::chrono::seconds{getInt64(payload, "duration")};

        YIO_LOG_INFO("Add timer, fireDelay=" << std::to_string(fireDelay.count()) << " sec");
        sendAddAlarmMessage(fireDelay);
    } else {
        std::vector<AlarmDirective> directives;
        directives.reserve(payload["directives"].size());

        std::stringstream ss;
        for (const auto& directive : payload["directives"]) {
            directives.emplace_back(AlarmDirective{getString(directive, "name"), jsonToString(directive["payload"])});
            ss << directives.back().name << ", ";
        }

        std::chrono::seconds fireDelay;
        if (payload.isMember("duration")) {
            YIO_LOG_INFO("Timer has DURATION field");
            fireDelay = std::chrono::seconds{getInt64(payload, "duration")};
        } else {
            YIO_LOG_INFO("Timer has TIMESTAMP field");
            auto timestamp = std::chrono::seconds{getInt64(payload, "timestamp")};
            auto nowts = std::chrono::system_clock::now().time_since_epoch();
            fireDelay = std::chrono::duration_cast<std::chrono::seconds>(timestamp - nowts);
        }

        YIO_LOG_INFO("Add timer with directives, fireDelay=" << std::to_string(fireDelay.count()) << " sec, directives: " << ss.str());
        sendAddAlarmMessage(fireDelay, std::move(directives));
    }
}

void AlarmCapability::sendAddAlarmMessage(std::chrono::seconds fireDelay, std::vector<AlarmDirective> directives)
{
    auto message = ipc::buildMessage([fireDelay, &directives](auto& msg) {
        auto uuid = makeUUID();
        auto nowts = std::chrono::system_clock::now().time_since_epoch();

        if (directives.size() == 0) {
            msg.mutable_alarm_message()->mutable_add_alarm()->set_alarm_type(proto::Alarm::TIMER);
        } else {
            msg.mutable_alarm_message()->mutable_add_alarm()->set_alarm_type(proto::Alarm::COMMAND_TIMER);
        }
        msg.mutable_alarm_message()->mutable_add_alarm()->set_id(TString(uuid));
        msg.mutable_alarm_message()->mutable_add_alarm()->set_start_timestamp_ms(std::chrono::duration_cast<std::chrono::milliseconds>(nowts).count());
        msg.mutable_alarm_message()->mutable_add_alarm()->set_duration_seconds(std::chrono::duration_cast<std::chrono::seconds>(fireDelay).count());

        for (const auto& directive : directives) {
            auto commandMessage = msg.mutable_alarm_message()->mutable_add_alarm()->mutable_command_list()->Add();
            commandMessage->set_name(TString(directive.name));
            commandMessage->set_payload(TString(directive.payload));
        }
    });

    alarmsConnector_->sendMessage(message);
}

void AlarmCapability::handleCancelTimer(const Json::Value& payload)
{
    const auto timerId = getString(payload, "timer_id");
    YIO_LOG_INFO("Cancelling timer " << timerId);

    auto message = ipc::buildMessage([&timerId](auto& msg) {
        msg.mutable_alarm_message()->set_remove_alarm_id(TString(timerId));
    });
    alarmsConnector_->sendMessage(message);
}

void AlarmCapability::handlePauseTimer(const Json::Value& payload) {
    const auto timerId = getString(payload, "timer_id");
    YIO_LOG_INFO("Pausing timer " << timerId);

    auto message = ipc::buildMessage([&timerId](auto& msg) {
        msg.mutable_alarm_message()->set_pause_alarm_id(TString(timerId));
    });
    alarmsConnector_->sendMessage(message);
}

void AlarmCapability::handleResumeTimer(const Json::Value& payload) {
    const auto timerId = getString(payload, "timer_id");
    YIO_LOG_INFO("Resuming timer " << timerId);

    auto message = ipc::buildMessage([&timerId](auto& msg) {
        msg.mutable_alarm_message()->set_resume_alarm_id(TString(timerId));
    });
    alarmsConnector_->sendMessage(message);
}

void AlarmCapability::handleAlarmStartDirective(const Json::Value& payload)
{
    proto::Alarm alarm;
    if (!alarm.ParseFromString(payload["alarm"].asString())) {
        YIO_LOG_ERROR_EVENT("AlarmCapability.AlarmStartParsingFailed", "payload['alarm'] parsing failed");
        return;
    }

    switch (state_) {
        case State::SOUND_FILE_ALARM:
            YIO_LOG_ERROR_EVENT("AlarmCapability.ApprovedWhile_SoundFileAlarm", "Stop playing sound_file_alarm: " << alarmId_);
            stopSoundFileAlarm();
            break;
        case State::WAIT_FOR_DIRECTIVES:
            YIO_LOG_ERROR_EVENT("AlarmCapability.ApprovedWhile_WaitForDirectives", "Cancel old requestId: " << alarmId_);
            cancelPendingRequest();
            stopMediaAlarm(true);
            break;
        case State::WAIT_FOR_MEDIA_CONFIRM:
        case State::MEDIA_ALARM:
            YIO_LOG_ERROR_EVENT("AlarmCapability.ApprovedWhile_MediaAlarm", "Stop playing media_alarm: " << alarmId_);
            stopMediaAlarm(true);
            break;
        case State::IDLE:
            break;
    }

    YIO_LOG_INFO("Handle ALARM_START directive, alarm: " << alarmToString(alarm));
    if (alarm.alarm_type() == proto::Alarm::MEDIA_ALARM) {
        if (!mediaAlarmSettings_.isNull() && mediaAlarmSettings_.isMember("server_action")) {
            startMediaAlarm(alarm, mediaAlarmSettings_["server_action"]);
            return;
        }
        alarm.set_alarm_type(proto::Alarm::ALARM);
    }
    startSoundFileAlarm(alarm);
}

void AlarmCapability::handleAlarmStopDirective(const Json::Value& payload) {
    if (isAlarmPlaying()) {
        const bool stopMedia = tryGetBool(payload, "stop_media", true);
        YIO_LOG_INFO("Stop playing alarm. Stop media: " << stopMedia);

        auto message = ipc::buildMessage([stopMedia](auto& msg) {
            msg.mutable_alarm_message()->mutable_stop_alarm()->set_stop_media(stopMedia);
        });
        alarmsConnector_->sendMessage(message);

        return;
    }

    if (!hasRemainingMedia_) {
        YIO_LOG_INFO("No remaining media found");
        return;
    }

    YIO_LOG_INFO("Stop remaining media");

    auto message = ipc::buildMessage([](auto& msg) {
        msg.mutable_alarm_message()->mutable_alarm_stop_directive();
    });
    alarmsConnector_->sendMessage(message);

    stopMediaPlayer();

    hasRemainingMedia_ = false;
}

void AlarmCapability::handleAlarmsUpdate(const Json::Value& payload)
{
    YIO_LOG_INFO("Alarms update: " + jsonToString(payload, true));
    const auto iCalString = getString(payload, "state");

    auto message = ipc::buildMessage([&iCalString](auto& msg) {
        msg.mutable_alarms_state()->set_icalendar_state(TString(iCalString));
    });
    alarmsConnector_->sendMessage(message);
}

void AlarmCapability::handleAlarmSetSound(const Json::Value& payload)
{
    const std::string payloadString = jsonToString(payload);
    YIO_LOG_INFO("Alarms Set Sound: " + payloadString);

    auto message = ipc::buildMessage([&payloadString](auto& msg) {
        msg.mutable_alarm_set_sound()->set_payload(TString(payloadString));
    });
    alarmsConnector_->sendMessage(message);
}

void AlarmCapability::handleAlarmResetSound()
{
    YIO_LOG_INFO("Alarms Reset Sound");

    auto message = ipc::buildMessage([](auto& msg) {
        msg.mutable_alarm_reset_sound();
    });
    alarmsConnector_->sendMessage(message);
}

void AlarmCapability::handleAlarmSetMaxLevel(const Json::Value& payload)
{
    int newLevel = getInt(payload, "new_level");
    YIO_LOG_INFO("Alarm Set Max Level " << newLevel);

    auto message = ipc::buildMessage([newLevel](auto& msg) {
        msg.mutable_alarms_settings()->set_max_volume_level(newLevel);
    });
    alarmsConnector_->sendMessage(message);
}

void AlarmCapability::handleAlarmdMessage(const ipc::SharedMessage& message)
{
    if (message->has_timers_state()) {
        YIO_LOG_DEBUG("has_timers_state");
        timersState_ = message->timers_state();
        deviceStateCapability_->setTimersState(YandexIO::convertTimers(timersState_, playingAlarm_));
    }

    if (message->has_alarms_state() && message->alarms_state().has_icalendar_state()) {
        YIO_LOG_DEBUG("has_icalendar_state");
        deviceStateCapability_->setICalendar(message->alarms_state().icalendar_state());
        alarmState_.SetICalendar(message->alarms_state().icalendar_state());
        const bool isCurrentlyPlaying = playingAlarm_.has_value() && (playingAlarm_->alarm_type() == proto::Alarm::ALARM || playingAlarm_->alarm_type() == proto::Alarm::MEDIA_ALARM);
        alarmState_.SetCurrentlyPlaying(isCurrentlyPlaying);
        alarmState_.ClearSoundAlarmSetting();
        if (message->alarms_state().has_media_alarm_setting()) {
            try {
                mediaAlarmSettings_ = parseJson(message->alarms_state().media_alarm_setting());
                auto alarmSettings = convertJsonToProtobuf<NAlice::TDeviceState::TAlarmState::TSoundAlarmSetting>(jsonToString(mediaAlarmSettings_["sound_alarm_setting"])).value();
                alarmState_.MutableSoundAlarmSetting()->Swap(&alarmSettings);
            } catch (const std::exception& ex) {
                YIO_LOG_ERROR_EVENT("AlarmCapability.BadJson.MediaAlarmSetting", ex.what());
            }
        }
        alarmState_.SetMaxSoundLevel(message->alarms_state().alarms_settings().max_volume_level());
        deviceStateCapability_->setAlarmState(alarmState_);
    }

    if (message->has_alarm_event()) {
        handleAlarmEvent(message->alarm_event());
    }
}

void AlarmCapability::handleAlarmEvent(const proto::AlarmEvent& event)
{
    if (event.has_alarm_fired()) {
        YIO_LOG_INFO("handleAlarmEvent alarm_fired, " << alarmToString(event.alarm_fired()));

        cancelPendingRequest();
        hasRemainingMedia_ = false;
        notifyAlarmEnqueued(event.alarm_fired());
    } else if (event.has_alarm_approved()) {
        YIO_LOG_INFO("handleAlarmEvent alarm_approved, enqueue directive ALARM_START. alarm: " << alarmToString(event.alarm_approved()));
        Json::Value payload;
        payload["alarm"] = event.alarm_approved().SerializeAsString();
        payload["is_media_alarm"] = event.alarm_approved().alarm_type() == proto::Alarm::MEDIA_ALARM;

        Directive::Data data(Directives::ALARM_START, "local_action", std::move(payload));
        data.requestId = event.alarm_approved().id();

        if (auto dp = directiveProcessor_.lock()) {
            dp->addDirectives({std::make_shared<Directive>(std::move(data))});
        }
    } else if (event.has_alarm_stopped()) {
        const auto& alarm = event.alarm_stopped().alarm();
        const bool stopMedia = event.alarm_stopped().stop_media();
        YIO_LOG_INFO("handleAlarmEvent alarm_stopped, " << alarmToString(alarm) << ", stop_media " << stopMedia);
        playingAlarm_.reset();
        alarmState_.SetCurrentlyPlaying(false);
        notifyAlarmStopped();

        if (state_ == State::SOUND_FILE_ALARM) {
            stopSoundFileAlarm();
        } else {
            stopMediaAlarm(stopMedia);
        }
    }
}

void AlarmCapability::startSoundFileAlarm(const proto::Alarm& alarm)
{
    YIO_LOG_DEBUG("startSoundFileAlarm");

    wasAlarmStartedConfirmed_ = true;
    onAlarmStarted(alarm);

    state_ = State::SOUND_FILE_ALARM;
    alarmId_ = alarm.id();
    alarmChannel_ = proto::DIALOG_CHANNEL;

    if (const auto filePlayerCapability = filePlayerCapability_.lock()) {
        YandexIO::IFilePlayerCapability::PlayParams params;
        params.playLooped = true;
        params.parentRequestId = alarmId_;

        filePlayerCapability->playSoundFile("timer.wav", proto::AudioChannel::DIALOG_CHANNEL, params);
    }
}

void AlarmCapability::stopSoundFileAlarm()
{
    if (const auto filePlayerCapability = filePlayerCapability_.lock()) {
        filePlayerCapability->stopSoundFile("timer.wav");
    }

    toIdleState();
}

void AlarmCapability::startMediaAlarm(const proto::Alarm& alarm, Json::Value serverActionJson)
{
    bool jsonIsValid = !serverActionJson.isNull();

    if (jsonIsValid) {
        try {
            // "alarm_id" field is required for backend
            if (serverActionJson["name"].asString() == "bass_action") {
                serverActionJson["payload"]["data"]["alarm_id"] = alarm.id();
                YIO_LOG_INFO("Requesting server action for alarm_id: " << alarm.id());
            } else {
                serverActionJson["payload"]["typed_semantic_frame"]["music_play_semantic_frame"]["alarm_id"]["string_value"] = alarm.id();
                YIO_LOG_INFO("Requesting semantic frame for alarm_id: " << alarm.id());
            }
        } catch (const Json::LogicError& e) {
            jsonIsValid = false;
        }
    }

    if (!jsonIsValid) {
        YIO_LOG_ERROR_EVENT("AlarmCapability.FailedSerializeJsonServerActionPayload", "serverActionPayload is not correct");

        auto alarmCopy = alarm;
        alarmCopy.set_alarm_type(proto::Alarm::ALARM);
        startSoundFileAlarm(alarmCopy);
        return;
    }

    state_ = State::WAIT_FOR_DIRECTIVES;
    alarmId_ = alarm.id();

    if (auto aliceCapability = aliceCapability_.lock()) {
        auto request = VinsRequest::createEventRequest(
            serverActionJson,
            VinsRequest::createSoftwareDirectiveEventSource(),
            alarmId_);
        request->setIsSilent(true);
        request->setIsIgnoreCriticalUpdate(true);

        aliceCapability->startRequest(std::move(request), nullptr);
    }

    setMediaAlarmTimeoutHandler(alarmId_);
}

void AlarmCapability::setMediaAlarmTimeoutHandler(const std::string& alarmId)
{
    worker_->addDelayed([this, alarmId]() {
        if (alarmId_ != alarmId) {
            return;
        }

        bool mediaAlarmWasNotConfirmed = false;
        if (state_ == State::WAIT_FOR_DIRECTIVES) {
            cancelPendingRequest();
            mediaAlarmWasNotConfirmed = true;
        } else if (state_ == State::WAIT_FOR_MEDIA_CONFIRM) {
            stopMediaPlayer();
            mediaAlarmWasNotConfirmed = true;
        }

        if (mediaAlarmWasNotConfirmed) {
            fallbackToSoundFileAlarm();
        }
    }, MEDIA_ALARM_TIMEOUT);
}

void AlarmCapability::fallbackToSoundFileAlarm() {
    YIO_LOG_WARN("Fallback to sound file alarm. alarm_id: " << alarmId_);
    Json::Value payload;
    payload["alarm_id"] = alarmId_;
    device_->telemetry()->reportEvent("mediaAlarmNotConfirmed", jsonToString(payload));

    proto::Alarm alarm;
    alarm.set_id(TString(alarmId_));
    alarm.set_alarm_type(proto::Alarm::ALARM);
    startSoundFileAlarm(alarm);
}

void AlarmCapability::stopMediaAlarm(bool stopMedia)
{
    YIO_LOG_INFO("stopMediaAlarm, stopMedia = " << stopMedia);
    switch (state_) {
        case State::WAIT_FOR_MEDIA_CONFIRM:
        case State::MEDIA_ALARM:
            toIdleState();
            if (stopMedia) {
                stopMediaPlayer();
            } else {
                hasRemainingMedia_ = true;
            }
            break;
        case State::IDLE:
        case State::SOUND_FILE_ALARM:
        case State::WAIT_FOR_DIRECTIVES:
            YIO_LOG_ERROR_EVENT("AlarmCapability.UnexpectedState", "stopMediaAlarm but state is " << stateToString(state_));
            break;
    }
}

void AlarmCapability::stopMediaPlayer()
{
    YIO_LOG_DEBUG("stopMediaPlayer")
    if (const auto playbackCapability = playbackCapability_.lock()) {
        playbackCapability->pause();
    }
}

void AlarmCapability::cancelPendingRequest()
{
    YIO_LOG_DEBUG("cancelPendingRequest");
    if (const auto aliceCapability = aliceCapability_.lock()) {
        aliceCapability->cancelDialog();
    }
}

std::string AlarmCapability::alarmToString(const proto::Alarm& alarm)
{
    std::stringstream ss;
    ss << "alarm_id: " << alarm.id() << " type: ";
    switch (alarm.alarm_type()) {
        case proto::Alarm::TIMER:
            ss << "TIMER";
            break;
        case proto::Alarm::ALARM:
            ss << "ALARM";
            break;
        case proto::Alarm::MEDIA_ALARM:
            ss << "MEDIA_ALARM";
            break;
        default:
            ss << "unexpected " << (int)alarm.alarm_type();
    }
    return ss.str();
}

const std::string& AlarmCapability::getPreprocessorName() const {
    static const std::string s_name = "AlarmCapabilityPreprocessor";
    return s_name;
}

void AlarmCapability::preprocessDirectives(std::list<std::shared_ptr<Directive>>& directives)
{
    const bool alarmIsPlaying = isAlarmPlaying();

    bool alarmShouldBeStopped = false;
    bool hasStopAlarm = false;
    bool stopRemainingMedia = false;

    for (const auto& directive : directives) {
        if (directive->is(Directives::ALARM_STOP)) {
            alarmShouldBeStopped = true;
            hasStopAlarm = true;

            /// alarm_stop from server should always stop media
            Json::Value payload = directive->getData().payload;
            payload["stop_media"] = true;
            directive->setPayload(std::move(payload));

        } else if (alarmIsPlaying && directive->getData().channel.has_value() && directive->getData().channel.value() >= alarmChannel_ && !isDirectiveRelatedToPlayingAlarm(directive)) {
            alarmShouldBeStopped = true;

        } else if (state_ == State::WAIT_FOR_DIRECTIVES && directive->getRequestId() == alarmId_ && directive->is(Directives::RADIO_PLAY)) {
            /// This is required for java radio player only
            /// Otherwise java player will not restart and will confirm wrong vinsRequestId
            Json::Value payload = directive->getData().payload;
            payload["force_restart_player"] = true;
            directive->setPayload(std::move(payload));
        } else if ((alarmIsPlaying || hasRemainingMedia_) && directive->is(Directives::MORDOVIA_SHOW)) {
            /// This should be fixed by server MEGAMIND-2967
            YIO_LOG_INFO("Stop remaining media by MORDOVIA_SHOW");
            stopRemainingMedia = true;
            alarmShouldBeStopped = true;
        }
    }

    if (alarmShouldBeStopped && !hasStopAlarm) {
        Json::Value payload;
        payload["stop_media"] = stopRemainingMedia || (alarmChannel_ != proto::CONTENT_CHANNEL);

        directives.push_front(Directive::createLocalAction(Directives::ALARM_STOP, payload));
    }
}

void AlarmCapability::onSequenceStateChanged()
{
    // do nothing
}

void AlarmCapability::onDialogChannelIsIdle()
{
    // do nothing
}

void AlarmCapability::onDirectiveHandled(const std::shared_ptr<Directive>& directive)
{
    if (state_ != State::WAIT_FOR_DIRECTIVES) {
        return;
    }

    if (!directive->getData().channel.has_value() && !isLegacyPlayerAlarmDirective(directive)) {
        return;
    }

    if (!isDirectiveRelatedToPlayingAlarm(directive)) {
        return;
    }

    YIO_LOG_INFO("onDirectiveHandled related directive, wait for media confirm. alarm_id: " << alarmId_ << ", directive: " << directive->format());
    state_ = State::WAIT_FOR_MEDIA_CONFIRM;
}

void AlarmCapability::onDirectiveStarted(const std::shared_ptr<Directive>& directive) {
    if (state_ != State::WAIT_FOR_DIRECTIVES && state_ != State::WAIT_FOR_MEDIA_CONFIRM) {
        return;
    }

    if (!directive->getData().channel.has_value() && !isLegacyPlayerAlarmDirective(directive)) {
        return;
    }

    if (!isDirectiveRelatedToPlayingAlarm(directive)) {
        return;
    }

    if (!wasAlarmStartedConfirmed_) {
        YIO_LOG_INFO("Media alarm was confirmed for first time. alarm_id: " << alarmId_ << ", directive: " << directive->format());
        wasAlarmStartedConfirmed_ = true;
        notifyMediaAlarmConfirmed();
    } else {
        YIO_LOG_INFO("Media alarm was confirmed again. alarm_id: " << alarmId_ << ", directive: " << directive->format());
    }
    alarmChannel_ = directive->getData().channel.value_or(proto::CONTENT_CHANNEL);
    state_ = State::MEDIA_ALARM;
}

void AlarmCapability::onDirectiveCompleted(const std::shared_ptr<Directive>& directive, IDirectiveProcessorListener::Result result) {
    if (state_ != State::WAIT_FOR_DIRECTIVES && state_ != State::WAIT_FOR_MEDIA_CONFIRM && state_ != State::MEDIA_ALARM) {
        return;
    }

    if (!directive->getData().channel.has_value() && !isLegacyPlayerAlarmDirective(directive)) {
        return;
    }

    if (!isDirectiveRelatedToPlayingAlarm(directive)) {
        return;
    }

    if (result != IDirectiveProcessorListener::Result::SUCCESS) {
        YIO_LOG_WARN("Directive failed or canceled, fallback to basic alarm. alarm_id: " << alarmId_ << ", directive: " << directive->format());
        fallbackToSoundFileAlarm();
        return;
    }

    YIO_LOG_INFO("Directive completed, wait for next directive. alarm_id: " << alarmId_ << ", directive: " << directive->format());
    state_ = State::WAIT_FOR_DIRECTIVES;
    setMediaAlarmTimeoutHandler(alarmId_);
}

void AlarmCapability::notifyMediaAlarmConfirmed()
{
    Json::Value payload;
    payload["alarm_id"] = alarmId_;
    device_->telemetry()->reportEvent("mediaAlarmConfirmed", jsonToString(payload));

    proto::Alarm alarm;
    alarm.set_id(TString(alarmId_));
    alarm.set_alarm_type(proto::Alarm::MEDIA_ALARM);
    onAlarmStarted(alarm);
}

void AlarmCapability::onAlarmStarted(const proto::Alarm& alarm) {
    playingAlarm_ = alarm;
    alarmState_.SetCurrentlyPlaying(playingAlarm_->alarm_type() == proto::Alarm::ALARM || playingAlarm_->alarm_type() == proto::Alarm::MEDIA_ALARM);
    notifyAlarmStarted();

    auto message = ipc::buildMessage([&alarm](auto& msg) {
        msg.mutable_alarm_event()->mutable_alarm_confirmed()->CopyFrom(alarm);
    });
    alarmsConnector_->sendMessage(message);
}

void AlarmCapability::notifyAlarmStarted() {
    deviceStateCapability_->setAlarmState(alarmState_);
    deviceStateCapability_->setTimersState(YandexIO::convertTimers(timersState_, playingAlarm_));
    for (const auto& wlistener : wlisteners_) {
        if (auto listener = wlistener.lock()) {
            listener->onAlarmStarted();
        }
    }
}

void AlarmCapability::notifyAlarmStopped() {
    deviceStateCapability_->setAlarmState(alarmState_);
    deviceStateCapability_->setTimersState(YandexIO::convertTimers(timersState_, playingAlarm_));
    for (const auto& wlistener : wlisteners_) {
        if (auto listener = wlistener.lock()) {
            listener->onAlarmStopped();
        }
    }
}

void AlarmCapability::notifyAlarmEnqueued(const quasar::proto::Alarm& alarm) {
    for (const auto& wlistener : wlisteners_) {
        if (auto listener = wlistener.lock()) {
            listener->onAlarmEnqueued(alarm);
        }
    }
}

bool AlarmCapability::isAlarmPlaying() const {
    return wasAlarmStartedConfirmed_;
}

bool AlarmCapability::isDirectiveRelatedToPlayingAlarm(const std::shared_ptr<Directive>& directive)
{
    return directive->getRequestId() == alarmId_ || directive->getData().parentRequestId == alarmId_;
}

bool AlarmCapability::isLegacyPlayerAlarmDirective(const std::shared_ptr<Directive>& directive)
{
    return directive->is(Directives::MUSIC_PLAY) || directive->is(Directives::RADIO_PLAY);
}

void AlarmCapability::toIdleState()
{
    alarmId_.clear();
    state_ = State::IDLE;
    wasAlarmStartedConfirmed_ = false;
}

void AlarmCapability::handleRemotingMessage(const quasar::proto::Remoting& message,
                                            std::shared_ptr<YandexIO::IRemotingConnection> /*connection*/) {
    if (!message.has_alarm_capability_method() || !message.alarm_capability_method().has_method()) {
        return;
    }
    const auto method = message.alarm_capability_method().method();
    switch (method) {
        case quasar::proto::Remoting::AlarmCapabilityMethod::STOP_ALARM: {
            stopAlarm();
            break;
        }
        default: {
            break;
        }
    }
}

std::string AlarmCapability::stateToString(State state)
{
    switch (state) {
        case State::IDLE:
            return "IDLE";
        case State::SOUND_FILE_ALARM:
            return "SOUND_FILE_ALARM";
        case State::WAIT_FOR_DIRECTIVES:
            return "WAIT_FOR_DIRECTIVES";
        case State::WAIT_FOR_MEDIA_CONFIRM:
            return "WAIT_FOR_MEDIA_CONFIRM";
        case State::MEDIA_ALARM:
            return "MEDIA_ALARM";
    }
}
