#pragma once

#include <yandex_io/capabilities/alarm/interfaces/i_alarm_capability.h>

#include <yandex_io/services/aliced/capabilities/alice_capability/directives/alice_request_directive.h>
#include <yandex_io/services/aliced/directive_processor/interface/i_directive_preprocessor.h>
#include <yandex_io/services/aliced/directive_processor/interface/i_directive_processor.h>
#include <yandex_io/services/aliced/directive_processor/interface/i_directive_processor_listener.h>
#include <yandex_io/services/aliced/device_state/alice_device_state.h>

#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/libs/threading/i_callback_queue.h>
#include <yandex_io/capabilities/alice/interfaces/i_alice_capability.h>
#include <yandex_io/capabilities/file_player/interfaces/i_file_player_capability.h>
#include <yandex_io/capabilities/playback_control/interfaces/i_playback_control_capability.h>
#include <yandex_io/capabilities/device_state/interfaces/i_device_state_capability.h>
#include <yandex_io/sdk/interfaces/i_directive_handler.h>

#include <yandex_io/sdk/private/remoting/i_remote_object.h>
#include <yandex_io/sdk/private/remoting/i_remoting_registry.h>

#include <list>
#include <memory>

namespace quasar {
    class AlarmCapability
        : public YandexIO::IAlarmCapability,
          public YandexIO::IDirectivePreprocessor,
          public YandexIO::IDirectiveHandler,
          public YandexIO::IDirectiveProcessorListener,
          public YandexIO::IRemoteObject,
          public std::enable_shared_from_this<AlarmCapability> {
    public:
        AlarmCapability(
            std::shared_ptr<quasar::ICallbackQueue> worker,
            std::shared_ptr<YandexIO::IDevice> device,
            std::shared_ptr<ipc::IIpcFactory> ipcFactory,
            std::weak_ptr<YandexIO::IRemotingRegistry> remotingRegistry,
            YandexIO::IDirectiveProcessorWeakPtr directiveProcessor,
            std::weak_ptr<YandexIO::IAliceCapability> aliceCapability,
            std::weak_ptr<YandexIO::IFilePlayerCapability> filePlayerCapability,
            std::weak_ptr<YandexIO::IPlaybackControlCapability> playbackCapability,
            std::shared_ptr<YandexIO::IDeviceStateCapability> deviceStateCapability);

        ~AlarmCapability();

        void init();
        void handleAlarmdMessage(const ipc::SharedMessage& message);

        /// IAlarmCapability
        void addListener(std::weak_ptr<YandexIO::IAlarmCapabilityListener> listener) override;
        void removeListener(std::weak_ptr<YandexIO::IAlarmCapabilityListener> listener) override;
        void stopAlarm() override;

        /// IDirectivePreprocessor
        const std::string& getPreprocessorName() const override;
        void preprocessDirectives(std::list<std::shared_ptr<YandexIO::Directive>>& directives) override;

        /// IDirectiveHandler implementation
        const std::string& getHandlerName() const override;
        const std::set<std::string>& getSupportedDirectiveNames() const override;
        void handleDirective(const std::shared_ptr<YandexIO::Directive>& directive) override;
        void cancelDirective(const std::shared_ptr<YandexIO::Directive>& directive) override;
        void prefetchDirective(const std::shared_ptr<YandexIO::Directive>& directive) override;

        /// IDirectiveProcessorListener implementation
        void onSequenceStateChanged() override;
        void onDialogChannelIsIdle() override;
        void onDirectiveHandled(const std::shared_ptr<YandexIO::Directive>& directive) override;
        void onDirectiveStarted(const std::shared_ptr<YandexIO::Directive>& directive) override;
        void onDirectiveCompleted(const std::shared_ptr<YandexIO::Directive>& directive, YandexIO::IDirectiveProcessorListener::Result result) override;

        /// IRemoteObject implementation
        void handleRemotingMessage(const quasar::proto::Remoting& message,
                                   std::shared_ptr<YandexIO::IRemotingConnection> connection) override;

    private:
        struct AlarmDirective {
            std::string name;
            std::string payload;
        };

        void handleAlarmEvent(const proto::AlarmEvent& event);

        void startSoundFileAlarm(const proto::Alarm& alarm);
        void stopSoundFileAlarm();

        void startMediaAlarm(const proto::Alarm& alarm, Json::Value serverActionJson);
        void setMediaAlarmTimeoutHandler(const std::string& alarmId);
        void fallbackToSoundFileAlarm();

        void cancelPendingRequest();

        void stopMediaAlarm(bool stopMedia);
        void stopMediaPlayer();

        void handleCancelTimer(const Json::Value& payload);
        void handleSetTimer(const Json::Value& payload);
        void handlePauseTimer(const Json::Value& payload);
        void handleResumeTimer(const Json::Value& payload);
        void handleAlarmsUpdate(const Json::Value& payload);
        void handleAlarmSetSound(const Json::Value& payload);
        void handleAlarmResetSound();
        void handleAlarmSetMaxLevel(const Json::Value& payload);
        void handleAlarmStartDirective(const Json::Value& payload);
        void handleAlarmStopDirective(const Json::Value& payload);

        void notifyMediaAlarmConfirmed();

        void sendAddAlarmMessage(std::chrono::seconds fireDelay, std::vector<AlarmDirective> directives = {});

        void onAlarmStarted(const proto::Alarm& alarm);
        void notifyAlarmStarted();
        void notifyAlarmStopped();
        void notifyAlarmEnqueued(const quasar::proto::Alarm& alarm);

        bool isAlarmPlaying() const;
        bool isDirectiveRelatedToPlayingAlarm(const std::shared_ptr<YandexIO::Directive>& directive);

        void toIdleState();

        static std::string alarmToString(const proto::Alarm& alarm);

        static bool isLegacyPlayerAlarmDirective(const std::shared_ptr<YandexIO::Directive>& directive);

    private:
        enum class State {
            IDLE,
            SOUND_FILE_ALARM,
            WAIT_FOR_DIRECTIVES,
            WAIT_FOR_MEDIA_CONFIRM,
            MEDIA_ALARM
        };
        static std::string stateToString(State state);

    private:
        const std::shared_ptr<quasar::ICallbackQueue> worker_;
        const std::shared_ptr<YandexIO::IDevice> device_;
        const YandexIO::IDirectiveProcessorWeakPtr directiveProcessor_;

        State state_{State::IDLE};
        bool wasAlarmStartedConfirmed_{false};
        std::string alarmId_;
        proto::AudioChannel alarmChannel_{proto::DIALOG_CHANNEL};

        bool hasRemainingMedia_ = false;

        std::list<std::weak_ptr<YandexIO::IAlarmCapabilityListener>> wlisteners_;

        const std::shared_ptr<ipc::IConnector> alarmsConnector_;
        const std::weak_ptr<YandexIO::IAliceCapability> aliceCapability_;
        const std::weak_ptr<YandexIO::IFilePlayerCapability> filePlayerCapability_;
        const std::weak_ptr<YandexIO::IPlaybackControlCapability> playbackCapability_;
        const std::shared_ptr<YandexIO::IDeviceStateCapability> deviceStateCapability_;

        std::optional<proto::Alarm> playingAlarm_;
        proto::TimersState timersState_;
        Json::Value mediaAlarmSettings_;
        NAlice::TDeviceState::TAlarmState alarmState_;
    };

} // namespace quasar
