#include "alice_capability.h"

#include <yandex_io/capabilities/alice/alice_capability_proxy.h>
#include <yandex_io/capabilities/alice/alice_request_events_proxy.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/base/directives.h>
#include <yandex_io/libs/ete_metrics/ete_util.h>
#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/protobuf_utils/json.h>
#include <yandex_io/sdk/interfaces/directive.h>
#include <speechkit/SoundLogger.h>

#include <util/system/yassert.h>

YIO_DEFINE_LOG_MODULE("alice_capability");

using namespace NAlice;
using namespace YandexIO;
using namespace quasar;

static const std::string s_remoteRegistryName = "AliceCapability";

void AliceBlockers::block(const std::string& id, const std::optional<std::string>& errorSound) {
    blockers_[id] = errorSound;
}

void AliceBlockers::unblock(const std::string& id) {
    blockers_.erase(id);
}

bool AliceBlockers::isBlocked() {
    return !blockers_.empty();
}

std::optional<std::string> AliceBlockers::getErrorSound() const {
    // Return first error message, because they don't have priorities
    for (const auto& [_, errorSound] : blockers_) {
        if (errorSound) {
            return errorSound;
        }
    }

    return std::nullopt;
}

std::vector<std::string> AliceBlockers::getBlockerIds() const {
    std::vector<std::string> res;

    for (const auto& [source, _] : blockers_) {
        res.push_back(source);
    }

    return res;
}

std::string AliceBlockers::getBlockersText() const {
    size_t length = 0;
    for (const auto& [source, _] : blockers_) {
        length += source.length() + 2;
    }

    std::string res;
    res.reserve(length);
    for (const auto& [source, _] : blockers_) {
        res += source;
    }
    return res;
}

AliceCapability::AliceCapability(
    const AliceConfig& aliceConfig,
    AliceDeviceState& aliceDeviceState,
    ActivityTracker& activityTracker,
    std::weak_ptr<IDirectiveProcessor> directiveProcessor,
    DeviceContext& deviceContext,
    const std::shared_ptr<ITelemetry>& telemetry,
    std::weak_ptr<QuasarVoiceDialog> voiceService,
    std::shared_ptr<ipc::IServer> server,
    std::shared_ptr<quasar::ipc::IConnector> interfacedConnector,
    std::shared_ptr<VoiceStats> voiceStats,
    std::weak_ptr<IRemotingRegistry> remotingRegistry,
    std::shared_ptr<IStereoPairProvider> stereoPairProvider,
    std::weak_ptr<IFilePlayerCapability> filePlayerCapability)
    : IRemoteObject(std::move(remotingRegistry))
    , activityTracker_(activityTracker)
    , directiveProcessor_(std::move(directiveProcessor))
    , telemetry_(telemetry)
    , aliceDeviceState_(aliceDeviceState)
    , aliceConfig_(aliceConfig)
    , deviceContext_(deviceContext)
    , voiceService_(std::move(voiceService))
    , server_(std::move(server))
    , voiceStats_(std::move(voiceStats))
    , aliceStateMachine_(server_, std::move(interfacedConnector))
    , stereoPairProvider_(std::move(stereoPairProvider))
    , filePlayerCapability_(std::move(filePlayerCapability))
{
}

AliceCapability::~AliceCapability() {
    if (auto remotingRegistry = getRemotingRegistry().lock()) {
        remotingRegistry->removeRemoteObject(s_remoteRegistryName);
    }
}

void AliceCapability::init() {
    if (auto remotingRegistry = getRemotingRegistry().lock()) {
        remotingRegistry->addRemoteObject(s_remoteRegistryName, weak_from_this());
    }
}

const std::string& AliceCapability::getHandlerName() const {
    static const std::string s_name = "AliceCapability";
    return s_name;
}

const std::set<std::string>& AliceCapability::getSupportedDirectiveNames() const {
    static std::set<std::string> s_names = {
        Directives::LISTEN,
        Directives::START_MUSIC_RECOGNIZER,
        Directives::TTS_PLAY_PLACEHOLDER,
        Directives::ALICE_REQUEST,
        Directives::ALICE_RESPONSE,
        Directives::TYPE,
        Directives::TYPE_SILENT,
        Directives::SET_COOKIES,
        Directives::CLEAR_QUEUE,
    };

    return s_names;
}

void AliceCapability::handleDirective(const std::shared_ptr<Directive>& directive)
{
    try {
        if (directive->is(Directives::LISTEN) ||
            directive->is(Directives::START_MUSIC_RECOGNIZER)) {
            activityTracker_.addActivity(shared_from_this());

            startAliceRequestDirective(directive, false);
        } else if (directive->is(Directives::TTS_PLAY_PLACEHOLDER)) {
            playTtsDirective_ = std::dynamic_pointer_cast<PlayTtsDirective>(directive);
            Y_VERIFY(playTtsDirective_ != nullptr);
            activityTracker_.addActivity(shared_from_this());

            const auto& data = directive->getData();
            playTtsDirective_->getPlayer()->setListener(weak_from_this());
            YIO_LOG_DEBUG("play_tts channel=" << proto::AudioChannel_Name(getAudioChannel()));

            proto::AudioAnalyticsContext analyticsContext;
            analyticsContext.set_vins_request_id(TString(data.requestId));
            analyticsContext.set_vins_parent_request_id(TString(data.parentRequestId));
            analyticsContext.set_name(TString(data.analyticsContextName));

            playTtsDirective_->getPlayer()->play(analyticsContext);

            if (!directive->getData().requestId.empty()) {
                aliceStateMachine_.onSayingBegin();
            }
        } else if (directive->is(Directives::ALICE_RESPONSE)) {
            if (directive->getData().isPrefetched) {
                if (auto voiceService = voiceService_.lock()) {
                    voiceService->updatePrevReqId(directive->getRequestId());
                }
            }

            quasar::proto::VinsResponse vinsResponse;
            const auto payloadString = jsonToString(directive->getData().payload);
            if (convertJsonToMessage(&vinsResponse, payloadString)) {
                aliceStateMachine_.onAliceResponse(vinsResponse);
            }
        } else if (directive->is(Directives::ALICE_REQUEST)) {
            startAliceRequestDirective(directive, false);
        } else if (directive->is(Directives::TYPE) || directive->is(Directives::TYPE_SILENT)) {
            const bool isSilent = directive->is(Directives::TYPE_SILENT);
            const auto text = tryGetString(directive->getData().payload, "text");

            if (text.empty()) {
                throw std::runtime_error("Empty text in 'type' directive");
            }

            auto request = VinsRequest::createEventRequest(VinsRequest::buildTextInputEvent(text), VinsRequest::createSoftwareDirectiveEventSource());
            request->setVoiceSession(!isSilent);
            if (isSilent) {
                request->setIsSilent(true);
            }

            auto aliceRequestDirective = std::make_shared<AliceRequestDirective>(std::move(request), nullptr, true);
            if (auto dp = directiveProcessor_.lock()) {
                dp->addDirectives({aliceRequestDirective});
            }
        } else if (directive->is(Directives::SET_COOKIES)) {
            const std::string megamindCookies = tryGetString(directive->getData().payload, "value", "");
            aliceDeviceState_.setMegamindCookie(megamindCookies);
        }
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("AliceCapability.FailedHandleDirective",
                            "Failed to execute directive " << directive->getData().name << ": " << e.what());
    }
}

void AliceCapability::startRequest(
    std::shared_ptr<VinsRequest> request,
    std::shared_ptr<IAliceRequestEvents> events)
{
    if (auto directiveProcessor = directiveProcessor_.lock()) {
        auto directive = std::make_shared<AliceRequestDirective>(std::move(request), std::move(events), true);
        switch (directive->getRequest()->getEventSource().type()) {
            case TSpeechKitRequestProto_TEventSource_EType_Voice:
            case TSpeechKitRequestProto_TEventSource_EType_Music: {
                directiveProcessor->addDirectives({directive});
                break;
            }
            default: {
                if (directive->getRequest()->getIsEnqueued()) {
                    directiveProcessor->enqueueDirective(std::move(directive));
                } else {
                    directiveProcessor->addDirectives({directive});
                }
                break;
            }
        }
    }
}

void AliceCapability::onRequestCompleted(
    const std::shared_ptr<VinsRequest>& request,
    VinsResponse response,
    const Json::Value& payload)
{
    if (auto directiveProcessor = directiveProcessor_.lock()) {
        if (auto aliceRequestDirective = findAliceRequestDirective(request)) {
            if (auto events = aliceRequestDirective->getEvents()) {
                events->onAliceRequestCompleted(request, payload);
            }

            completeRequestDirective(aliceRequestDirective);
            if (request->isPrefetch()) {
                directiveProcessor->onPrefetchDirectiveCompleted(aliceRequestDirective, response.directives);
            } else {
                directiveProcessor->onHandleDirectiveCompleted(aliceRequestDirective, response.directives);
            }
        }
    }
}

void AliceCapability::handleAudioClientMessage(const ipc::SharedMessage& sharedMessage)
{
    const auto& message = *sharedMessage;
    if (playTtsDirective_ != nullptr && message.has_audio_client_event()) {
        playTtsDirective_->getPlayer()->handleAudioClientEvent(message.audio_client_event());
    }
}

void AliceCapability::onRequestError(
    const std::shared_ptr<VinsRequest>& request,
    const std::string& errorCode,
    const std::string& errorText)
{
    onRequestError(request, errorCode, errorText, false);
}

void AliceCapability::onRequestError(
    const std::shared_ptr<VinsRequest>& request,
    const SpeechKit::Error& error)
{
    std::string errorCode = "SpeechKit::Error#" + std::to_string((int)error.getCode());
    std::string errorText = "Произошла какая-то ошибка, спросите попозже пожалуйста";

    onRequestError(request, errorCode, errorText, error.getCode() == ::SpeechKit::Error::ErrorNoSpeechDetected);
}

void AliceCapability::handleAudioClientConnectionStatus(bool connected) {
    if (playTtsDirective_) {
        playTtsDirective_->getPlayer()->handleAudioClientConnectionStatus(connected);
    }
}

void AliceCapability::onRequestError(
    const std::shared_ptr<VinsRequest>& request,
    const std::string& errorCode,
    const std::string& errorText,
    bool success)
{
    if (auto directive = findAliceRequestDirective(request)) {
        if (auto events = directive->getEvents()) {
            events->onAliceRequestError(request, errorCode, errorText);
        }

        completeRequestDirective(directive);
        if (auto directiveProcessor = directiveProcessor_.lock()) {
            directiveProcessor->onHandleDirectiveCompleted(directive, success);
        }
    }
}

std::shared_ptr<VinsRequest> AliceCapability::findAliceRequest(const std::string& requestId) const {
    for (const auto& directive : aliceRequestDirectives_) {
        if (directive->getRequest()->getId() == requestId) {
            return directive->getRequest();
        }
    }

    return nullptr;
}

std::shared_ptr<VinsRequest> AliceCapability::findNonParallelAliceRequest() const {
    for (const auto& directive : aliceRequestDirectives_) {
        if (!directive->getRequest()->getIsParallel()) {
            return directive->getRequest();
        }
    }

    return nullptr;
}

std::shared_ptr<AliceRequestDirective> AliceCapability::findNonParallelAliceDirective() const {
    for (const auto& directive : aliceRequestDirectives_) {
        if (!directive->getRequest()->getIsParallel()) {
            return directive;
        }
    }

    return nullptr;
}

AliceBlockers& AliceCapability::getAliceBlockers()
{
    return aliceBlockers_;
}

void AliceCapability::completeTtsDirective()
{
    if (playTtsDirective_ != nullptr) {
        if (playTtsDirective_->isReminder()) {
            auto message = ipc::buildMessage([&](auto& msg) {
                msg.mutable_reminder_message()->mutable_reminder_stopped();
            });
            server_->sendToAll(message);
        }

        activityTracker_.removeActivity(shared_from_this());
        playTtsDirective_.reset();
    }
}

void AliceCapability::completeRequestDirective(const std::shared_ptr<Directive>& directive)
{
    directive->setPrefetchInProgress(false);

    if (auto aliceRequestDirective = popAliceRequestDirective(directive)) {
        switch (aliceRequestDirective->getRequest()->getEventSource().type()) {
            case TSpeechKitRequestProto_TEventSource_EType_Voice:
            case TSpeechKitRequestProto_TEventSource_EType_Music: {
                activityTracker_.removeActivity(shared_from_this());
                break;
            }
            default: {
                break;
            }
        }
    }
}

void AliceCapability::cancelDirective(const std::shared_ptr<Directive>& directive)
{
    Y_VERIFY(directive != nullptr);

    if (playTtsDirective_ == directive) {
        playTtsDirective_->getPlayer()->cancel();
        completeTtsDirective();
    } else {
        if (auto aliceRequestDirective = findAliceRequestDirective(directive)) {
            if (auto voiceService = voiceService_.lock()) {
                voiceService->cancel(true);
            }

            if (auto events = aliceRequestDirective->getEvents()) {
                events->onAliceRequestError(aliceRequestDirective->getRequest(), "RequestInterrupted", "RequestInterrupted");
            }
        }

        completeRequestDirective(directive);
    }
}

void AliceCapability::prefetchDirective(const std::shared_ptr<Directive>& directive)
{
    if (directive->is(Directives::ALICE_REQUEST)) {
        startAliceRequestDirective(directive, true);
    }
}

void AliceCapability::onPlayingBegin(std::shared_ptr<VinsRequest> request)
{
    if (playTtsDirective_ != nullptr) {
        if (playTtsDirective_->isReminder()) {
            auto message = ipc::buildMessage([&](auto& msg) {
                msg.mutable_reminder_message()->mutable_reminder_started();
            });
            server_->sendToAll(message);
        }

        if (auto directiveProcessor = directiveProcessor_.lock()) {
            directiveProcessor->onHandleDirectiveStarted(playTtsDirective_);
        }
        if (auto voiceService = voiceService_.lock()) {
            const auto messageId = request == nullptr ? "" : request->getMessageId();
            voiceService->onTtsStarted(messageId);
            voiceService->startInterruptionSpotter();
        }
    }
}

void AliceCapability::onPlayingEnd(std::shared_ptr<VinsRequest> request)
{
    if (playTtsDirective_ != nullptr) {
        if (auto voiceService = voiceService_.lock()) {
            const auto messageId = request == nullptr ? "" : request->getMessageId();
            voiceService->onTtsCompleted(messageId);
        }
        auto ttsDirective = playTtsDirective_;
        completeTtsDirective();
        if (auto directiveProcessor = directiveProcessor_.lock()) {
            directiveProcessor->onHandleDirectiveCompleted(ttsDirective, true);
        }
        aliceStateMachine_.onTtsCompleted();
    }
}

void AliceCapability::onPlayingError(std::shared_ptr<VinsRequest> request, const SpeechKit::Error& error)
{
    if (playTtsDirective_ != nullptr) {
        if (auto voiceService = voiceService_.lock()) {
            const auto messageId = request == nullptr ? "" : request->getMessageId();
            voiceService->onTtsError(error, messageId);
        }
        auto ttsDirective = playTtsDirective_;
        completeTtsDirective();
        if (auto directiveProcessor = directiveProcessor_.lock()) {
            directiveProcessor->onHandleDirectiveCompleted(ttsDirective, false);
        }
    }
}

std::string AliceCapability::activityName() const {
    return "AliceCapability";
}

void AliceCapability::setBackground()
{
    // do nothing
}

void AliceCapability::setForeground()
{
    // do nothing
}

proto::AudioChannel AliceCapability::getAudioChannel() const {
    if (playTtsDirective_ != nullptr) {
        return playTtsDirective_->getData().channel.value_or(proto::DIALOG_CHANNEL);
    }

    return proto::DIALOG_CHANNEL;
}

bool AliceCapability::isLocal() const {
    return true;
}

void AliceCapability::handleRemotingMessage(
    const quasar::proto::Remoting& message,
    std::shared_ptr<YandexIO::IRemotingConnection> connection)
{
    Y_UNUSED(connection);

    if (message.has_alice_capability_method()) {
        const auto& method = message.alice_capability_method();

        if (method.method() == quasar::proto::Remoting::AliceCapabilityMethod::START_REQUEST) {
            try {
                auto request = VinsRequest::createVinsRequestFromProtobuf(method.vins_request());
                std::shared_ptr<IAliceRequestEvents> events;
                if (method.has_events()) {
                    events = std::make_shared<AliceRequestEventsProxy>(method.vins_request().id(), connection);
                }

                startRequest(std::move(request), std::move(events));
            } catch (const std::exception& e) {
                YIO_LOG_ERROR_EVENT("AliceCapability.FailedToStartRequestFromRemote", e.what());
            }
        } else if (method.method() == quasar::proto::Remoting::AliceCapabilityMethod::CANCEL_DIALOG) {
            cancelDialog();
        } else if (method.method() == quasar::proto::Remoting::AliceCapabilityMethod::CANCEL_DIALOG_AND_CLEAR_QUEUE) {
            cancelDialogAndClearQueue();
        } else if (method.method() == quasar::proto::Remoting::AliceCapabilityMethod::START_CONVERSATION) {
            startConversation(method.vins_request().event_source());
        } else if (method.method() == quasar::proto::Remoting::AliceCapabilityMethod::STOP_CONVERSATION) {
            stopConversation();
        } else if (method.method() == quasar::proto::Remoting::AliceCapabilityMethod::TOGGLE_CONVERSATION) {
            toggleConversation(method.vins_request().event_source());
        } else if (method.method() == quasar::proto::Remoting::AliceCapabilityMethod::FINISH_CONVERSATION_VOICE_INPUT) {
            finishConversationVoiceInput();
        }
    }
}

void AliceCapability::startAliceRequest(
    const std::shared_ptr<Directive>& directive,
    const std::shared_ptr<VinsRequest>& request,
    bool isPrefetch)
{
    auto eventJson = makeETEEvent(directive->getRequestId());
    directive->setPrefetchInProgress(isPrefetch);
    if (directive->isGetNext() && !isPrefetch) {
        request->setIsGetNext(true);
        telemetry_->reportEvent("getNextRequest", jsonToString(eventJson));
    } else {
        telemetry_->reportEvent("startRequest", jsonToString(eventJson));
    }

    request->setIsPrefetch(isPrefetch);
    request->setEteShouldBindToPrevRequest(true);

    if (auto directiveProcessor = directiveProcessor_.lock()) {
        directiveProcessor->onHandleDirectiveStarted(directive);
    }

    startRequestImpl(request);
}

void AliceCapability::startRequestImpl(std::shared_ptr<VinsRequest> request)
{
    const auto header = vinsRequestHeader(request);
    const auto payloadJson = vinsRequestPayload(request);
    const std::string payload = jsonToString(payloadJson);

    if (auto voiceService = voiceService_.lock()) {
        switch (request->getEventSource().type()) {
            case TSpeechKitRequestProto_TEventSource_EType_Voice: {
                if (auto logger = SpeechKit::SoundLogger::getInstance()) {
                    logger->setExtraPayload(jsonToString(aliceDeviceState_.buildSoundLogExtra(aliceConfig_)));
                }

                voiceService->startVoiceInput(header, payload, aliceConfig_.getJingle());
                break;
            }
            case TSpeechKitRequestProto_TEventSource_EType_Music: {
                voiceService->startMusicInput(header, payload, aliceConfig_.getJingle());
                break;
            }
            case TSpeechKitRequestProto_TEventSource_EType_Text: {
                if (request->getIgnoreAnswer()) {
                    voiceService->sendEvent(header, payload);
                } else {
                    voiceService->startTextInput(header, payload);
                }
                break;
            }
            case TSpeechKitRequestProto_TEventSource_EType_VoiceprintMatch: {
                Y_VERIFY(request->getIgnoreAnswer());
                voiceService->sendEvent(header, payload);
                break;
            }
            default: {
                YIO_LOG_ERROR_EVENT("AliceCapability.FailedToStartRequest", "Unsupported eventSource type: " << request->getEventSource().type());
                return;
            }
        }
    }

    if (request->isVoiceInput()) {
        const bool isMusic = request->getEventSource().type() == TSpeechKitRequestProto_TEventSource_EType_Music;
        aliceStateMachine_.onRecognitionBegin(request->getId(), isMusic);
    }
}

void AliceCapability::startAliceRequestDirective(const std::shared_ptr<Directive>& directive, bool isPrefetch)
{
    auto aliceRequestDirective = std::dynamic_pointer_cast<AliceRequestDirective>(directive);
    Y_VERIFY(aliceRequestDirective != nullptr);

    auto request = aliceRequestDirective->getRequest();
    request->setEnqueued(request->getIsEnqueued() && !request->getIsParallel());
    if (request->isServerAction()) {
        request->setIsIgnoreCriticalUpdate(true);
    }
    if (!request->getIgnoreAnswer()) {
        aliceRequestDirectives_.push_back(aliceRequestDirective);
    }

    if (request->getEventSource().event() == TSpeechKitRequestProto_TEventSource_EEvent_Spotter && aliceDeviceState_.getStereoPairState()->isFollower()) {
        YIO_LOG_INFO("Spotting is blocked due stereo pair follower mode");
        onRequestError(request, "SpottingIsBlocked", "Stereo pair follower mode");
        return;
    }

    const auto state = getDialogProcessorState(request);
    if (!state.isReady) {
        YIO_LOG_WARN(state.message);

        std::shared_ptr<Directive> errorPlayDirective;

        onRequestError(request, "DeviceNotReady", state.message);

        if (request->isUserInitiatedAction() && state.errorSound) {
            if (auto filePlayerCapability = filePlayerCapability_.lock()) {
                filePlayerCapability->playSoundFile(state.errorSound.value(), proto::DIALOG_CHANNEL);
            }
        }

        return;
    }

    startAliceRequest(aliceRequestDirective, request, isPrefetch);

    if (auto events = aliceRequestDirective->getEvents()) {
        events->onAliceRequestStarted(aliceRequestDirective->getRequest());
    }
}

std::shared_ptr<AliceRequestDirective> AliceCapability::findAliceRequestDirective(const std::shared_ptr<VinsRequest>& request) const {
    for (const auto& directive : aliceRequestDirectives_) {
        if (directive->getRequest() == request) {
            return directive;
        }
    }

    return nullptr;
}

std::shared_ptr<AliceRequestDirective> AliceCapability::findAliceRequestDirective(const std::shared_ptr<Directive>& directive) const {
    for (const auto& aliceRequestDirective : aliceRequestDirectives_) {
        if (aliceRequestDirective == directive) {
            return aliceRequestDirective;
        }
    }

    return nullptr;
}

std::shared_ptr<AliceRequestDirective> AliceCapability::popAliceRequestDirective(const std::shared_ptr<Directive>& directive)
{
    for (auto iter = aliceRequestDirectives_.begin(); iter != aliceRequestDirectives_.end(); ++iter) {
        auto aliceRequestDirective = *iter;
        if (aliceRequestDirective == directive) {
            aliceRequestDirectives_.erase(iter);
            return aliceRequestDirective;
        }
    }

    return nullptr;
}

AliceCapability::DialogProcessorState AliceCapability::getDialogProcessorState(
    const std::shared_ptr<VinsRequest>& request)
{
    if (getAliceBlockers().isBlocked()) {
        std::stringstream ss;
        ss << "Следующие сервисы блокируют Алису: ";
        const auto blockerIds = getAliceBlockers().getBlockerIds();
        printRangeInStream(ss, blockerIds.begin(), blockerIds.end());

        return DialogProcessorState{
            .isReady = false,
            .message = ss.str(),
            .errorSound = getAliceBlockers().getErrorSound()};
    }

    if (aliceConfig_.getRequireAuthorization() &&
        aliceDeviceState_.isConfiguringOrUpdating(request->isIgnoreCriticalUpdate())) {
        YIO_LOG_INFO("Configuring or updating");
        telemetry_->reportEvent("phraseSpottedDuringActivation");

        return DialogProcessorState{
            .isReady = false,
            .message = "Настройте станцию заново",
        };
    }

    if (aliceDeviceState_.getAuthFailed()) {
        if (!request->getIsSilent()) {
            deviceContext_.fireConversationError();
        }
        telemetry_->reportEvent("phraseSpotterAuthFailed");

        return DialogProcessorState{
            .isReady = false,
            .message = "Настройте устройство заново",
            .errorSound = "auth_failed.wav"};
    }

    return getBrickState(request);
}

AliceCapability::DialogProcessorState AliceCapability::getBrickState(const std::shared_ptr<VinsRequest>& request) const {
    if (request->getIsReminder()) {
        return DialogProcessorState{
            .isReady = true,
            .message = "",
            .isBrick = true};
    }

    switch (aliceDeviceState_.getBrickStatus()) {
        case proto::BrickStatus::BRICK:
        case proto::BrickStatus::BRICK_BY_TTL:
            return DialogProcessorState{
                .isReady = false,
                .message = "Продлите подписку, и устройство продолжит вас радовать. \n"
                           "Проверьте привязанную карту в яндекс паспорте. \n"
                           "Возможно на ней не хватает денег или она больше не действительна.",
                .isBrick = true,
                .errorSound = "brick.wav"};
        case proto::BrickStatus::NOT_BRICK:
        case proto::BrickStatus::UNKNOWN_BRICK_STATUS:
            return DialogProcessorState{
                .isReady = true,
                .message = "",
                .errorSound = "brick.wav"};
        default:
            return DialogProcessorState{
                .isReady = false,
                .message = "Произошла какая-то ошибка. Попробуйте еще раз."};
    }
}

void AliceCapability::cancelDialog()
{
    YIO_LOG_INFO("cancelDialog");

    if (auto directive = findNonParallelAliceDirective()) {
        const auto requestType = directive->getRequest()->getEventSource().type();
        if (requestType == TSpeechKitRequestProto_TEventSource_EType_Voice || requestType == TSpeechKitRequestProto_TEventSource_EType_Music) {
            if (auto voiceService = voiceService_.lock()) {
                voiceService->playCancelSound();
            }
        }

        cancelDirective(directive);
        if (auto directiveProcessor = directiveProcessor_.lock()) {
            directiveProcessor->onHandleDirectiveCompleted(directive, false);
        }

        return;
    }

    if (auto directive = playTtsDirective_) {
        if (auto voiceService = voiceService_.lock()) {
            voiceService->playCancelSound();
            voiceService->stopInterruptionSpotter();
        }

        cancelDirective(directive);
        if (auto directiveProcessor = directiveProcessor_.lock()) {
            directiveProcessor->onHandleDirectiveCompleted(directive, false);
        }

        return;
    }
}

void AliceCapability::cancelDialogAndClearQueue()
{
    YIO_LOG_INFO("cancelDialogAndClearQueue");

    cancelDialog();

    if (auto directiveProcessor = directiveProcessor_.lock()) {
        directiveProcessor->addDirectives(
            {YandexIO::Directive::createLocalAction(Directives::CLEAR_QUEUE, Json::Value())});
    }
}

void AliceCapability::startConversation(const VinsRequest::EventSource& eventSource)
{
    YIO_LOG_INFO("startConversation");

    if (aliceDeviceState_.getStereoPairState()->isFollower()) {
        YIO_LOG_INFO("Suppress start conversation: Stereo pair follower mode");
        if (stereoPairProvider_) {
            stereoPairProvider_->startConversationOnLeader();
        }
    } else {
        startVoiceInput(eventSource);
    }
}

void AliceCapability::stopConversation()
{
    YIO_LOG_INFO("stopConversation");

    if (aliceDeviceState_.getStereoPairState()->isFollower()) {
        if (stereoPairProvider_) {
            stereoPairProvider_->stopConversationOnLeader();
        }
    } else {
        if (hasInteraction()) {
            cancelDialog();
        }
    }
}

void AliceCapability::toggleConversation(const VinsRequest::EventSource& eventSource)
{
    YIO_LOG_INFO("toggleConversation");

    const bool isHasInteraction = hasInteraction();
    if (isHasInteraction) {
        cancelDialog();
    }

    if (aliceDeviceState_.getStereoPairState()->isFollower()) {
        YIO_LOG_INFO("Suppress toggle conversation: Stereo pair follower mode");
        if (stereoPairProvider_) {
            stereoPairProvider_->toggleConversationOnLeader();
        }
    } else {
        if (!isHasInteraction) {
            startVoiceInput(eventSource);
        }
    }
}

void AliceCapability::finishConversationVoiceInput()
{
    if (auto request = findNonParallelAliceRequest()) {
        YIO_LOG_INFO("finishConversationVoiceInput");
        if (auto voiceService = voiceService_.lock()) {
            voiceService->stopInput();
        }
    }

    if (aliceDeviceState_.getStereoPairState()->isFollower()) {
        if (stereoPairProvider_) {
            stereoPairProvider_->finishConversationOnLeader();
        }
    }
}

void AliceCapability::addListener(std::weak_ptr<IAliceCapabilityListener> listener)
{
    aliceStateMachine_.addListener(listener);
}

void AliceCapability::removeListener(std::weak_ptr<IAliceCapabilityListener> listener)
{
    aliceStateMachine_.removeListener(listener);
}

const quasar::proto::AliceState& AliceCapability::getAliceState() const {
    return aliceStateMachine_.getState();
}

void AliceCapability::onExternalAliceState(quasar::proto::AliceState state)
{
    aliceStateMachine_.onAliceStateOverride(std::move(state));
}

void AliceCapability::onHasStartupInfo(bool hasAllStartupInfo)
{
    aliceStateMachine_.onHasStartupInfo(hasAllStartupInfo);
}

void AliceCapability::onStartSpotting(const std::string& activationModelName, bool hasStartupInfo)
{
    aliceStateMachine_.onStartSpotting(activationModelName, hasStartupInfo);
}

void AliceCapability::onVinsRequestBegin(const std::string& requestId) {
    if (auto request = findAliceRequest(requestId)) {
        YIO_LOG_INFO("onVinsRequestBegin: requestId=" << requestId);

        if (!request->getIsSilent() && !request->getIsParallel()) {
            aliceStateMachine_.onAliceRequest();
        }
    } else {
        YIO_LOG_INFO("onVinsRequestBegin: request with id='" << requestId << "' not found");
    }
}

void AliceCapability::onRecognitionResults(
    const std::string& requestId,
    const SpeechKit::Recognition& recognition,
    bool endOfUtterance) {
    if (auto request = findAliceRequest(requestId)) {
        aliceStateMachine_.onPartialResult(request->getId(), recognition.getBestResultText());

        if (endOfUtterance) {
            request->setAsrText(recognition.getBestResultText());
            YIO_LOG_INFO("RecognizedPhrase: [" << request->getAsrText() << "]");
        }
    }
}

bool AliceCapability::hasInteraction() const {
    if (const auto& directive = findNonParallelAliceDirective()) {
        return !directive->getRequest()->isPrefetch();
    }

    return playTtsDirective_ != nullptr;
}

SpeechKit::UniProxy::Header AliceCapability::vinsRequestHeader(std::shared_ptr<VinsRequest> vinsRequest)
{
    std::string name;
    std::string nameSpace;

    switch (vinsRequest->getEventSource().type()) {
        case TSpeechKitRequestProto_TEventSource_EType_Text: {
            name = "TextInput";
            nameSpace = "Vins";
            break;
        }
        case TSpeechKitRequestProto_TEventSource_EType_Voice: {
            name = "VoiceInput";
            nameSpace = "Vins";
            break;
        }
        case TSpeechKitRequestProto_TEventSource_EType_Music: {
            name = "MusicInput";
            nameSpace = "Vins";
            break;
        }
        case TSpeechKitRequestProto_TEventSource_EType_VoiceprintMatch: {
            name = "MatchedUser";
            nameSpace = "System";
            break;
        }
        default:
            throw std::runtime_error("Unknown vins request type");
    }

    SpeechKit::UniProxy::Header header(makeUUID(), nameSpace, name);
    header.requestId = vinsRequest->getId();
    header.parentMessageId = vinsRequest->getParentMessageId();
    header.isParallelRequest = vinsRequest->getIsParallel();

    return header;
}

Json::Value AliceCapability::vinsRequestPayload(std::shared_ptr<VinsRequest> vinsRequest) const {
    Json::Value payload;

    {
        Json::Value header;
        header["request_id"] = vinsRequest->getId();
        if (!aliceConfig_.getSingleDialogMode().empty()) {
            header["dialog_id"] = aliceConfig_.getSingleDialogMode();
        }

        payload["header"] = std::move(header);
    }

    {
        Json::Value request;
        request["event"] = vinsRequest->getEvent();

        if (vinsRequest->isPrefetch()) {
            request["event"]["is_warmup"] = true;
        }
        if (vinsRequest->getResetSession()) {
            request["reset_session"] = true;
        }
        if (!aliceDeviceState_.getMegamindCookie().empty()) {
            request["megamind_cookies"] = aliceDeviceState_.getMegamindCookie();
        }
        auto location = aliceDeviceState_.formatLocation();
        if (location != Json::nullValue) {
            request["location"] = std::move(location);
        }

        request["voice_session"] = vinsRequest->getVoiceSession();
        request["activation_type"] = vinsRequest->getActivationType();

        // System.MatchedUser is supposed to use by uniproxy and they're sensitive to message size.
        // Don't fill 'heavy' parts to decrease message size.
        if (!vinsRequest->isVoiceprintMatch()) {
            request["experiments"] = aliceDeviceState_.formatExperiments();

            Json::Value spotterRms;
            if (voiceStats_ != nullptr) {
                spotterRms = VoiceStats::rmsToJsonPacked(voiceStats_->getRms());
            }
            request["additional_options"] = aliceDeviceState_.formatAdditionalOptions(aliceConfig_, std::move(spotterRms));

            request["device_state"] = aliceDeviceState_.formatJson();
            request["environment_state"] = aliceDeviceState_.getEnvironmentState().formatJson();
        }

        if (const auto guestOptions = quasar::convertMessageToJson(vinsRequest->getGuestOptions()); guestOptions.has_value()) {
            request["guest_user_options"] = *guestOptions;
        }

        if (auto enrollmentHeaders = aliceDeviceState_.formatEnrollmentHeaders(); !enrollmentHeaders.isNull()) {
            request["enrollment_headers"] = std::move(enrollmentHeaders);
        }

        payload["request"] = std::move(request);
    }

    if (const auto eventSource = convertMessageToJson(vinsRequest->getEventSource()); eventSource.has_value()) {
        payload["event_source"] = *eventSource;
    }

    if (vinsRequest->getStartingSilenceTimeout().has_value()) {
        payload["startingSilenceTimeoutMs"] = (int)vinsRequest->getStartingSilenceTimeout().value().count();
    }

    if (!aliceConfig_.getVinsUrl().empty()) {
        payload["vinsUrl"] = aliceConfig_.getVinsUrl();
    }

    return payload;
}
