#pragma once

#include "alice_state_machine.h"
#include "i_player.h"
#include "directives/alice_request_directive.h"
#include "directives/play_tts_directive.h"
#include "vins/vins_response.h"

#include <yandex_io/capabilities/alice/interfaces/i_alice_capability.h>
#include <yandex_io/capabilities/file_player/interfaces/i_file_player_capability.h>
#include <yandex_io/interfaces/stereo_pair/i_stereo_pair_provider.h>
#include <yandex_io/services/aliced/alice_config/alice_config.h>
#include <yandex_io/services/aliced/device_state/alice_device_state.h>
#include <yandex_io/services/aliced/directive_processor/directive_processor.h>
#include <yandex_io/services/aliced/speechkit_facade/quasar_voice_dialog.h>

#include <yandex_io/libs/activity_tracker/activity_tracker.h>
#include <yandex_io/libs/activity_tracker/interface/i_activity.h>
#include <yandex_io/libs/ipc/i_connector.h>
#include <yandex_io/libs/telemetry/telemetry.h>
#include <yandex_io/libs/voice_stats/voice_stats.h>

#include <yandex_io/sdk/private/device_context.h>
#include <yandex_io/sdk/interfaces/i_directive_handler.h>
#include <yandex_io/sdk/private/remoting/i_remote_object.h>
#include <yandex_io/sdk/private/remoting/i_remoting_registry.h>

#include <speechkit/Error.h>
#include <speechkit/Recognition.h>

#include <list>
#include <memory>

namespace YandexIO {

    class AliceBlockers {
    public:
        void block(const std::string& id, const std::optional<std::string>& errorSound);
        void unblock(const std::string& id);

        bool isBlocked();
        std::optional<std::string> getErrorSound() const;
        std::vector<std::string> getBlockerIds() const;
        std::string getBlockersText() const;

    private:
        std::map<std::string, std::optional<std::string>> blockers_;
    };

    class AliceCapability
        : public IDirectiveHandler,
          public IActivity,
          public IPlayer::IListener,
          public IAliceCapability,
          public IRemoteObject,
          public std::enable_shared_from_this<AliceCapability> {
        struct DialogProcessorState {
            bool isReady;
            std::string message;
            bool isBrick = false;
            std::optional<std::string> errorSound = std::nullopt;
        };

    public:
        AliceCapability(const quasar::AliceConfig& aliceConfig,
                        quasar::AliceDeviceState& aliceDeviceState,
                        ActivityTracker& activityTracker,
                        std::weak_ptr<IDirectiveProcessor> directiveProcessor,
                        DeviceContext& deviceContext,
                        const std::shared_ptr<ITelemetry>& telemetry,
                        std::weak_ptr<quasar::QuasarVoiceDialog> voiceService,
                        std::shared_ptr<quasar::ipc::IServer> server,
                        std::shared_ptr<quasar::ipc::IConnector> interfacedConnector,
                        std::shared_ptr<quasar::VoiceStats> voiceStats,
                        std::weak_ptr<IRemotingRegistry> remotingRegistry,
                        std::shared_ptr<quasar::IStereoPairProvider> stereoPairProvider,
                        std::weak_ptr<IFilePlayerCapability> filePlayerCapability);
        ~AliceCapability();

        void init();

        // IAliceCapability implementation
        //
        void startRequest(std::shared_ptr<VinsRequest> request, std::shared_ptr<IAliceRequestEvents> events) override;
        void cancelDialog() override;
        void cancelDialogAndClearQueue() override;
        void startConversation(const VinsRequest::EventSource& eventSource) override;
        void stopConversation() override;
        void toggleConversation(const VinsRequest::EventSource& eventSource) override;
        void finishConversationVoiceInput() override;
        void addListener(std::weak_ptr<IAliceCapabilityListener> listener) override;
        void removeListener(std::weak_ptr<IAliceCapabilityListener> listener) override;

        const quasar::proto::AliceState& getAliceState() const;
        void onExternalAliceState(quasar::proto::AliceState state);
        void onHasStartupInfo(bool hasAllStartupInfo);
        void onStartSpotting(const std::string& activationModelName, bool hasStartupInfo);
        //

        void onVinsRequestBegin(const std::string& requestId);
        void onRecognitionResults(const std::string& requestId, const SpeechKit::Recognition& recognition, bool endOfUtterance);

        void onRequestCompleted(
            const std::shared_ptr<VinsRequest>& request,
            VinsResponse response,
            const Json::Value& payload);
        void onRequestError(
            const std::shared_ptr<VinsRequest>& request,
            const SpeechKit::Error& error);
        void onRequestError(
            const std::shared_ptr<VinsRequest>& request,
            const std::string& errorCode,
            const std::string& errorText);

        bool hasInteraction() const;

        void handleAudioClientMessage(const quasar::ipc::SharedMessage& message);
        void handleAudioClientConnectionStatus(bool connected);

        std::shared_ptr<VinsRequest> findAliceRequest(const std::string& requestId) const;
        std::shared_ptr<VinsRequest> findNonParallelAliceRequest() const;

        AliceBlockers& getAliceBlockers();

        // IDirectiveHandler implementation
        //
        const std::string& getHandlerName() const override;
        const std::set<std::string>& getSupportedDirectiveNames() const override;
        void handleDirective(const std::shared_ptr<Directive>& directive) override;
        void cancelDirective(const std::shared_ptr<Directive>& directive) override;
        void prefetchDirective(const std::shared_ptr<Directive>& directive) override;

        // IPlayer::IListener implementation
        //
        void onPlayingBegin(std::shared_ptr<VinsRequest> request) override;
        void onPlayingEnd(std::shared_ptr<VinsRequest> request) override;
        void onPlayingError(std::shared_ptr<VinsRequest> request, const SpeechKit::Error& error) override;

        // IActivity implementation
        //
        std::string activityName() const override;
        void setBackground() override;
        void setForeground() override;
        quasar::proto::AudioChannel getAudioChannel() const override;
        bool isLocal() const override;

        // IRemoteObject implementation
        //
        void handleRemotingMessage(const quasar::proto::Remoting& message,
                                   std::shared_ptr<YandexIO::IRemotingConnection> connection) override;

    private:
        void startRequestImpl(std::shared_ptr<VinsRequest> request);
        static SpeechKit::UniProxy::Header vinsRequestHeader(std::shared_ptr<VinsRequest> vinsRequest);
        Json::Value vinsRequestPayload(std::shared_ptr<VinsRequest> vinsRequest) const;

        void onRequestError(
            const std::shared_ptr<VinsRequest>& request,
            const std::string& errorCode,
            const std::string& errorText,
            bool success);

        void startAliceRequest(
            const std::shared_ptr<Directive>& directive,
            const std::shared_ptr<VinsRequest>& request,
            bool isPrefetch);
        void startAliceRequestDirective(const std::shared_ptr<Directive>& directive, bool isPrefetch);
        void completeTtsDirective();
        void completeRequestDirective(const std::shared_ptr<Directive>& directive);

        std::shared_ptr<AliceRequestDirective> findNonParallelAliceDirective() const;
        std::shared_ptr<AliceRequestDirective> findAliceRequestDirective(const std::shared_ptr<VinsRequest>& request) const;
        std::shared_ptr<AliceRequestDirective> popAliceRequestDirective(const std::shared_ptr<Directive>& directive);
        std::shared_ptr<AliceRequestDirective> findAliceRequestDirective(const std::shared_ptr<Directive>& directive) const;

        DialogProcessorState getBrickState(const std::shared_ptr<VinsRequest>& request) const;
        DialogProcessorState getDialogProcessorState(const std::shared_ptr<VinsRequest>& request);

    private:
        ActivityTracker& activityTracker_;
        const std::weak_ptr<IDirectiveProcessor> directiveProcessor_;
        std::shared_ptr<PlayTtsDirective> playTtsDirective_;
        const std::shared_ptr<ITelemetry> telemetry_;
        quasar::AliceDeviceState& aliceDeviceState_;
        std::list<std::shared_ptr<AliceRequestDirective>> aliceRequestDirectives_;
        AliceBlockers aliceBlockers_;
        const quasar::AliceConfig& aliceConfig_;
        DeviceContext& deviceContext_;
        std::weak_ptr<quasar::QuasarVoiceDialog> voiceService_;
        std::shared_ptr<quasar::ipc::IServer> server_;
        std::shared_ptr<quasar::VoiceStats> voiceStats_;
        AliceStateMachine aliceStateMachine_;
        const std::shared_ptr<quasar::IStereoPairProvider> stereoPairProvider_;
        const std::weak_ptr<IFilePlayerCapability> filePlayerCapability_;
    };

} // namespace YandexIO
