#include "alice_state_machine.h"

#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/debug.h>
#include <yandex_io/protos/quasar_proto.pb.h>

#include <google/protobuf/util/message_differencer.h>

using namespace quasar;
using namespace YandexIO;

AliceStateMachine::AliceStateMachine(
    std::shared_ptr<ipc::IServer> server, std::shared_ptr<ipc::IConnector> interfacedConnector)
    : server_(std::move(server))
    , interfacedConnector_(std::move(interfacedConnector))
{
    aliceState_.set_state(proto::AliceState::NONE);
}

const quasar::proto::AliceState& AliceStateMachine::getState() const {
    return aliceState_;
}

void AliceStateMachine::onHasStartupInfo(bool hasAllStartupInfo)
{
    proto::AliceState newState = aliceState_;
    newState.set_has_startup_requirements(hasAllStartupInfo);

    setState(std::move(newState));
}

void AliceStateMachine::onStartSpotting(const std::string& activationSpotterModel, bool hasAllStartupInfo)
{
    proto::AliceState newState;
    newState.set_state(proto::AliceState::IDLE);
    newState.set_activation_spotter_model(TString(activationSpotterModel));
    newState.set_has_startup_requirements(hasAllStartupInfo);

    setState(std::move(newState));
}

void AliceStateMachine::onRecognitionBegin(const std::string& requestId, bool isMusic)
{
    if (isMusic) {
        onShazam();
    }

    proto::AliceState newState;
    newState.set_request_id(TString(requestId));
    newState.set_state(proto::AliceState::LISTENING);
    newState.set_has_startup_requirements(aliceState_.has_startup_requirements());

    setState(std::move(newState));
}

void AliceStateMachine::onShazam()
{
    proto::AliceState newState;
    newState.set_state(proto::AliceState::SHAZAM);
    newState.set_has_startup_requirements(aliceState_.has_startup_requirements());

    setState(std::move(newState));
}

void AliceStateMachine::onPartialResult(const std::string& requestId, const std::string& partialResult)
{
    proto::AliceState newState;
    newState.set_request_id(TString(requestId));
    newState.set_state(proto::AliceState::LISTENING);
    newState.set_recognized_phrase(TString(partialResult));
    newState.set_has_startup_requirements(aliceState_.has_startup_requirements());

    setState(std::move(newState));
}

void AliceStateMachine::onAliceRequest()
{
    proto::AliceState newState;
    newState.set_state(proto::AliceState::BUSY);
    newState.set_has_startup_requirements(aliceState_.has_startup_requirements());

    setState(std::move(newState));
}

void AliceStateMachine::onAliceResponse(proto::VinsResponse vinsResponse)
{
    proto::AliceState state = aliceState_;
    *state.mutable_vins_response() = vinsResponse;

    setState(std::move(state));
}

void AliceStateMachine::onSayingBegin()
{
    proto::AliceState newState = aliceState_;
    newState.set_state(proto::AliceState::SPEAKING);

    setState(std::move(newState));
}

void AliceStateMachine::onTtsCompleted()
{
    {
        auto message = quasar::ipc::buildMessage([&](auto& msg) {
            auto method = msg.mutable_remoting()->mutable_alice_capability_listener_method();
            msg.mutable_remoting()->set_remote_object_id("AliceCapability");

            method->set_method(quasar::proto::Remoting::AliceCapabilityListenerMethod::TTS_COMPLETED);
        });

        server_->sendToAll(message);
    }

    for (const auto& wlistener : wlisteners_) {
        if (auto listener = wlistener.lock()) {
            listener->onAliceTtsCompleted();
        }
    }
}

void AliceStateMachine::onAliceStateOverride(proto::AliceState state)
{
    state.set_has_startup_requirements(aliceState_.has_startup_requirements());
    setState(std::move(state));
}

void AliceStateMachine::addListener(std::weak_ptr<IAliceCapabilityListener> listener)
{
    wlisteners_.push_back(listener);
}

void AliceStateMachine::removeListener(std::weak_ptr<IAliceCapabilityListener> listener)
{
    const auto iter = std::find_if(wlisteners_.begin(), wlisteners_.end(),
                                   [listener](const auto& wp) {
                                       return !(wp.owner_before(listener) || listener.owner_before(wp));
                                   });

    if (iter != wlisteners_.end()) {
        wlisteners_.erase(iter);
    }
}

void AliceStateMachine::broadcastAliceState()
{
    auto message = ipc::buildMessage([&](auto& msg) {
        msg.mutable_alice_state()->CopyFrom(aliceState_);
    });

    if (interfacedConnector_ != nullptr) {
        interfacedConnector_->sendMessage(message);
    }

    server_->sendToAll(message);

    {
        auto message = quasar::ipc::buildMessage([&](auto& msg) {
            auto method = msg.mutable_remoting()->mutable_alice_capability_listener_method();
            msg.mutable_remoting()->set_remote_object_id("AliceCapability");

            method->mutable_state()->CopyFrom(aliceState_);
            method->set_method(quasar::proto::Remoting::AliceCapabilityListenerMethod::STATE_CHANGED);
        });

        server_->sendToAll(message);
    }

    for (const auto& wlistener : wlisteners_) {
        if (auto listener = wlistener.lock()) {
            listener->onAliceStateChanged(aliceState_);
        }
    }
}

void AliceStateMachine::setState(proto::AliceState newState)
{
    if (newState.activation_spotter_model().empty()) {
        newState.set_activation_spotter_model(aliceState_.activation_spotter_model());
    }

    if (google::protobuf::util::MessageDifferencer::Equals(newState, aliceState_)) {
        return;
    }

    YIO_LOG_INFO("onAliceStateChanged: " << shortUtf8DebugString(newState));
    aliceState_ = std::move(newState);
    broadcastAliceState();
}
