#include "audio_client_tts_player.h"

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/logging/logging.h>

#include <chrono>
#include <vector>

using namespace YandexIO;

namespace {
    std::vector<char> generateSilence(int rate, int sampleSize, std::chrono::milliseconds time) {
        const size_t dataInMs = rate * sampleSize * time.count() / 1000;
        constexpr char silenceValue = 0;
        return std::vector<char>(dataInMs, silenceValue);
    }
} // anonymous namespace

quasar::proto::AudioPlayerDescriptor AudioClientTtsPlayer::createDescriptor() {
    quasar::proto::AudioPlayerDescriptor result;
    result.set_type(quasar::proto::AudioPlayerDescriptor::VOICE);
    result.set_player_id(quasar::makeUUID());
    return result;
}

AudioClientTtsPlayer::AudioClientTtsPlayer(
    std::shared_ptr<quasar::ICallbackQueue> worker,
    SpeechKit::TTSDataProvider::SharedPtr ttsDataProvider,
    std::shared_ptr<quasar::ipc::IConnector> audioClientConnector,
    std::chrono::milliseconds silenceTimeAfterTts,
    std::shared_ptr<VinsRequest> request)
    : worker_(std::move(worker))
    , ttsDataProvider_(ttsDataProvider)
    , audioClientConnector_(audioClientConnector)
    , descriptor_(createDescriptor())
    , silenceTimeAfterTts_(silenceTimeAfterTts)
    , request_(std::move(request))
{
}

void AudioClientTtsPlayer::play(const quasar::proto::AudioAnalyticsContext& analyticsContext) {
    quasar::proto::QuasarMessage command;
    auto& mediaRequest = *command.mutable_media_request();
    *mediaRequest.mutable_player_descriptor() = descriptor_;
    auto& playAudio = *mediaRequest.mutable_play_audio();
    playAudio.set_format(quasar::proto::Audio::PCM_STREAM);
    playAudio.mutable_analytics_context()->CopyFrom(analyticsContext);
    // avoid metrics for TTS
    playAudio.set_report_metrics(false);

    canceled_ = false;
    audioClientConnector_->sendMessage(std::move(command));
    ttsDataProvider_->init(shared_from_this());
    ttsDataProvider_->start();
}

void AudioClientTtsPlayer::pause() {
    if (canceled_) {
        YIO_LOG_INFO("Skipping pause command due to player cancellation");
        return;
    }

    quasar::proto::QuasarMessage command;
    auto& mediaRequest = *command.mutable_media_request();
    *mediaRequest.mutable_player_descriptor() = descriptor_;
    mediaRequest.mutable_pause();
    audioClientConnector_->sendMessage(std::move(command));
    ttsDataProvider_->pause();
}

void AudioClientTtsPlayer::resume() {
    if (canceled_) {
        YIO_LOG_INFO("Skipping resume command due to player cancellation");
        return;
    }

    quasar::proto::QuasarMessage command;
    auto& mediaRequest = *command.mutable_media_request();
    *mediaRequest.mutable_player_descriptor() = descriptor_;
    mediaRequest.set_resume(quasar::proto::MediaRequest::CURRENT);
    audioClientConnector_->sendMessage(std::move(command));
    ttsDataProvider_->start();
}

void AudioClientTtsPlayer::cancel() {
    if (canceled_) {
        YIO_LOG_INFO("Skipping cancel command due to player cancellation");
        return;
    }

    sendDestroyPlayer();
    canceled_ = true;
}

void AudioClientTtsPlayer::sendDestroyPlayer() {
    quasar::proto::QuasarMessage command;
    auto& mediaRequest = *command.mutable_media_request();
    *mediaRequest.mutable_player_descriptor() = descriptor_;
    mediaRequest.mutable_clean_players();
    audioClientConnector_->sendMessage(std::move(command));
}

void AudioClientTtsPlayer::handleAudioClientConnectionStatus(bool connected) {
    Y_ENSURE_THREAD(worker_);
    if (!connected) {
        SpeechKit::Error error(SpeechKit::Error::ErrorAudioPlayer, "Audioclient disconnected.");
        cancel();
        onPlayingError(error);
    }
}

void AudioClientTtsPlayer::handleAudioClientEvent(const quasar::proto::AudioClientEvent& event) {
    Y_ENSURE_THREAD(worker_);
    if (event.has_player_descriptor() && event.player_descriptor().player_id() == descriptor_.player_id()) {
        if (!event.has_state()) {
            return;
        }
        switch (event.state()) {
            case quasar::proto::AudioClientState::PLAYING: {
                onPlayingBegin();
                break;
            }
            case quasar::proto::AudioClientState::FINISHED: {
                sendDestroyPlayer();
                onPlayingEnd();
                break;
            }
            case quasar::proto::AudioClientState::FAILED: {
                SpeechKit::Error error(SpeechKit::Error::ErrorAudioPlayer, "Gstreamer audioclient player error: " + event.error_text());
                sendDestroyPlayer();
                onPlayingError(error);
                break;
            }
            default:
                // ignore all other player states
                break;
        }
    }
}

void AudioClientTtsPlayer::setListener(std::weak_ptr<IListener> listener)
{
    listener_ = std::move(listener);
}

void AudioClientTtsPlayer::onStreamBegin() {
    // do nothing
}

void AudioClientTtsPlayer::onStreamData(SpeechKit::SoundBuffer::SharedPtr data) {
    if (canceled_) {
        return;
    }

    sendStream(reinterpret_cast<const char*>(data->getData().data()), data->getData().size());
}

void AudioClientTtsPlayer::onStreamEnd() {
    if (canceled_) {
        YIO_LOG_INFO("Skipping end of stream command due to player cancellation");
        return;
    }

    if (silenceTimeAfterTts_ > std::chrono::milliseconds(0)) {
        // Hack for TTS to split TTS from piluk
        // Generate silence with sampleSize=2 and rate=48000
        YIO_LOG_DEBUG("Generating " << silenceTimeAfterTts_.count() << " ms of silence at the end of TTS");
        const auto silence = generateSilence(48000, 2, silenceTimeAfterTts_);
        sendStream(silence.data(), silence.size());
    }

    quasar::proto::QuasarMessage command;
    auto& mediaRequest = *command.mutable_media_request();
    *mediaRequest.mutable_player_descriptor() = descriptor_;
    mediaRequest.mutable_stream_data_end();
    audioClientConnector_->sendMessage(std::move(command));
}

void AudioClientTtsPlayer::onStreamError(const SpeechKit::Error& error) {
    worker_->add([this, error]() {
        onPlayingError(error);
    });
}

void AudioClientTtsPlayer::sendStream(const char* data, size_t size) {
    quasar::proto::QuasarMessage command;
    auto& mediaRequest = *command.mutable_media_request();
    *mediaRequest.mutable_player_descriptor() = descriptor_;
    mediaRequest.set_stream_data(data, size);
    audioClientConnector_->sendMessage(std::move(command));
}

void AudioClientTtsPlayer::onPlayingBegin()
{
    Y_ENSURE_THREAD(worker_);
    if (!std::exchange(isStarted_, true)) {
        if (auto listener = listener_.lock()) {
            listener->onPlayingBegin(request_);
        }
    }
}

void AudioClientTtsPlayer::onPlayingEnd()
{
    Y_ENSURE_THREAD(worker_);
    if (!std::exchange(isFinished_, true)) {
        if (auto listener = listener_.lock()) {
            listener->onPlayingEnd(request_);
        }
    }
}

void AudioClientTtsPlayer::onPlayingError(const SpeechKit::Error& error)
{
    Y_ENSURE_THREAD(worker_);
    if (!std::exchange(isFinished_, true)) {
        if (auto listener = listener_.lock()) {
            listener->onPlayingError(request_, error);
        }
    }
}
