#include "alice_capability_preprocessor.h"

#include <yandex_io/libs/base/directives.h>

using namespace quasar;

AliceCapabilityPreprocessor::AliceCapabilityPreprocessor(
    bool isBtCapability, AliceDeviceState& aliceDeviceState)
    : isBtCapability_(isBtCapability)
    , aliceDeviceState_(aliceDeviceState)
{
}

const std::string& AliceCapabilityPreprocessor::getPreprocessorName() const {
    static const std::string s_name = "AliceCapabilityPreprocessor";
    return s_name;
}

void AliceCapabilityPreprocessor::preprocessDirectives(std::list<std::shared_ptr<YandexIO::Directive>>& directives)
{
    moveDrawLedDirectivesToBeginning(directives);
    expandGoHomeDirectives(directives);
}

void AliceCapabilityPreprocessor::moveDrawLedDirectivesToBeginning(
    std::list<std::shared_ptr<YandexIO::Directive>>& directives)
{
    directives.sort(
        [](const std::shared_ptr<YandexIO::Directive>& left, const std::shared_ptr<YandexIO::Directive>& right) {
            return left->is(Directives::DRAW_LED_SCREEN) &&
                   !right->is(Directives::DRAW_LED_SCREEN);
        });
}

void AliceCapabilityPreprocessor::expandGoHomeDirectives(
    std::list<std::shared_ptr<YandexIO::Directive>>& directives)
{
    for (auto iter = directives.begin(); iter != directives.end(); ++iter) {
        if (const auto directive = *iter; directive->is(Directives::GO_HOME)) {
            iter = directives.erase(iter);
            iter = directives.insert(iter, YandexIO::Directive::createLocalAction(Directives::SHOW_HOME_SCREEN, Json::Value(), directive->getData().endpointId));
            iter = directives.insert(iter, YandexIO::Directive::createLocalAction(getPlayerPauseDirective(), Json::Value(), directive->getData().endpointId));
        }
    }
}

std::string AliceCapabilityPreprocessor::getPlayerPauseDirective() const {
    const auto playerType = aliceDeviceState_.getCurrentPlayerType();
    const bool isLegacyPlayer = aliceDeviceState_.getCurrentPlayerType() != AudioPlayerType::AUDIO_CLIENT || aliceDeviceState_.hasVideoPlayerScreen();

    if (isBtCapability_ && playerType == AudioPlayerType::BLUETOOTH) {
        return Directives::BLUETOOTH_PLAYER_PAUSE;
    } else if (isLegacyPlayer) {
        return Directives::PLAYER_PAUSE;
    } else {
        return Directives::AUDIO_STOP;
    }
}
