#include "local_vins_preprocessor.h"

#include <yandex_io/libs/base/directives.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/json.h>
#include <alice/protos/endpoint/capability.pb.h>

using namespace quasar;

namespace {

    bool containsAll(const std::string& string, const std::vector<std::string>& subStrings) {
        for (const auto& subString : subStrings) {
            if (string.find(subString) == std::string::npos) {
                return false;
            }
        }
        return true;
    }

    bool containsAny(const std::string& string, const std::vector<std::string>& subStrings) {
        for (const auto& subString : subStrings) {
            if (string.find(subString) != std::string::npos) {
                return true;
            }
        }
        return false;
    }

} // namespace

LocalVinsPreprocessor::LocalVinsPreprocessor(std::shared_ptr<YandexIO::IDevice> device)
    : device_(std::move(device))
{
}

const std::string& LocalVinsPreprocessor::getPreprocessorName() const {
    static const std::string s_name = "LocalVinsPreprocessor";
    return s_name;
}

void LocalVinsPreprocessor::preprocessDirectives(std::list<std::shared_ptr<YandexIO::Directive>>& directives)
{
    if (directives.empty()) {
        return;
    }

    const auto context = (*directives.begin())->getData();
    const auto& text = context.asrText;

    if (device_->platform() == "yandexmini") {
        if (containsAny(text, {"Подорожник", "подорожник"})) {
            directives = {createBioStartSoundEnrollment()};
        }
    }

    if (device_->platform() == "yandexmini_2") {
        if (containsAny(text, {"апуста", "ветная"})) {
            YIO_LOG_INFO("Local vins DRAW_SCLED_ANIMATION directive");

            bool isLooped = false;
            bool isDuringTTS = false;
            if (containsAny(text, {"цикл"})) {
                isLooped = true;
            }
            if (containsAny(text, {"диалог"})) {
                isDuringTTS = true;
            }
            directives = {
                createDrawSegmentAnimationDirective(context, isLooped, isDuringTTS)};
        }
        return;
    }

    if (device_->platform() == "yandexmidi") {
        if (isLavaLampTurnOff(text)) {
            YIO_LOG_INFO("Turning lava lamp off: " << text);

            directives = {
                createColorSceneDirective(context, false, ""),
                createRepeatDirective("выключаю")};
        } else if (isCandleTurnOn(text)) {
            YIO_LOG_INFO("Turning candle lamp on: " << text);

            directives = {
                createColorSceneDirective(context, true, "candle"),
            };
        } else if (isLavaLampTurnOn(text)) {
            YIO_LOG_INFO("Turning lava lamp on: " << text);

            directives = {
                createColorSceneDirective(context, true, "lava"),
            };
        } else if (isNightLampTurnOn(text)) {
            YIO_LOG_INFO("Turning night lamp on: " << text);

            directives = {
                createColorSceneDirective(context, true, "night"),
            };
        } else if (containsAny(text, {"огурец", "Огурец"})) {
            NAlice::TAnimationCapability::TDrawAnimationDirective directive;
            auto* animation = directive.AddAnimations();
            animation->MutableS3Directory()->SetBucket("https://quasar.s3.yandex.net");
            animation->MutableS3Directory()->SetPath("animations/weather/clear");
            directive.SetSpeakingAnimationPolicy(NAlice::TAnimationCapability::TDrawAnimationDirective::PlaySpeakingEndOfTts);

            YandexIO::Directive::Data data("draw_animation_directive", "client_action");
            data.payload = quasar::convertMessageToJson(directive).value();
            directives.push_front(std::make_shared<YandexIO::Directive>(std::move(data)));

        } else if (isZigbeeEnable(text)) {
            YIO_LOG_INFO("Enabling Zigbee network");

            Json::Value payload;
            payload["protocol"] = "Zigbee";
            payload["enabled"] = true;

            directives = {
                createExternalCommandDirective(context, "iot_enable_network_directive", std::move(payload)),
                createRepeatDirective("включаю сеть"),
            };
        } else if (isZigbeeDisable(text)) {
            YIO_LOG_INFO("Disabling Zigbee network");

            Json::Value payload;
            payload["protocol"] = "Zigbee";
            payload["enabled"] = false;

            directives = {
                createExternalCommandDirective(context, "iot_enable_network_directive", std::move(payload)),
                createRepeatDirective("выключаю сеть"),
            };
        } else if (isIdleAnimationEnable(text)) {
            directives = {
                createExternalCommandDirective(context, "enable_idle_animation"),
            };
        } else if (isIdleAnimationDisable(text)) {
            directives = {
                createExternalCommandDirective(context, "disable_idle_animation"),
            };
        }

        return;
    }

    if (isTOFWasNotUsedToday(text)) {
        YIO_LOG_DEBUG("LocalVinsPreprocessor. Tof not used today");
        device_->telemetry()->reportEvent("TOFNotUsedReport");

        directives = {
            createRepeatDirective("ок поняла, отправляю статистику"),
        };
    } else if (isStartSensorDumpToCsv(text)) {
        YIO_LOG_INFO("LocalVinsPreprocessor. Start dump to csv");

        directives = {
            createExternalCommandDirective(context, Directives::TOF_TO_CSV_START),
            createRepeatDirective("ок поняла, начинаю запись датчика")};
    } else if (isStopSensorDumpToCsv(text)) {
        YIO_LOG_INFO("LocalVinsPreprocessor. Stop dump to csv");

        directives = {
            createExternalCommandDirective(context, Directives::TOF_TO_CSV_STOP),
            createRepeatDirective("запись окончена, можете забрать файлик")};
    } else if (isFindRcu(text)) {
        YIO_LOG_INFO("LocalVinsPreprocessor. Find RCU");

        directives = {
            createExternalCommandDirective(context, Directives::SETUP_RCU),
            createRepeatDirective("начинаю поиск пультика")};
    }
}

bool LocalVinsPreprocessor::isTurnOnThereminvox(const std::string& text) {
    return containsAll(text, {"включи терменвокс"}) || (containsAll(text, {"пентатоник"}) && containsAll(text, {"гра"})) ||
           containsAll(text, {"огурец"});
}

bool LocalVinsPreprocessor::isTOFWasNotUsedToday(const std::string& text) {
    // TODO "jasta", "jester", "justus"
    return containsAll(text, {"я сегодня не", "пользовал", "жест"});
}

int LocalVinsPreprocessor::getPositiveNumber(const std::string& text) {
    for (size_t i = 1; i <= 9; ++i) {
        if (text.find(std::to_string(i)) != std::string::npos) {
            return i;
        }
    }
    return -1;
}

bool LocalVinsPreprocessor::isStartSensorDumpToCsv(const std::string& text) {
    return containsAll(text, {"начни писать фотоны"});
}

bool LocalVinsPreprocessor::isStopSensorDumpToCsv(const std::string& text) {
    return containsAll(text, {"перестань писать фотоны"});
}

std::shared_ptr<YandexIO::Directive> LocalVinsPreprocessor::createBioStartSoundEnrollment() {
    YandexIO::Directive::Data data("bio_start_sound_enrollment", "client_action");
    auto directive = std::make_shared<YandexIO::Directive>(std::move(data));
    return directive;
}

std::shared_ptr<YandexIO::Directive> LocalVinsPreprocessor::createColorSceneDirective(
    const YandexIO::Directive::Data& context, bool enabled, const std::string& scene) {
    YandexIO::Directive::Data data("set_color_scene_directive", "client_action");
    if (!enabled) {
        data.payload["color_scene"] = "Inactive";
    } else {
        if (scene == "lava") {
            data.payload["color_scene"] = "LavaLampScene";
        } else if (scene == "candle") {
            data.payload["color_scene"] = "CandleScene";
        } else if (scene == "night") {
            data.payload["color_scene"] = "NightScene";
        } else {
            data.payload["color_scene"] = "Inactive";
        }
    }

    data.setContext(context.asrText, context.requestId, "", context.displayedText);
    auto directive = std::make_shared<YandexIO::Directive>(std::move(data));

    return directive;
}

std::shared_ptr<YandexIO::Directive> LocalVinsPreprocessor::createDrawSegmentAnimationDirective(
    const YandexIO::Directive::Data& context, bool isLooped, bool isDuringTTS) {
    YandexIO::Directive::Data data(Directives::DRAW_SCLED_ANIMATION, "client_action");

    Json::Value animation;
    animation["name"] = "test_pause";
    animation["compression_type"] = "None";
    animation["base64_encoded_value"] = "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABQAAAAAAAAAAAASEgAAAAAAAAAAEhIAAAAAAAAAAAAAAAAAFAAAAAAAAAAAACQkAAAAAAAAAAAkJAAAAAAAAAAAAAAAAAAUAAAAAAAAAAAANzcAAAAAAAAAADc3AAAAAAAAAAAAAAAAABQAAAAAAAAAAABJSQAAAAAAAAAASUkAAAAAAAAAAAAAAAAAFAAAAAAAAAAAAFtbAAAAAAAAAABbWwAAAAAAAAAAAAAAAAAUAAAAAAAAAAAAbW0AAAAAAAAAAG1tAAAAAAAAAAAAAAAAABQAAAAAAAAAAACAgAAAAAAAAAAAgIAAAAAAAAAAAAAAAAAAFAAAAAAAAAAAAJKSAAAAAAAAAACSkgAAAAAAAAAAAAAAAAAUAAAAAAAAAAAApKQAAAAAAAAAAKSkAAAAAAAAAAAAAAAAABQAAAAAAAAAAAC2tgAAAAAAAAAAtrYAAAAAAAAAAAAAAAAAFAAAAAAAAAAAAMjIAAAAAAAAAADIyAAAAAAAAAAAAAAAAAAUAAAAAAAAAAAA29sAAAAAAAAAANvbAAAAAAAAAAAAAAAAABQAAAAAAAAAAADt7QAAAAAAAAAA7e0AAAAAAAAAAAAAAAAAFAAAAAAAAAAAAP//AAAAAAAAAAD//wAAAAAAAAAAAAAAAAakAAAAAAAAAAAA6uoAAAAAAAAAAOrqAAAAAAAAAAAAAAAAABQAAAAAAAAAAADV1QAAAAAAAAAA1dUAAAAAAAAAAAAAAAAAFAAAAAAAAAAAAL+/AAAAAAAAAAC/vwAAAAAAAAAAAAAAAAAUAAAAAAAAAAAAqqoAAAAAAAAAAKqqAAAAAAAAAAAAAAAAABQAAAAAAAAAAACVlQAAAAAAAAAAlZUAAAAAAAAAAAAAAAAAFAAAAAAAAAAAAICAAAAAAAAAAACAgAAAAAAAAAAAAAAAAAAUAAAAAAAAAAAAamoAAAAAAAAAAGpqAAAAAAAAAAAAAAAAABQAAAAAAAAAAABVVQAAAAAAAAAAVVUAAAAAAAAAAAAAAAAAFAAAAAAAAAAAAEBAAAAAAAAAAABAQAAAAAAAAAAAAAAAAAAUAAAAAAAAAAAAKysAAAAAAAAAACsrAAAAAAAAAAAAAAAAABQAAAAAAAAAAAAVFQAAAAAAAAAAFRUAAAAAAAAAAAAAAAAAFA==";
    data.payload["enabled"]["animations"].append(animation);

    Json::Value compressedAnimation;
    compressedAnimation["name"] = "test_pause_compressed";
    compressedAnimation["compression_type"] = "Gzip";
    animation["base64_encoded_value"] = "H4sIAAAAAAAAA4XMsQ1EAABGYZUB5GygsoBWawCVDSRmYAaJDVQGEJ3WADobSMzgcpEo3ivur77i5Q+Cf/u8iiILSZJYSLLMQlIUFpKqspA0jYWk6ywkfW8hGUcLyTxbSLbNQnIcFpLrspDct/VbuLw8Twsv+24hWVcLyTRZSIbBQtK2FpK6tpCUpYUkzy0kaWohiWPrSb58QFRojgMAAA==";
    data.payload["enabled"]["animations"].append(compressedAnimation);

    if (!isLooped && !isDuringTTS) {
        data.payload["animation_stop_policy"] = "PlayOnce";
    } else if (isLooped && !isDuringTTS) {
        data.payload["animation_stop_policy"] = "RepeatLastTillNextDirective";
    } else if (!isLooped && isDuringTTS) {
        data.payload["animation_stop_policy"] = "PlayOnceTillEndOfTTS";
    } else if (isLooped && isDuringTTS) {
        data.payload["animation_stop_policy"] = "RepeatLastTillEndOfTTS";
    }

    data.setContext(context.asrText, context.requestId, "", context.displayedText);
    auto directive = std::make_shared<YandexIO::Directive>(std::move(data));

    return directive;
}

std::shared_ptr<YandexIO::Directive> LocalVinsPreprocessor::createRepeatDirective(const std::string& text) {
    YandexIO::Directive::Data data(Directives::TYPE, "server_action");
    data.payload["text"] = "повтори за мной, " + text;

    return std::make_shared<YandexIO::Directive>(std::move(data));
}

std::shared_ptr<YandexIO::Directive> LocalVinsPreprocessor::createExternalCommandDirective(
    const YandexIO::Directive::Data& context, const std::string& name, Json::Value payload) {
    YandexIO::Directive::Data data(name, "client_action");
    data.payload = std::move(payload);

    data.setContext(context.asrText, context.requestId, "", context.displayedText);
    auto directive = std::make_shared<YandexIO::Directive>(std::move(data));

    return directive;
}

bool LocalVinsPreprocessor::isFindRcu(const std::string& text) {
    return containsAll(text, {"пульт"}) || containsAll(text, {"пультик"});
}

bool LocalVinsPreprocessor::isLavaLampTurnOff(const std::string& text) {
    return containsAll(text, {"Выключи режим"});
}

bool LocalVinsPreprocessor::isLavaLampTurnOn(const std::string& text) {
    return containsAll(text, {"Включи", "режим", "лава"});
}

bool LocalVinsPreprocessor::isCandleTurnOn(const std::string& text) {
    return containsAll(text, {"Включи", "режим", "свеча"});
}

bool LocalVinsPreprocessor::isNightLampTurnOn(const std::string& text) {
    return containsAll(text, {"Включи", "режим", "ночник"});
}

bool LocalVinsPreprocessor::isZigbeeEnable(const std::string& text) {
    return containsAll(text, {"Включи локальную сеть"});
}

bool LocalVinsPreprocessor::isZigbeeDisable(const std::string& text) {
    return containsAll(text, {"Выключи локальную сеть"});
}

bool LocalVinsPreprocessor::isIdleAnimationEnable(const std::string& text) {
    return containsAll(text, {"Включи анимацию просто"});
}

bool LocalVinsPreprocessor::isIdleAnimationDisable(const std::string& text) {
    return containsAll(text, {"Выключи анимацию просто"});
}
