#include "speechkit_tts_player.h"

#include <yandex_io/libs/logging/logging.h>

using namespace YandexIO;

SpeechkitTtsPlayer::SpeechkitTtsPlayer(
    std::shared_ptr<quasar::ICallbackQueue> worker,
    SpeechKit::AudioPlayer::SharedPtr player,
    SpeechKit::TTSDataProvider::SharedPtr ttsDataProvider,
    std::shared_ptr<VinsRequest> request)
    : worker_(std::move(worker))
    , player_(std::move(player))
    , dataProvider_(std::move(ttsDataProvider))
    , request_(std::move(request))
{
}

void SpeechkitTtsPlayer::play(const quasar::proto::AudioAnalyticsContext& /* analyticsContext */) {
    if (player_ != nullptr) {
        player_->cancel();
        player_->subscribe(shared_from_this());

        dataProvider_->init(shared_from_this());
        dataProvider_->start();
    }
}

void SpeechkitTtsPlayer::resume() {
    if (player_ != nullptr) {
        player_->play();
        dataProvider_->start();
    }
}

void SpeechkitTtsPlayer::pause() {
    if (player_ != nullptr) {
        dataProvider_->pause();
        player_->pause();
    }
}

void SpeechkitTtsPlayer::cancel() {
    if (player_ != nullptr && !isFinished_) {
        player_->cancel();
        player_->unsubscribe(shared_from_this());
        player_.reset();
    }
}

void SpeechkitTtsPlayer::setListener(std::weak_ptr<IListener> listener)
{
    listener_ = std::move(listener);
}

void SpeechkitTtsPlayer::onStreamBegin() {
    if (auto player = player_) {
        player->play();
    }
}

void SpeechkitTtsPlayer::onStreamData(SpeechKit::SoundBuffer::SharedPtr data) {
    if (auto player = player_) {
        player->playData(std::move(data));
    }
}

void SpeechkitTtsPlayer::onStreamEnd() {
    if (auto player = player_) {
        player->setDataEnd();
    }
}

void SpeechkitTtsPlayer::onStreamError(const SpeechKit::Error& error) {
    onPlayingError(nullptr, error);
}

void SpeechkitTtsPlayer::onPlayingBegin(SpeechKit::AudioPlayer::SharedPtr /*player*/) {
    worker_->add([this]() {
        if (!std::exchange(isStarted_, true)) {
            if (auto listener = listener_.lock()) {
                listener->onPlayingBegin(request_);
            }
        }
    });
}

void SpeechkitTtsPlayer::onPlayingPaused(SpeechKit::AudioPlayer::SharedPtr /*player*/) {
}

void SpeechkitTtsPlayer::onPlayingResumed(SpeechKit::AudioPlayer::SharedPtr /*player*/) {
}

void SpeechkitTtsPlayer::onPlayingDone(SpeechKit::AudioPlayer::SharedPtr /*player*/) {
    worker_->add([this]() {
        if (!std::exchange(isFinished_, true)) {
            if (auto listener = listener_.lock()) {
                listener->onPlayingEnd(request_);
            }
        }
    });
}

void SpeechkitTtsPlayer::onPlayingError(SpeechKit::AudioPlayer::SharedPtr /*player*/, const SpeechKit::Error& error) {
    worker_->add([this, error]() {
        if (!std::exchange(isFinished_, true)) {
            if (auto listener = listener_.lock()) {
                listener->onPlayingError(request_, error);
            }
        }
    });
}
