#include "vins_response.h"

#include "vins_utils.h"

#include <yandex_io/services/aliced/capabilities/alice_capability/directives/alice_request_directive.h>
#include <yandex_io/services/aliced/capabilities/alice_capability/directives/play_tts_directive.h>

#include <yandex_io/libs/base/directives.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/json.h>

#include <util/system/yassert.h>

using namespace NAlice;
using namespace YandexIO;

bool VinsResponse::Context::isReminder() const {
    return request == nullptr ? false : request->getIsReminder();
}

bool VinsResponse::Context::isPrefetch() const {
    return request == nullptr ? false : request->isPrefetch();
}

std::string VinsResponse::Context::getAsrText() const {
    return request == nullptr ? "" : request->getAsrText();
}

VinsResponse VinsResponse::parse(
    const std::shared_ptr<VinsRequest>& request,
    const SpeechKit::UniProxy::Header& header,
    const Json::Value& payload,
    std::shared_ptr<IPlayer> ttsPlayer,
    const std::set<std::string>& blockShowCardDirectiveNames) {
    VinsResponse result;

    auto context = Context{
        .request = request,
        .ttsPlayer = ttsPlayer};
    std::string outputSpeech;

    context.requestId = quasar::getString(quasar::getJson(payload, "header"), "request_id");
    context.parentRequestId = quasar::tryGetString(quasar::getJson(payload, "header"), "parent_request_id");
    result.hasMegamindError = payload.isMember("error");

    const Json::Value voiceResponse = quasar::getJson(payload, "voice_response");
    const bool shouldListen = quasar::tryGetBool(voiceResponse, "should_listen", false);

    const Json::Value outputSpeechJson = quasar::tryGetJson(voiceResponse, "output_speech");
    if (!outputSpeechJson.isNull()) {
        outputSpeech = quasar::tryGetString(outputSpeechJson, "text");
    }

    const Json::Value response = quasar::getJson(payload, "response");
    if (response.isMember("cards")) {
        const Json::Value cards = response["cards"];
        if (cards.size() > 0) {
            context.displayedText = quasar::tryGetString(quasar::getArrayElement(cards, 0), "text");
        }
    }

    auto vinsResponse = parseSuggests(response);

    std::string divCard = quasar::VinsUtils::getDivCardFromVins(response);
    if (!divCard.empty()) {
        vinsResponse.set_div_card(std::move(divCard));
    }

    bool hasTtsPlaceholder = false;
    const Json::Value directivesJson = quasar::getJson(response, "directives");
    if (directivesJson.empty()) {
        if (!outputSpeech.empty()) {
            vinsResponse.set_text_only(true);
            vinsResponse.set_output_speech(TString(context.displayedText));
        }
    } else {
        bool blockShowCard = false;
        for (const auto& directiveJson : directivesJson) {
            Directive::Data data = Directive::Data::fromJson(directiveJson);
            data.parentMessageId = header.refMessageId;

            if (data.name == quasar::Directives::TTS_PLAY_PLACEHOLDER) {
                if (hasTtsPlaceholder || ttsPlayer == nullptr) {
                    // supports only one tts_play_placeholder for now
                    continue;
                }

                hasTtsPlaceholder = true;
            }

            // Issue https://st.yandex-team.ru/QUASARSUP-172
            if (quasar::Directives::contains(data.name) && blockShowCardDirectiveNames.count(data.name) == 0) {
                blockShowCard = true;
            }

            result.directives.push_back(createDirective(std::move(data), context));
        }

        const bool showCard = !context.displayedText.empty() && !blockShowCard;
        vinsResponse.set_text_only(showCard);
    }

    if (!hasTtsPlaceholder && ttsPlayer != nullptr) {
        Directive::Data data(quasar::Directives::TTS_PLAY_PLACEHOLDER, "local_action");
        auto ttsDirective = createDirective(std::move(data), context);
        result.directives.insert(result.directives.begin(), ttsDirective);
    }

    if (!context.displayedText.empty()) {
        vinsResponse.set_output_speech(TString(context.displayedText));
    }

    if (!result.directives.empty()) {
        (*result.directives.begin())->setFirstInChain();
    }

    if (request == nullptr || !request->getIsParallel()) {
        Directive::Data data(quasar::Directives::ALICE_RESPONSE, "local_action");
        data.setContext(context.getAsrText(), context.requestId, context.parentRequestId, context.displayedText);
        data.isPrefetched = context.isPrefetch();
        data.payload = quasar::convertMessageToJson(vinsResponse).value_or(Json::Value());

        result.directives.push_front(std::make_shared<Directive>(std::move(data)));
    }

    if (shouldListen) {
        Directive::Data data(quasar::Directives::LISTEN, "local_action");
        data.setContext(context.getAsrText(), context.requestId, context.parentRequestId, context.displayedText);

        VinsRequest::EventSource eventSource;
        eventSource.set_event(TSpeechKitRequestProto_TEventSource_EEvent_AutoListen);
        eventSource.set_source(TSpeechKitRequestProto_TEventSource_ESource_Software);
        eventSource.set_type(TSpeechKitRequestProto_TEventSource_EType_Voice);
        eventSource.set_id(TString(quasar::makeUUID()));

        result.directives.push_back(createListenDirective(std::move(data), std::move(eventSource)));
    }

    return result;
}

VinsResponse VinsResponse::parseAsyncDirectives(const SpeechKit::UniProxy::Header& header, const Json::Value& payload)
{
    VinsResponse result;

    const Json::Value directivesJson = quasar::getJson(payload, "directives");
    for (const auto& directiveJson : directivesJson) {
        auto data = Directive::Data::fromJson(directiveJson);
        if (data.name == quasar::Directives::TTS_PLAY_PLACEHOLDER) {
            continue;
        }

        data.parentMessageId = header.refMessageId;
        auto directive = createDirective(std::move(data), Context());
        result.directives.push_back(std::move(directive));
    }

    return result;
}

quasar::proto::VinsResponse VinsResponse::parseSuggests(const Json::Value& vinsPayload)
{
    quasar::proto::VinsResponse vinsResponse;
    if (vinsPayload["suggest"].isNull() || quasar::getJson(vinsPayload, "suggest")["items"].isNull()) {
        return vinsResponse;
    }

    const Json::Value suggests = quasar::getJson(quasar::getJson(vinsPayload, "suggest"), "items");
    for (size_t i = 0; i < suggests.size(); ++i) {
        const Json::Value suggest = suggests[(int)i];

        if (suggest["type"].isNull() || !("action" == quasar::getString(suggest, "type")) || suggest["directives"].isNull()) {
            continue;
        }

        const Json::Value directives = quasar::getJson(suggest, "directives");

        for (size_t j = 0; j < directives.size(); ++j) {
            const Json::Value directive = directives[(int)j];

            if (quasar::tryGetString(directive, "type", "") != "client_action") {
                continue;
            }

            if (directive["payload"].isNull() || quasar::getJson(directive, "payload")["text"].isNull()) {
                continue;
            }

            const Json::Value payload = quasar::getJson(directive, "payload");
            vinsResponse.add_suggests()->set_text(quasar::getString(payload, "text"));
        }
    }

    return vinsResponse;
}

std::shared_ptr<Directive> VinsResponse::createListenDirective(Directive::Data data, VinsRequest::EventSource eventSource) {
    auto listenRequest = VinsRequest::createListenRequest(data, eventSource);
    auto directive = std::make_shared<AliceRequestDirective>(std::move(listenRequest), nullptr, true);

    return directive;
}

std::shared_ptr<Directive> VinsResponse::createDirective(
    Directive::Data data, const Context& context)
{
    data.setContext(context.getAsrText(), context.requestId, context.parentRequestId, context.displayedText);
    auto directive = createDirectiveInternal(data, context);

    return directive;
}

std::shared_ptr<Directive> VinsResponse::createDirectiveInternal(
    Directive::Data data, const Context& context)
{
    if (data.isServerAction()) {
        auto request = VinsRequest::createServerActionRequest(data);
        return std::make_shared<AliceRequestDirective>(std::move(request), nullptr, true, std::move(data));
    }

    if (data.name == quasar::Directives::AUDIO_PLAY) {
        data.channel = quasar::proto::CONTENT_CHANNEL;
        return std::make_shared<YandexIO::Directive>(std::move(data), true);
    }

    if (data.name == quasar::Directives::TTS_PLAY_PLACEHOLDER) {
        Y_VERIFY(context.ttsPlayer != nullptr);
        return std::make_shared<PlayTtsDirective>(std::move(data), context.ttsPlayer, context.isReminder());
    }

    if (data.name == quasar::Directives::LISTEN) {
        VinsRequest::EventSource eventSource;
        eventSource.set_event(TSpeechKitRequestProto_TEventSource_EEvent_Directive);
        eventSource.set_source(TSpeechKitRequestProto_TEventSource_ESource_Software);
        eventSource.set_type(TSpeechKitRequestProto_TEventSource_EType_Voice);
        eventSource.set_id(TString(quasar::makeUUID()));
        return createListenDirective(std::move(data), std::move(eventSource));
    }

    if (data.name == quasar::Directives::START_MUSIC_RECOGNIZER) {
        VinsRequest::EventSource eventSource;
        eventSource.set_event(TSpeechKitRequestProto_TEventSource_EEvent_AutoListen);
        eventSource.set_source(TSpeechKitRequestProto_TEventSource_ESource_Software);
        eventSource.set_type(TSpeechKitRequestProto_TEventSource_EType_Music);
        eventSource.set_id(TString(quasar::makeUUID()));
        return createListenDirective(std::move(data), std::move(eventSource));
    }

    return std::make_shared<YandexIO::Directive>(std::move(data));
}
