#include "audio_client_fail_handler.h"

#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/json_utils/json_utils.h>

using namespace quasar;

AudioClientFailHandler::AudioClientFailHandler(
    YandexIO::DeviceContext& deviceContext,
    std::shared_ptr<YandexIO::ITelemetry> telemetry,
    std::weak_ptr<YandexIO::IFilePlayerCapability> filePlayerCapability)
    : deviceContext_(deviceContext)
    , telemetry_(telemetry)
    , filePlayerCapability_(std::move(filePlayerCapability))
    , lastVoiceInputTime_(std::chrono::steady_clock::now())
{
}

void AudioClientFailHandler::onVoiceInputStarted() {
    lastVoiceInputTime_ = std::chrono::steady_clock::now();
}

void AudioClientFailHandler::onAudioClientFailed(const YandexIO::Directive& directive) {
    YIO_LOG_INFO("Firing error on content channel");
    deviceContext_.fireMediaError(proto::MediaContentType::MUSIC);

    auto now = std::chrono::steady_clock::now();
    bool hasRecentUserInteraction = now - lastVoiceInputTime_ < errorSoundTimeout_;
    bool hasRecentErrorSound = now - lastAudioErrorSound_ < errorSoundDebounce_;
    if (hasRecentUserInteraction && !hasRecentErrorSound) {
        lastAudioErrorSound_ = now;
        if (const auto filePlayerCapability = filePlayerCapability_.lock()) {
            filePlayerCapability->playSoundFile("playback_error.mp3", proto::AudioChannel::DIALOG_CHANNEL);
        }
        Json::Value payload;
        payload["vins_parent_request_id"] = directive.getParentRequestId();
        payload["vins_request_id"] = directive.getRequestId();
        telemetry_->reportEvent("notifyOnAudioPlaybackError", jsonToString(payload));
    }
}
