#include "audio_play_session.h"

#include <yandex_io/services/aliced/capabilities/alice_capability/directives/alice_request_directive.h>
#include <yandex_io/services/aliced/capabilities/alice_capability/vins/vins_utils.h>

#include <yandex_io/libs/base/directives.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/proto_trace.h>
#include <yandex_io/protos/quasar_proto.pb.h>

#include <util/system/yassert.h>

#include <utility>

YIO_DEFINE_LOG_MODULE("audio_play_session");

using namespace quasar;
using namespace YandexIO;

AudioPlaySession::AudioPlaySession(
    YandexIO::IDirectiveProcessorWeakPtr directiveProcessor,
    std::shared_ptr<YandexIO::Directive> directive)
    : directive_(std::move(directive))
    , descriptor_(createDescriptor(directive_))
    , directiveProcessor_(std::move(directiveProcessor))
{
}

bool AudioPlaySession::wasStarted() const {
    return wasStarted_;
}

void AudioPlaySession::setWasStarted(bool value)
{
    wasStarted_ = value;
}

bool AudioPlaySession::wasPaused() const {
    return wasPaused_;
}

void AudioPlaySession::setWasPaused(bool value)
{
    wasPaused_ = value;
}

AudioPlaySession::BackgroundMode AudioPlaySession::getBackgroundMode() const {
    const std::string mode = tryGetString(directive_->getData().payload, "background_mode");

    if (mode == "Pause") {
        return BackgroundMode::PAUSE;
    }

    return BackgroundMode::DUCKING;
}

const proto::AudioPlayerDescriptor& AudioPlaySession::getDescriptor() const {
    return descriptor_;
}

proto::AudioPlayerDescriptor AudioPlaySession::createDescriptor(const std::shared_ptr<YandexIO::Directive>& directive)
{
    proto::AudioPlayerDescriptor result;

    try {
        Json::Value stream = getJson(directive->getData().payload, "stream");
        result.set_stream_id(tryGetString(stream, "id"));
        result.set_player_id(result.stream_id());
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("AudioPlaySession.BadJson.StreamId", "Failed to parse stream.id: " << e.what());
    }

    result.set_type(proto::AudioPlayerDescriptor::AUDIO);
    result.set_audio_channel(proto::AudioChannel::CONTENT_CHANNEL);

    return result;
}

void AudioPlaySession::reportState(
    const YandexIO::IDirectiveProcessorWeakPtr& directiveProcessor, const proto::AudioClientEvent& event)
{
    if (event.has_player_descriptor() &&
        event.player_descriptor().has_type() &&
        event.player_descriptor().type() != proto::AudioPlayerDescriptor::AUDIO) {
        return;
    }

    YIO_LOG_INFO("reportState=" << proto::AudioClientState_Name(event.state())
                                << " playerDescriptorId=" << shortUtf8DebugString(event.player_descriptor()));

    try {
        const std::string payloadKey = audioClientStateToPayloadKey(event.state());

        if (!payloadKey.empty()) {
            const Json::Value context = parseJson(event.audio().context());

            if (context.isMember("callbacks")) {
                const Json::Value callbacks = getJson(context, "callbacks");
                if (callbacks.isMember(payloadKey)) {
                    const Json::Value payload = getJson(callbacks, payloadKey);
                    auto data = YandexIO::Directive::Data::fromJson(payload);
                    data.ignoreAnswer = true; // https://st.yandex-team.ru/SK-5295

                    auto callbackDirective = std::make_shared<AliceRequestDirective>(VinsRequest::createServerActionRequest(data), nullptr, false);

                    if (auto dp = directiveProcessor.lock()) {
                        dp->addDirectives({std::move(callbackDirective)});
                    }
                }
            }
        }
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("AudioPlaySession.FailedReportState", "Failed to send AudioClientEvent : " << e.what());
    }
}

std::string AudioPlaySession::audioClientStateToPayloadKey(proto::AudioClientState value)
{
    switch (value) {
        case proto::AudioClientState::IDLE:
            return "";
        case proto::AudioClientState::BUFFERING:
            return "";
        case proto::AudioClientState::PLAYING:
            return "on_started";
        case proto::AudioClientState::PAUSED:
            return "on_stopped";
        case proto::AudioClientState::STOPPED:
            return "on_stopped";
        case proto::AudioClientState::FINISHED:
            return "on_finished";
        case proto::AudioClientState::FAILED:
            return "on_failed";
    }
}
