#include "audio_player_capability.h"
#include "audio_player_capability.h"
#include <yandex_io/libs/protobuf_utils/debug.h>
#include "media_request_factory.h"

#include <yandex_io/capabilities/device_state/converters/converters.h>
#include <yandex_io/services/aliced/capabilities/alice_capability/directives/alice_request_directive.h>

#include <yandex_io/libs/base/directives.h>
#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/capabilities/device_state/converters/converters.h>

YIO_DEFINE_LOG_MODULE("audio_player_capability");

using namespace quasar;
using namespace YandexIO;

AudioPlayerCapability::AudioPlayerCapability(
    ActivityTracker& activityTracker,
    std::weak_ptr<IDirectiveProcessor> directiveProcessor,
    std::shared_ptr<ipc::IConnector> audioClientConnector,
    DeviceContext& deviceContext,
    std::shared_ptr<IDevice> device,
    std::weak_ptr<YandexIO::IFilePlayerCapability> filePlayerCapability,
    std::shared_ptr<YandexIO::IDeviceStateCapability> deviceState)
    : device_(std::move(device))
    , audioClientConnector_(std::move(audioClientConnector))
    , directiveProcessor_(std::move(directiveProcessor))
    , deviceState_(std::move(deviceState))
    , mediaStats_(device_)
    , activityTracker_(activityTracker)
    , deviceContext_(deviceContext)
    , audioClientFailHandler_(deviceContext, device_->telemetry(), std::move(filePlayerCapability))
{
}

const std::string& AudioPlayerCapability::getHandlerName() const {
    static const std::string s_name = "AudioPlayerCapability";
    return s_name;
}

const std::set<std::string>& AudioPlayerCapability::getSupportedDirectiveNames() const {
    static std::set<std::string> s_names = {
        Directives::AUDIO_PLAY,
        Directives::AUDIO_STOP,
        Directives::AUDIO_REWIND,
        Directives::GLAGOL_METADATA,
        Directives::AUDIO_CLIENT_PREV_TRACK,
        Directives::AUDIO_CLIENT_NEXT_TRACK,
    };

    return s_names;
}

void AudioPlayerCapability::handleDirective(const std::shared_ptr<Directive>& directive)
{
    try {
        if (directive->is(Directives::AUDIO_PLAY)) {
            const auto& session = findOrCreateSession(directive);
            activityTracker_.addActivity(shared_from_this());

            auto command = MediaRequestFactory::createPlayCommand(directive, session->getDescriptor(), false);
            audioClientConnector_->sendMessage(command);
            session->setWasStarted(true);

            YIO_LOG_INFO("handleDirective AUDIO_PLAY streamId=" << command->media_request().play_audio().id());
        } else if (directive->is(Directives::AUDIO_STOP)) {
            proto::AudioPlayerDescriptor descriptor;
            descriptor.set_type(proto::AudioPlayerDescriptor::AUDIO);

            auto command = MediaRequestFactory::createCleanCommand(descriptor);
            audioClientConnector_->sendMessage(command);
        } else if (directive->is(Directives::AUDIO_REWIND)) {
            proto::AudioPlayerDescriptor descriptor;
            descriptor.set_type(proto::AudioPlayerDescriptor::AUDIO);

            auto command = MediaRequestFactory::createRewindCommand(directive, descriptor);
            audioClientConnector_->sendMessage(command);
        } else if (directive->is(Directives::GLAGOL_METADATA)) {
            proto::AudioPlayerDescriptor descriptor;
            descriptor.set_type(proto::AudioPlayerDescriptor::AUDIO);

            auto command = MediaRequestFactory::createMetadataCommand(directive, descriptor);
            audioClientConnector_->sendMessage(command);
        } else if (directive->is(Directives::AUDIO_CLIENT_NEXT_TRACK)) {
            deviceContext_.fireMediaSwitchedForward(proto::MediaContentType::MUSIC);

            const bool setPause = tryGetBool(directive->getData().payload, "setPause", false);
            auto request = VinsRequest::createEventRequest(
                VinsRequest::buildNextTrackSemanticFrame(setPause),
                VinsRequest::createSoftwareDirectiveEventSource());
            request->setIsSilent(directive->getData().isLedSilent);
            auto directive = std::make_shared<AliceRequestDirective>(std::move(request), nullptr, true);
            if (auto directiveProcessor = directiveProcessor_.lock()) {
                directiveProcessor->addDirectives({std::move(directive)});
            }
        } else if (directive->is(Directives::AUDIO_CLIENT_PREV_TRACK)) {
            deviceContext_.fireMediaSwitchedBackward(proto::MediaContentType::MUSIC);

            const bool setPause = tryGetBool(directive->getData().payload, "setPause", false);
            auto request = VinsRequest::createEventRequest(
                VinsRequest::buildPrevTrackSemanticFrame(setPause),
                VinsRequest::createSoftwareDirectiveEventSource());
            request->setIsSilent(directive->getData().isLedSilent);
            auto directive = std::make_shared<AliceRequestDirective>(std::move(request), nullptr, true);
            if (auto directiveProcessor = directiveProcessor_.lock()) {
                directiveProcessor->addDirectives({std::move(directive)});
            }
        }
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("AudioPlayerCapability.FailedHandleDirective",
                            "Failed to execute '" << directive->getData().name << "' directive: " << e.what());
    }
}

void AudioPlayerCapability::cancelDirective(const std::shared_ptr<Directive>& directive)
{
    completeDirective(directive);
}

void AudioPlayerCapability::completeDirective(const std::shared_ptr<Directive>& directive)
{
    const auto& session = findSession(directive);

    if (session != nullptr && session->wasStarted()) {
        try {
            activityTracker_.removeActivity(shared_from_this());

            auto command = MediaRequestFactory::createCleanCommand(session->getDescriptor());
            audioClientConnector_->sendMessage(command);
        } catch (const std::exception& e) {
            YIO_LOG_ERROR_EVENT("AudioPlayerCapability.FailedCancelDirective", e.what());
        }
    }

    sessionByDirective_.erase(directive);
}

void AudioPlayerCapability::prefetchDirective(const std::shared_ptr<Directive>& directive)
{
    try {
        if (directive->is(Directives::AUDIO_PLAY)) {
            const auto& session = findOrCreateSession(directive);

            auto command = MediaRequestFactory::createPlayCommand(directive, session->getDescriptor(), true);
            audioClientConnector_->sendMessage(command);

            YIO_LOG_INFO("prefetchDirective AUDIO_PLAY streamId=" << command->media_request().play_audio().id());
        }
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("AudioPlayerCapability.FailedPrefetchDirective", e.what());
    }
}

void AudioPlayerCapability::restoreDirective(const std::shared_ptr<YandexIO::Directive>& directive)
{
    try {
        if (directive->is(Directives::AUDIO_PLAY)) {
            const auto& session = findOrCreateSession(directive);
            activityTracker_.addActivity(shared_from_this());
            session->setWasStarted(true);

            YIO_LOG_INFO("restoreDirective AUDIO_PLAY streamId=" << session->getDescriptor().stream_id());
        }
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("AudioPlayerCapability.FailedRestoreDirective", e.what());
    }
}

std::string AudioPlayerCapability::activityName() const {
    return "AudioPlayerCapability";
}

void AudioPlayerCapability::setBackground()
{
    if (const auto& session = findActiveSession()) {
        if (session->getBackgroundMode() == AudioPlaySession::BackgroundMode::PAUSE) {
            auto command = MediaRequestFactory::createPauseCommand(session->getDescriptor());
            audioClientConnector_->sendMessage(command);

            session->setWasPaused(true);
        }
    }
}

void AudioPlayerCapability::setForeground()
{
    if (const auto& session = findActiveSession()) {
        if (session->getBackgroundMode() == AudioPlaySession::BackgroundMode::PAUSE && session->wasPaused()) {
            auto command = MediaRequestFactory::createResumeCommand(session->getDescriptor());
            audioClientConnector_->sendMessage(command);

            session->setWasPaused(false);
        }
    }
}

quasar::proto::AudioChannel AudioPlayerCapability::getAudioChannel() const {
    return proto::CONTENT_CHANNEL;
}

bool AudioPlayerCapability::isLocal() const {
    return true;
}

void AudioPlayerCapability::onAliceStateChanged(quasar::proto::AliceState state)
{
    if (state.has_state() && state.state() == quasar::proto::AliceState::LISTENING) {
        onVoiceInputStarted(state.request_id());
    }
}

void AudioPlayerCapability::onAliceTtsCompleted()
{
    mediaStats_.onTtsCompleted();
}

void AudioPlayerCapability::onVoiceInputStarted(const std::string& vinsRequestId)
{
    mediaStats_.onVoiceInputStarted(vinsRequestId);
    audioClientFailHandler_.onVoiceInputStarted();
}

void AudioPlayerCapability::onQuasarMessage(const ipc::SharedMessage& sharedMessage)
{
    if (sharedMessage->has_audio_client_event()) {
        const auto& event = sharedMessage->audio_client_event();

        if (!event.has_player_descriptor()) {
            return;
        }
        const auto& playerDescriptor = event.player_descriptor();
        if (!playerDescriptor.has_type() || playerDescriptor.type() != proto::AudioPlayerDescriptor::AUDIO) {
            return;
        }

        mediaStats_.onAudioClientEvent(event);
        deviceContext_.fireAudioClientEvent(event);
        deviceState_->setAudioPlayerState(YandexIO::convertAudioPlayerState(event));

        if (event.event() == proto::AudioClientEvent::STATE_CHANGED) {
            AudioPlaySession::reportState(directiveProcessor_, event);

            auto sessionByDirectiveCopy = sessionByDirective_;
            for (const auto& [directive, session] : sessionByDirectiveCopy) {
                if (!playerDescriptor.has_player_id() || playerDescriptor.player_id() != session->getDescriptor().player_id()) {
                    continue;
                }

                onAudioClientStateChanged(directive, event);
            }
        }
    } else if (sharedMessage->has_legacy_player_state_changed()) {
        mediaStats_.onLegacyPlayerStateChanged(sharedMessage->legacy_player_state_changed());
    }
}

std::shared_ptr<AudioPlaySession> AudioPlayerCapability::findSession(const std::shared_ptr<Directive>& directive)
{
    auto iter = sessionByDirective_.find(directive);
    if (iter == sessionByDirective_.end()) {
        return nullptr;
    } else {
        return iter->second;
    }
}

std::shared_ptr<AudioPlaySession> AudioPlayerCapability::findActiveSession() const {
    for (const auto& [directive, session] : sessionByDirective_) {
        if (session->wasStarted()) {
            return session;
        }
    }

    return nullptr;
}

std::shared_ptr<AudioPlaySession> AudioPlayerCapability::findOrCreateSession(const std::shared_ptr<Directive>& directive)
{
    auto iter = sessionByDirective_.find(directive);
    if (iter != sessionByDirective_.end()) {
        return iter->second;
    } else {
        iter = sessionByDirective_.insert({directive,
                                           std::make_shared<AudioPlaySession>(directiveProcessor_, directive)})
                   .first;
        return iter->second;
    }
}

void AudioPlayerCapability::onAudioClientStateChanged(
    const std::shared_ptr<Directive>& directive,
    const proto::AudioClientEvent& audioClientEvent)
{
    Y_VERIFY(audioClientEvent.has_player_descriptor());
    Y_VERIFY(audioClientEvent.event() == proto::AudioClientEvent::STATE_CHANGED);
    Y_VERIFY(audioClientEvent.player_descriptor().has_type());
    Y_VERIFY(audioClientEvent.player_descriptor().type() == proto::AudioPlayerDescriptor::AUDIO);

    if (auto directiveProcessor = directiveProcessor_.lock()) {
        switch (audioClientEvent.state()) {
            case proto::AudioClientState::PLAYING:
                if (directive->isBlocksSubsequentPrefetch()) {
                    directive->setBlocksSubsequentPrefetch(false);
                    directiveProcessor->onBlockPrefetchChanged(directive);
                }

                directiveProcessor->onHandleDirectiveStarted(directive);
                break;
            case proto::AudioClientState::FINISHED:
            case proto::AudioClientState::STOPPED:
                completeDirective(directive);
                directiveProcessor->onHandleDirectiveCompleted(directive, true);
                break;
            case proto::AudioClientState::FAILED:
                completeDirective(directive);
                directiveProcessor->onHandleDirectiveCompleted(directive, false);

                audioClientFailHandler_.onAudioClientFailed(*directive);
                break;
            default: {
                // do nothing
            }
        }
    }
}
