#include "audio_player_preprocessor.h"

#include <yandex_io/libs/base/directives.h>

using namespace quasar;

const std::string& AudioPlayerPreprocessor::getPreprocessorName() const {
    static const std::string s_name = "AudioPlayerPreprocessor";
    return s_name;
}

void AudioPlayerPreprocessor::preprocessDirectives(std::list<std::shared_ptr<YandexIO::Directive>>& directives)
{
    markGetNextDirectives(directives);
    injectStopOppositePlayerDirectives(directives);
    markAudioPlayDirectivesAsPrefetchBlockers(directives);
}

void AudioPlayerPreprocessor::markGetNextDirectives(std::list<std::shared_ptr<YandexIO::Directive>>& directives)
{
    std::shared_ptr<YandexIO::Directive> prevDirective;
    for (auto& nextDirective : directives) {
        if (prevDirective != nullptr && nextDirective->getData().isServerAction()) {
            if (prevDirective->is(Directives::AUDIO_PLAY)) {
                nextDirective->setIsGetNext();
            }
        }

        prevDirective = nextDirective;
    }
}

void AudioPlayerPreprocessor::injectStopOppositePlayerDirectives(std::list<std::shared_ptr<YandexIO::Directive>>& directives)
{
    for (auto iter = directives.begin(); iter != directives.end(); ++iter) {
        auto& directive = *iter;

        if (directive->is(Directives::AUDIO_PLAY)) {
            iter = directives.insert(iter, YandexIO::Directive::createClientAction(Directives::STOP_LEGACY_PLAYER));
            std::advance(iter, 1);
        } else if (isLegacyPlayerDirective(directive)) {
            iter = directives.insert(iter, YandexIO::Directive::createLocalAction(Directives::CLEAR_QUEUE));
            std::advance(iter, 1);
        }
    }
}

void AudioPlayerPreprocessor::markAudioPlayDirectivesAsPrefetchBlockers(std::list<std::shared_ptr<YandexIO::Directive>>& directives)
{
    for (auto directive : directives) {
        if (directive->is(Directives::AUDIO_PLAY)) {
            directive->setBlocksSubsequentPrefetch(true);
        }
    }
}

bool AudioPlayerPreprocessor::isLegacyPlayerDirective(const std::shared_ptr<YandexIO::Directive>& directive) {
    return directive->is(quasar::Directives::MUSIC_PLAY) ||
           directive->is(quasar::Directives::RADIO_PLAY) ||
           directive->is(quasar::Directives::VIDEO_PLAY) ||
           directive->is(quasar::Directives::PLAYER_CONTINUE) ||
           directive->is(quasar::Directives::PLAYER_NEXT_TRACK) ||
           directive->is(quasar::Directives::PLAYER_PREVIOUS_TRACK) ||
           directive->is(quasar::Directives::PLAYER_REPLAY) ||
           directive->is(quasar::Directives::PLAYER_DISLIKE) ||
           directive->is(quasar::Directives::GO_HOME);
}
