#include "media_request_factory.h"

#include <yandex_io/libs/logging/logging.h>

YIO_DEFINE_LOG_MODULE("media_request_factory");

using namespace quasar;

ipc::SharedMessage MediaRequestFactory::createPlayCommand(
    const std::shared_ptr<YandexIO::Directive>& directive,
    const proto::AudioPlayerDescriptor& descriptor,
    bool isPrefetch)
{
    return ipc::buildMessage([&](auto& msg) {
        const auto& data = directive->getData();

        auto request = msg.mutable_media_request();
        auto playAudio = request->mutable_play_audio();

        request->set_is_prefetch(isPrefetch);
        request->mutable_player_descriptor()->CopyFrom(descriptor);

        Json::Value stream = getJson(data.payload, "stream");
        playAudio->set_id(tryGetString(stream, "id"));
        playAudio->set_type(tryGetString(stream, "type", "track"));
        playAudio->set_url(tryGetString(stream, "url"));
        const auto offsetMs = tryGetInt(stream, "offset_ms", 0);
        if (offsetMs > 0) {
            playAudio->set_initial_offset_ms(offsetMs);
        }
        playAudio->set_context(jsonToString(data.payload));
        playAudio->set_format(getFormat(stream));
        playAudio->set_provider_name(tryGetString(data.payload, "provider_name"));
        playAudio->set_screen_type(getScreenType(data.payload));
        playAudio->set_set_pause(tryGetBool(data.payload, "set_pause", false));
        playAudio->set_multiroom_token(tryGetString(data.payload, "multiroom_token"));
        if (stream.isMember("normalization")) {
            const auto& normalization = stream["normalization"];
            static const std::string truePeak = "true_peak";
            static const std::string loudness = "integrated_loudness";
            if (checkHasDoubleField(normalization, truePeak) &&
                checkHasDoubleField(normalization, loudness)) {
                auto normalizationMsg = playAudio->mutable_normalization();
                normalizationMsg->set_true_peak(normalization[truePeak].asDouble());
                normalizationMsg->set_integrated_loudness(normalization[loudness].asDouble());
            }
        }

        auto analyticsContext = playAudio->mutable_analytics_context();
        analyticsContext->set_vins_request_id(TString(data.requestId));
        analyticsContext->set_vins_parent_request_id(TString(data.parentRequestId));
        analyticsContext->set_name(TString(data.analyticsContextName));

        const auto metadata = tryGetJson(data.payload, "metadata", Json::Value());
        auto audioMeta = playAudio->mutable_metadata();
        audioMeta->set_title(tryGetString(metadata, "title"));
        audioMeta->set_subtitle(tryGetString(metadata, "subtitle"));
        audioMeta->set_art_image_url(tryGetString(metadata, "art_image_url"));
        audioMeta->set_hide_progress_bar(tryGetBool(metadata, "hide_progress_bar", false));

        const auto jsonGlagolMeta = tryGetJson(metadata, "glagol_metadata", Json::Value());
        const auto jsonMusicMeta = tryGetJson(jsonGlagolMeta, "music_metadata", Json::Value());
        auto musicMeta = audioMeta->mutable_music_metadata();
        musicMeta->set_id(tryGetString(jsonMusicMeta, "id"));
        musicMeta->set_type(tryGetString(jsonMusicMeta, "type"));
        musicMeta->set_description(tryGetString(jsonMusicMeta, "description"));
        const auto jsonPrev = tryGetJson(jsonMusicMeta, "prev_track_info", Json::Value());
        musicMeta->mutable_prev()->set_id(tryGetString(jsonPrev, "id"));
        musicMeta->mutable_prev()->set_type(tryGetString(jsonPrev, "type"));
        const auto jsonNext = tryGetJson(jsonMusicMeta, "next_track_info", Json::Value());
        musicMeta->mutable_next()->set_id(tryGetString(jsonNext, "id"));
        musicMeta->mutable_next()->set_type(tryGetString(jsonNext, "type"));
        if (jsonMusicMeta.isMember("shuffled")) {
            musicMeta->set_shuffled(jsonMusicMeta["shuffled"].asBool());
        }
        if (jsonMusicMeta.isMember("repeat_mode")) {
            musicMeta->set_repeat_mode(jsonMusicMeta["repeat_mode"].asString());
        }
    });
}

ipc::SharedMessage MediaRequestFactory::createCleanCommand(const proto::AudioPlayerDescriptor& descriptor)
{
    return ipc::buildMessage([&](auto& msg) {
        msg.mutable_media_request()->mutable_clean_players();
        msg.mutable_media_request()->mutable_player_descriptor()->CopyFrom(descriptor);
    });
}

ipc::SharedMessage MediaRequestFactory::createRewindCommand(
    const std::shared_ptr<YandexIO::Directive>& directive,
    const proto::AudioPlayerDescriptor& descriptor)
{
    return ipc::buildMessage([&](auto& msg) {
        auto request = msg.mutable_media_request();
        request->mutable_player_descriptor()->CopyFrom(descriptor);

        auto rewind = request->mutable_rewind();
        const auto& data = directive->getData();
        if (data.payload.isMember("amount_ms")) {
            rewind->set_amount(tryGetInt64(data.payload, "amount_ms", 0) / 1000.0);
        } else {
            rewind->set_amount(tryGetInt64(data.payload, "amount", 0));
        }
        rewind->set_type(jsonValueToRewindType(tryGetString(data.payload, "type")));
    });
}

ipc::SharedMessage MediaRequestFactory::createPauseCommand(const proto::AudioPlayerDescriptor& descriptor)
{
    return ipc::buildMessage([&](auto& msg) {
        msg.mutable_media_request()->mutable_player_descriptor()->CopyFrom(descriptor);
        msg.mutable_media_request()->mutable_pause()->set_smooth(false);
    });
}

ipc::SharedMessage MediaRequestFactory::createResumeCommand(const proto::AudioPlayerDescriptor& descriptor)
{
    return ipc::buildMessage([&](auto& msg) {
        msg.mutable_media_request()->mutable_player_descriptor()->CopyFrom(descriptor);
        msg.mutable_media_request()->set_resume(proto::MediaRequest::AUDIO);
    });
}

ipc::SharedMessage MediaRequestFactory::createMetadataCommand(
    const std::shared_ptr<YandexIO::Directive>& directive,
    const proto::AudioPlayerDescriptor& descriptor)
{
    return ipc::buildMessage([&](auto& msg) {
        msg.mutable_media_request()->mutable_player_descriptor()->CopyFrom(descriptor);

        const auto& payload = directive->getData().payload;
        const auto jsonGlagolMeta = tryGetJson(payload, "glagol_metadata", Json::Value());
        const auto jsonMusicMeta = tryGetJson(jsonGlagolMeta, "music_metadata", Json::Value());

        auto metadata = msg.mutable_media_request()->mutable_metadata()->mutable_music_metadata();
        metadata->set_id(tryGetString(jsonMusicMeta, "id"));
        metadata->set_type(tryGetString(jsonMusicMeta, "type"));
        metadata->set_description(tryGetString(jsonMusicMeta, "description"));
        const auto jsonPrev = tryGetJson(jsonMusicMeta, "prev_track_info", Json::Value());
        metadata->mutable_prev()->set_id(tryGetString(jsonPrev, "id"));
        metadata->mutable_prev()->set_type(tryGetString(jsonPrev, "type"));
        const auto jsonNext = tryGetJson(jsonMusicMeta, "next_track_info", Json::Value());
        metadata->mutable_next()->set_id(tryGetString(jsonNext, "id"));
        metadata->mutable_next()->set_type(tryGetString(jsonNext, "type"));
        if (jsonMusicMeta.isMember("shuffled")) {
            metadata->set_shuffled(jsonMusicMeta["shuffled"].asBool());
        }
        if (jsonMusicMeta.isMember("repeat_mode")) {
            metadata->set_repeat_mode(jsonMusicMeta["repeat_mode"].asString());
        }
    });
}

proto::MediaRequest::RewindType MediaRequestFactory::jsonValueToRewindType(const std::string& value)
{
    if (value == "Forward") {
        return proto::MediaRequest::FORWARD;
    } else if (value == "Backward") {
        return proto::MediaRequest::BACKWARD;
    } else if (value == "Absolute" || value == "absolute") {
        return proto::MediaRequest::ABSOLUTE;
    } else {
        throw std::invalid_argument(value);
    }
}

proto::Audio::Screen MediaRequestFactory::getScreenType(const Json::Value& payload)
{
    const std::string type = tryGetString(payload, "screen_type");

    if (type == "Default") {
        return proto::Audio::Screen::Audio_Screen_DEFAULT;
    } else if (type == "Music") {
        return proto::Audio::Screen::Audio_Screen_MUSIC;
    } else if (type == "Radio") {
        return proto::Audio::Screen::Audio_Screen_RADIO;
    }

    YIO_LOG_ERROR_EVENT("MediaRequestFactory.InvalidAudioScreenType", "Unsupported screen type received: " << type);
    return proto::Audio::Screen::Audio_Screen_DEFAULT;
}

proto::Audio::Format MediaRequestFactory::getFormat(const Json::Value& stream)
{
    const std::string format = tryGetString(stream, "format");

    if (format == "Unknown") {
        return proto::Audio::UNKNOWN;
    } else if (format == "MP3") {
        return proto::Audio::MP3;
    } else if (format == "HLS") {
        return proto::Audio::HLS;
    }

    YIO_LOG_WARN("Unsupported audio format received: " << format);
    return proto::Audio::UNKNOWN;
}
