#include "media_stats.h"

#include <yandex_io/libs/ete_metrics/ete_util.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

YIO_DEFINE_LOG_MODULE("media_stats");

using namespace quasar;

namespace {

    constexpr const std::chrono::milliseconds MAX_METRIC_VALUE{30000};

    std::string getVinsRequestId(const proto::AudioClientEvent& event) {
        return event.audio().analytics_context().vins_request_id();
    }

    std::string getVinsRequestId(const proto::LegacyPlayerStateChanged& event) {
        return event.vins_request_id();
    }

} // anonymous namespace

MediaStats::MediaStats(std::shared_ptr<YandexIO::IDevice> device)
    : device_(std::move(device))
{
}

void MediaStats::onVoiceInputStarted(const std::string& vinsRequestId) {
    if (vinsRequestId != metrics_.vinsRequestId) {
        metrics_ = Metrics();
        metrics_.voiceInputStarted = Clock::now();
        metrics_.vinsRequestId = vinsRequestId;
    }
}

void MediaStats::onTtsCompleted() {
    metrics_.ttsCompleted = Clock::now();
}

void MediaStats::onAudioClientEvent(const proto::AudioClientEvent& event) {
    Y_VERIFY(event.has_player_descriptor());
    Y_VERIFY(event.player_descriptor().has_type());
    Y_VERIFY(event.player_descriptor().type() == proto::AudioPlayerDescriptor::AUDIO);

    if (!event.has_event() || event.event() != proto::AudioClientEvent::STATE_CHANGED) {
        return;
    }
    if (!event.has_state()) {
        return;
    }
    if (metrics_.playerType == PlayerType::MUSIC_PLAYER) {
        // Don't send metrics if players have switched
        metrics_ = Metrics();
        return;
    }

    metrics_.playerType = PlayerType::AUDIO_CLIENT;
    if (event.state() == proto::AudioClientState::PLAYING) {
        onPlayerStarted(getVinsRequestId(event));
    } else if (event.state() == proto::AudioClientState::FINISHED) {
        metrics_.prevTrackFinished = Clock::now();
    }
}

void MediaStats::onLegacyPlayerStateChanged(const proto::LegacyPlayerStateChanged& event) {
    if (!event.has_player_type() || event.player_type() != proto::LegacyPlayerStateChanged::YANDEX_MUSIC) {
        return;
    }
    if (!event.has_state()) {
        return;
    }
    if (metrics_.playerType == PlayerType::AUDIO_CLIENT) {
        // Don't send metrics if players have switched
        metrics_ = Metrics();
        return;
    }

    metrics_.playerType = PlayerType::MUSIC_PLAYER;
    if (event.state() == proto::LegacyPlayerStateChanged::STARTED) {
        onPlayerStarted(getVinsRequestId(event));
    }
}

void MediaStats::onPlayerStarted(const std::string& vinsRequestId) {
    if (metrics_.vinsRequestId == vinsRequestId && metrics_.voiceInputStarted.has_value()) {
        sendMetric(metrics_.voiceInputStarted.value(), "StartedFromVoiceInput");
    }
    if (metrics_.vinsRequestId == vinsRequestId && metrics_.ttsCompleted.has_value()) {
        sendMetric(metrics_.ttsCompleted.value(), "StartedFromTtsEnd");
    }
    if (metrics_.prevTrackFinished.has_value()) {
        metrics_.vinsRequestId = vinsRequestId;
        sendMetric(metrics_.prevTrackFinished.value(), "StartedFromPrev");
    }

    metrics_ = Metrics();
}

void MediaStats::sendMetric(const TimePoint& begin, const std::string& name) const {
    const auto diff = std::chrono::duration_cast<std::chrono::milliseconds>(Clock::now() - begin);

    auto json = makeETEEvent(metrics_.vinsRequestId);
    json["value"] = static_cast<int64_t>(diff.count());
    json["vinsRequestId"] = metrics_.vinsRequestId;

    if (metrics_.vinsRequestId.empty()) {
        YIO_LOG_ERROR_EVENT("MediaStats.DropMetric.EmptyVinsRequestId", "Drop metric " << formatMetricName(name) << ": " << jsonToString(json));
        return;
    }
    if (metrics_.playerType == PlayerType::UNKNOWN_PLAYER) {
        YIO_LOG_ERROR_EVENT("MediaStats.DropMetric.UnknownPlayer", "Drop metric " << formatMetricName(name) << ": " << jsonToString(json));
        return;
    }
    if (diff > MAX_METRIC_VALUE) {
        YIO_LOG_ERROR_EVENT("MediaStats.DropMetric.LargeMetricValue", "Drop metric " << formatMetricName(name) << ": " << jsonToString(json));
        return;
    }

    device_->telemetry()->reportEvent(formatMetricName(name), jsonToString(json));
}

std::string MediaStats::formatMetricName(const std::string& name) const {
    switch (metrics_.playerType) {
        case MediaStats::PlayerType::UNKNOWN_PLAYER:
            return "unknownPlayer" + name;
        case MediaStats::PlayerType::AUDIO_CLIENT:
            return "audioClient" + name;
        case MediaStats::PlayerType::MUSIC_PLAYER:
            return "musicPlayer" + name;
    }
}
