#include "bluetooth_capability.h"

#include <yandex_io/libs/base/directives.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/proto_trace.h>
#include <yandex_io/libs/protobuf_utils/json.h>
#include <yandex_io/sdk/interfaces/directive.h>

using namespace YandexIO;
using namespace quasar;

YIO_DEFINE_LOG_MODULE("bluetooth")

BluetoothCapability::BluetoothCapability(std::shared_ptr<quasar::ICallbackQueue> worker,
                                         std::shared_ptr<IDeviceStateCapability> deviceState,
                                         std::shared_ptr<quasar::ipc::IIpcFactory> ipcFactory,
                                         YandexIO::ActivityTracker& activityTracker,
                                         YandexIO::IDirectiveProcessorWeakPtr directiveProcessor,
                                         std::shared_ptr<quasar::ipc::IConnector> toInterface,
                                         std::shared_ptr<quasar::ipc::IConnector> toMedia)
    : worker_(std::move(worker))
    , deviceState_(std::move(deviceState))
    , deviceContext_(ipcFactory)
    , activityTracker_(activityTracker)
    , directiveProcessor_(std::move(directiveProcessor))
    , toInterface_(toInterface)
    , toMedia_(std::move(toMedia))
{
    deviceContext_.onBluetoothSinkMediaEvent = [this](const proto::BluetoothSinkEvent& sinkEvent) {
        worker_->add([this, sinkEvent]() {
            providePlayerState(sinkEvent);
        });
    };
}

const std::string& BluetoothCapability::getHandlerName() const {
    static const std::string CAPABILITY_NAME = "BluetoothCapability";
    return CAPABILITY_NAME;
}

const std::set<std::string>& BluetoothCapability::getSupportedDirectiveNames() const {
    static const std::set<std::string> SUPPORTED_DIRECTIVES = {
        Directives::START_BLUETOOTH,
        Directives::STOP_BLUETOOTH,
        Directives::BLUETOOTH_PLAYER_PLAY,
        Directives::BLUETOOTH_PLAYER_PAUSE,
        Directives::BLUETOOTH_PLAYER_NEXT,
        Directives::BLUETOOTH_PLAYER_PREV};
    return SUPPORTED_DIRECTIVES;
}

void BluetoothCapability::handleDirective(const std::shared_ptr<Directive>& directive) {
    if (directive->is(Directives::BLUETOOTH_PLAYER_PLAY)) {
        resume();
    } else if (directive->is(Directives::BLUETOOTH_PLAYER_PAUSE)) {
        pause();
        sendPauseToInterface();
    } else if (directive->is(Directives::BLUETOOTH_PLAYER_NEXT)) {
        next();
        sendNextToInterface();
    } else if (directive->is(Directives::BLUETOOTH_PLAYER_PREV)) {
        const bool forced = quasar::tryGetBool(directive->getData().payload, "forced", true);
        prev(forced);
        sendPrevToInterface();
    }
}

void BluetoothCapability::cancelDirective(const std::shared_ptr<Directive>& /*directive*/) {
    // ¯\_(ツ)_/¯
}

void BluetoothCapability::prefetchDirective(const std::shared_ptr<Directive>& /*directive*/) {
    // ¯\_(ツ)_/¯
}

void BluetoothCapability::resume() {
    deviceContext_.fireBtAvrcpResume();
}

void BluetoothCapability::pause() {
    deviceContext_.fireBtAvrcpPause();
}

void BluetoothCapability::next() {
    deviceContext_.fireBtAvrcpNext();
    deviceContext_.fireMediaSwitchedForward(proto::MediaContentType::MUSIC);
}

void BluetoothCapability::prev(bool forced) {
    deviceContext_.fireBtAvrcpPrev(forced);
    deviceContext_.fireMediaSwitchedBackward(proto::MediaContentType::MUSIC);
}

std::string BluetoothCapability::activityName() const {
    return "BluetoothCapability";
}

void BluetoothCapability::setBackground() {
    if (isPlaying_) {
        deviceContext_.fireBtFreeAudioFocus();
    }
}

void BluetoothCapability::setForeground() {
    if (isPlaying_) {
        deviceContext_.fireBtTakeAudioFocus();
    }
}

quasar::proto::AudioChannel BluetoothCapability::getAudioChannel() const {
    return quasar::proto::CONTENT_CHANNEL;
}

bool BluetoothCapability::isLocal() const {
    return true;
}

void BluetoothCapability::updateMetaInfo(const quasar::proto::BluetoothSinkEvent& sinkEvent) {
    if (sinkEvent.has_bluetooth_track_meta_info()) {
        YIO_LOG_DEBUG("Bluetooth meta info received");
        const auto& meta = sinkEvent.bluetooth_track_meta_info();
        Json::Value trackInfo = Json::nullValue;
        if (meta.has_title()) {
            trackInfo["title"] = meta.title();
        }
        if (meta.has_artist()) {
            Json::Value name;
            name["name"] = meta.artist();
            trackInfo["artists"] = Json::arrayValue;
            trackInfo["artists"].append(name);
        }
        if (meta.has_album()) {
            Json::Value title;
            title["title"] = meta.album();
            trackInfo["albums"] = Json::arrayValue;
            trackInfo["albums"].append(title);
        }
        trackInfo["id"] = "stub";
        metaInfo_ = sinkEvent.bluetooth_track_meta_info();
        auto track = quasar::convertJsonToProtobuf<google::protobuf::Struct>(quasar::jsonToString(trackInfo));
        if (!track.has_value()) {
            state_.MutableCurrentlyPlaying()->ClearRawTrackInfo();
            YIO_LOG_ERROR_EVENT("Bluetooth.FailedToParseMeta", "Failed to convert meta to proto struct");
        } else {
            state_.MutableCurrentlyPlaying()->MutableRawTrackInfo()->Swap(&(*track));
        }
    }
}

void BluetoothCapability::updatePlayingState(const quasar::proto::BluetoothSinkEvent& sinkEvent) {
    if (sinkEvent.has_audio_event()) {
        if (sinkEvent.audio_event() == proto::BluetoothSinkEvent_BtSinkAudioEvent_PLAYING) {
            YIO_LOG_INFO("Bluetooth player is playing");
            state_.SetLastPlayTimestamp(getNowTimestampMs());
            state_.MutablePlayer()->ClearTimestamp();
            if (!std::exchange(isPlaying_, true)) {
                deviceContext_.fireMediaResumed(proto::MediaContentType::MUSIC);
            }
            activityTracker_.addActivity(shared_from_this());

            auto pauseDirective = std::invoke([]() {
                Json::Value payload = Json::objectValue;
                payload["smooth"] = true;
                Directive::Data data(Directives::PLAYER_PAUSE, "local_action");
                data.payload = std::move(payload);
                return std::make_shared<Directive>(std::move(data));
            });

            auto audioClientStop = std::invoke([]() {
                Directive::Data data(Directives::AUDIO_STOP, "local_action");
                return std::make_shared<Directive>(std::move(data));
            });

            if (const auto dp = directiveProcessor_.lock()) {
                dp->addDirectives({std::move(pauseDirective), std::move(audioClientStop)});
            }
        } else {
            YIO_LOG_INFO("Bluetooth player stops");
            const bool previousIsPlaying = std::exchange(isPlaying_, false);
            activityTracker_.removeActivity(shared_from_this());
            if (previousIsPlaying) {
                state_.MutablePlayer()->SetTimestamp(getNowTimestampMs() / 1000);
                deviceContext_.fireMediaPaused(proto::MediaContentType::MUSIC);
            }
        }
        state_.MutablePlayer()->SetPause(!isPlaying_);
    }
}

void BluetoothCapability::updateConnections(const quasar::proto::BluetoothSinkEvent& sinkEvent) {
    if (sinkEvent.has_connection_event()) {
        const auto& connectionEvent = sinkEvent.connection_event();
        if (connectionEvent.connection_event() == proto::BtConnection::CONNECTED && connectionEvent.has_network()) {
            YIO_LOG_INFO("Source connected: " << connectionEvent.network().name());
            btNetworks_.insert(connectionEvent.network().name());
        } else if (connectionEvent.connection_event() == proto::BtConnection::DISCONNECTED && connectionEvent.has_network()) {
            YIO_LOG_INFO("Source disconnected: " << connectionEvent.network().name());
            btNetworks_.erase(connectionEvent.network().name());
        }
        state_.ClearCurrentConnections();
        for (const auto& btNetowrk : btNetworks_) {
            state_.AddCurrentConnections()->SetName(btNetowrk);
        }
    }
}

void BluetoothCapability::providePlayerState(const proto::BluetoothSinkEvent& sinkEvent) {
    updatePlayingState(sinkEvent);
    updateConnections(sinkEvent);
    updateMetaInfo(sinkEvent);

    deviceState_->setBluetoothState(state_);

    // TODO(slavashel): Make mediad listening TDeviceStateCapability / migrate mediad to TDeviceState
    toMedia_->sendMessage(quasar::ipc::buildMessage([&](auto& msg) {
        auto* playerState = msg.mutable_provide_bluetooth_player_state();
        if (state_.HasLastPlayTimestamp()) {
            playerState->set_last_play_timestamp_ms(state_.GetLastPlayTimestamp());
        }
        if (state_.GetPlayer().HasTimestamp()) {
            playerState->set_timestamp_ms(state_.GetPlayer().GetTimestamp());
        }
        playerState->set_is_paused(state_.GetPlayer().GetPause());
        playerState->mutable_track_meta_info()->CopyFrom(metaInfo_);
    }));
}

void BluetoothCapability::sendPauseToInterface() {
    if (toInterface_ != nullptr) {
        proto::QuasarMessage message;
        message.mutable_media_request()->mutable_pause();
        toInterface_->sendMessage(std::move(message));
    }
}

void BluetoothCapability::sendPrevToInterface() {
    if (toInterface_ != nullptr) {
        proto::QuasarMessage message;
        message.mutable_media_request()->mutable_previous();
        toInterface_->sendMessage(std::move(message));
    }
}

void BluetoothCapability::sendNextToInterface() {
    if (toInterface_ != nullptr) {
        proto::QuasarMessage message;
        message.mutable_media_request()->mutable_next()->set_set_pause(false);
        toInterface_->sendMessage(std::move(message));
    }
}
