#include "bluetooth_capability_preprocessor.h"

#include <yandex_io/libs/base/directives.h>
#include <yandex_io/libs/json_utils/json_utils.h>

using namespace YandexIO;

BluetoothCapabilityPreprocessor::BluetoothCapabilityPreprocessor(quasar::AliceDeviceState& deviceState)
    : deviceState_(deviceState)
{
}

const std::string& BluetoothCapabilityPreprocessor::getPreprocessorName() const {
    static const std::string s_name = "BluetoothCapabilityPreprocessor";
    return s_name;
}

void BluetoothCapabilityPreprocessor::preprocessDirectives(std::list<std::shared_ptr<Directive>>& directives)
{
    convertToBluetoothDirective(directives);
    injectBluetoothPauseToPlayCommands(directives);
}

void BluetoothCapabilityPreprocessor::convertToBluetoothDirective(std::list<std::shared_ptr<Directive>>& directives) {
    for (auto& directive : directives) {
        const bool isBtPlayer = quasar::tryGetString(directive->getData().payload, "player", "") == "bluetooth";
        if (isBtPlayer && directive->is(quasar::Directives::PLAYER_CONTINUE)) {
            Directive::Data data = directive->getData();
            data.name = quasar::Directives::BLUETOOTH_PLAYER_PLAY;
            directive = std::make_shared<Directive>(std::move(data));
        } else if (isBtPlayer && directive->is(quasar::Directives::PLAYER_NEXT_TRACK)) {
            Directive::Data data = directive->getData();
            data.name = quasar::Directives::BLUETOOTH_PLAYER_NEXT;
            directive = std::make_shared<Directive>(std::move(data));
        } else if (isBtPlayer && directive->is(quasar::Directives::PLAYER_PREVIOUS_TRACK)) {
            Directive::Data data = directive->getData();
            data.name = quasar::Directives::BLUETOOTH_PLAYER_PREV;
            directive = std::make_shared<Directive>(std::move(data));
        } else if (directive->is(quasar::Directives::PLAYER_PAUSE) &&
                   deviceState_.getCurrentlyPlayingPlayerType() == quasar::AudioPlayerType::BLUETOOTH) {
            Directive::Data data = directive->getData();
            data.name = quasar::Directives::BLUETOOTH_PLAYER_PAUSE;
            directive = std::make_shared<Directive>(std::move(data));
        }
    }
}

// NOLINTNEXTLINE(readability-convert-member-functions-to-static)
void BluetoothCapabilityPreprocessor::injectBluetoothPauseToPlayCommands(std::list<std::shared_ptr<Directive>>& directives) {
    if (const auto it = std::find_if(directives.begin(), directives.end(), [](const auto& directive) {
            return directive->is(quasar::Directives::AUDIO_PLAY) ||
                   directive->is(quasar::Directives::MUSIC_PLAY) ||
                   directive->is(quasar::Directives::RADIO_PLAY) ||
                   directive->is(quasar::Directives::VIDEO_PLAY);
        });
        it != std::end(directives)) {
        auto data = Directive::Data(quasar::Directives::BLUETOOTH_PLAYER_PAUSE, "local_action");
        directives.insert(it, std::make_shared<Directive>(std::move(data)));
    }
}
