#include "external_command_capability.h"

#include "notifications/notification_utils.h"

#include <yandex_io/libs/base/directives.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/ete_metrics/ete_util.h>
#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

#include <fstream>
#include <streambuf>

YIO_DEFINE_LOG_MODULE("external_command_capability");

using namespace quasar;

ExternalCommandCapability::ExternalCommandCapability(
    std::shared_ptr<YandexIO::IDevice> device,
    std::shared_ptr<ipc::IConnector> toBugReport,
    std::shared_ptr<ipc::IConnector> toInterface,
    std::shared_ptr<ipc::IConnector> toCalld,
    std::shared_ptr<ipc::IConnector> toNotificationd)
    : device_(std::move(device))
    , toBugReport_(std::move(toBugReport))
    , toInterface_(std::move(toInterface))
    , toCalld_(std::move(toCalld))
    , toNotificationd_(std::move(toNotificationd))
{
}

const std::string& ExternalCommandCapability::getHandlerName() const {
    static const std::string s_name = "ExternalCommandCapability";
    return s_name;
}

const std::set<std::string>& ExternalCommandCapability::getSupportedDirectiveNames() const {
    static const std::set<std::string> s_names = {
        Directives::NOTIFY,
        Directives::MESSENGER_CALL,
        Directives::SEND_BUG_REPORT,
    };

    return s_names;
}

void ExternalCommandCapability::handleDirective(const std::shared_ptr<YandexIO::Directive>& directive)
{
    try {
        const auto& data = directive->getData();
        const std::string& name = data.name;
        const Json::Value& payload = data.payload;
        const std::string& vinsRequestId = data.requestId;

        if (!vinsRequestId.empty()) {
            Json::Value etePayload;
            appendETEHeader(vinsRequestId, etePayload);
            etePayload["command_name"] = name;

            device_->telemetry()->reportEvent("external_command_received", jsonToString(etePayload));
        }

        if (name == Directives::SEND_BUG_REPORT) {
            handleBugReport(payload);
        } else if (name == Directives::MESSENGER_CALL) {
            handleCallCommand(payload);
        } else if (name == Directives::NOTIFY) {
            handleNotification(payload);
        }
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("ExternalCommandCapabitlity.FailedHandleDirective", "handleExternalCommand failed: " << e.what());
    }
}

void ExternalCommandCapability::cancelDirective(const std::shared_ptr<YandexIO::Directive>& /*directive*/)
{
    // do nothing
}

void ExternalCommandCapability::prefetchDirective(const std::shared_ptr<YandexIO::Directive>& /*directive*/)
{
    // do nothing
}

void ExternalCommandCapability::handleBugReport(const Json::Value& payload)
{
    std::string id = getString(payload, "id");

    sendSystemLog(id);
    proto::QuasarMessage message;
    message.set_bug_report_id(std::move(id));
    ipc::SharedMessage sharedMessage{std::move(message)};

    toBugReport_->sendMessage(sharedMessage);
    if (toInterface_ != nullptr) {
        toInterface_->sendMessage(sharedMessage);
    }
}

void ExternalCommandCapability::sendSystemLog(const std::string& id)
{
    YIO_LOG_INFO("Sending bug report");
    try {
        std::ifstream file("/logger/aw_system.log.last");
        std::string log((std::istreambuf_iterator<char>(file)),
                        std::istreambuf_iterator<char>());
        Json::Value report;
        report["bugReportId"] = id;
        report["msg"] = log;
        device_->telemetry()->reportEvent("bug_report_system_log", jsonToString(report));
    } catch (std::exception& e) {
        YIO_LOG_ERROR_EVENT("ExternalCommandCapability.FailedSendBugReport", "Can't send boot event to Metrica: " << e.what());
    }
}

void ExternalCommandCapability::handleCallCommand(const Json::Value& payload) {
    proto::QuasarMessage message;

    if (!payload["accept_call"].isNull()) {
        message.mutable_call_action()->mutable_accept_call();
        message.mutable_call_action()->set_call_guid(getString(payload["accept_call"], "call_guid"));
    } else if (!payload["decline_incoming_call"].isNull()) {
        message.mutable_call_action()->mutable_decline_call();
        message.mutable_call_action()->set_call_guid(getString(payload["decline_incoming_call"], "call_guid"));
    } else if (!payload["decline_current_call"].isNull()) {
        message.mutable_call_action()->mutable_hangup_call();
        message.mutable_call_action()->set_call_guid(getString(payload["decline_current_call"], "call_guid"));
    } else if (const auto& recipient = payload["call_to_recipient"]["recipient"]; !recipient.isNull()) {
        if (const auto deviceId = tryGetString(recipient, "device_id"); !deviceId.empty()) {
            message.mutable_call_action()->mutable_make_call_to_own_device();
            message.mutable_call_action()->set_device_id(TString(deviceId));
        } else if (!tryGetString(recipient, "guid").empty()) {
            // Ignore calls to other users.
        } else {
            throw std::runtime_error("Got call_to_recipient dirrective with empty device_id and guid");
        }

        if (const auto callPayload = tryGetString(payload["call_to_recipient"], "payload"); !callPayload.empty()) {
            message.mutable_call_action()->set_payload(TString(callPayload));
        }
    }

    toCalld_->sendMessage(std::move(message));
}

void ExternalCommandCapability::handleNotification(const Json::Value& payload) {
    const auto& directive = NotificationUtils::fromJson(payload);
    proto::QuasarMessage notificationMessage;
    *notificationMessage.mutable_notify_directive() = directive;
    toNotificationd_->sendMessage(std::move(notificationMessage));
}
