#include "notification_utils.h"

#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

YIO_DEFINE_LOG_MODULE("external_commands");

using namespace quasar;

proto::NotifyDirective NotificationUtils::fromJsonString(const std::string& notifyDirective) {
    auto json = tryParseJson(notifyDirective);
    if (!json.has_value()) {
        proto::NotifyDirective empty = proto::NotifyDirective();
        YIO_LOG_ERROR_EVENT("NotificationUtils.BadJson.NotifyDirective", "Failed to parse json string: " + notifyDirective);
        return empty;
    }
    return fromJson(*json);
}

proto::NotifyDirective NotificationUtils::fromJson(const Json::Value& notifyDirective) {
    proto::NotifyDirective result;
    if (notifyDirective.empty()) {
        return result;
    }
    if (notifyDirective.isMember("notifications")) {
        const Json::Value& notifications = notifyDirective["notifications"];
        if (notifications.isArray()) {
            for (auto& i : notifications) {
                auto notification = result.add_notifications();
                if (i.isMember("text")) {
                    notification->set_text(tryGetString(i, "text"));
                }
                if (i.isMember("id")) {
                    notification->set_id(tryGetString(i, "id"));
                }
            }
        }
    }
    if (notifyDirective.isMember("ring")) {
        auto ring = notifyDirective["ring"].asString();
        std::transform(ring.begin(), ring.end(), ring.begin(), ::tolower);
        if (ring == "nosound") {
            result.set_ring(proto::NotifyDirective_RingType_NO_SOUND);
        } else if (ring == "delicate") {
            result.set_ring(proto::NotifyDirective_RingType_DELICATE);
        } else if (ring == "proactive") {
            result.set_ring(proto::NotifyDirective_RingType_PROACTIVE);
        } else {
            YIO_LOG_ERROR_EVENT("NotificationUtils.InvalidRingType", "Unknown ring type " + ring);
        }
    }
    if (notifyDirective.isMember("version_id")) {
        result.set_versionid(notifyDirective["version_id"].asString());
    }
    return result;
}
