#include "file_player_capability.h"

#include <yandex_io/capabilities/file_player/play_sound_file_listener.h>

#include <yandex_io/libs/base/directives.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

#include <util/system/yassert.h>

YIO_DEFINE_LOG_MODULE("file_player_capability");

using namespace quasar;
using namespace YandexIO;

static const std::string s_capabilityName = "FilePlayerCapability";

FilePlayerCapability::FilePlayerCapability(std::shared_ptr<quasar::ICallbackQueue> worker,
                                           std::shared_ptr<IDevice> device,
                                           ActivityTracker& activityTracker,
                                           IDirectiveProcessorWeakPtr directiveProcessor,
                                           const std::shared_ptr<ipc::IIpcFactory>& ipcFactory,
                                           std::weak_ptr<IRemotingRegistry> remotingRegistry)
    : IRemoteObject(std::move(remotingRegistry))
    , activityTracker_(activityTracker)
    , directiveProcessor_(std::move(directiveProcessor))
    , fileStoragePath_(TFsPath(getString(device->configuration()->getServiceConfig("soundd"), "soundsPath")))
    , worker_(std::move(worker))
    , audioClientConnector_(ipcFactory->createIpcConnector("audioclient"))
{
    audioClientConnector_->setMessageHandler([this](const auto& message) {
        worker_->add([this, message]() {
            onAudioClientMessage(message);
        });
    });
    audioClientConnector_->setConnectHandler([this]() {
        worker_->add([this]() {
            YIO_LOG_INFO("Connected to audioClient");
            isReady_ = true;
            while (!queuedMessages_.empty()) {
                const auto& msg = queuedMessages_.front();
                handleRemotingMessage(msg.message, msg.connection);
                queuedMessages_.pop();
            }
        });
    });
    audioClientConnector_->connectToService();
}

void FilePlayerCapability::init()
{
    if (auto remotingRegistry = getRemotingRegistry().lock()) {
        remotingRegistry->addRemoteObject(s_capabilityName, weak_from_this());
    }
}

FilePlayerCapability::~FilePlayerCapability() {
    if (auto remotingRegistry = getRemotingRegistry().lock()) {
        remotingRegistry->removeRemoteObject(s_capabilityName);
    }

    audioClientConnector_->shutdown();
}

void FilePlayerCapability::playSoundFile(const std::string& fileName,
                                         std::optional<quasar::proto::AudioChannel> channel,
                                         std::optional<PlayParams> params,
                                         std::shared_ptr<IPlaySoundFileListener> listener)
{
    auto copy = sessionByDirective_;
    for (const auto& [directive, session] : copy) {
        if (session->getFileName() == fileName) {
            YIO_LOG_INFO("New play directive for file " << fileName << ", stop current session");
            completeDirective(directive);
            handleOnCompletedCallback(directive);
            if (auto directiveProcessor = directiveProcessor_.lock()) {
                directiveProcessor->onHandleDirectiveCompleted(directive, true);
            }
            break;
        }
    }

    proto::Remoting::FilePlayerCapabilityMethod methodData;
    methodData.set_file_name(TString(fileName));

    if (params.has_value()) {
        methodData.mutable_play_params()->set_play_looped(params->playLooped);
        methodData.mutable_play_params()->set_play_times(params->playTimes);
    }

    Json::Value payload;
    payload["method_data"] = methodData.SerializeAsString();

    Directive::Data data(Directives::SOUND_FILE_PLAY, "local_action", std::move(payload));
    if (params.has_value()) {
        if (!params->requestId.empty()) {
            data.requestId = params->requestId;
        }
        if (!params->parentRequestId.empty()) {
            data.parentRequestId = params->parentRequestId;
        }
    }
    if (data.requestId.empty()) {
        data.requestId = makeUUID();
    }

    if (channel.has_value()) {
        data.channel = channel.value();
    }

    auto directive = std::make_shared<Directive>(std::move(data));
    if (listener) {
        listenerByDirective_[directive] = std::move(listener);
    }

    if (auto dp = directiveProcessor_.lock()) {
        dp->addDirectives({directive});
    }
}

void FilePlayerCapability::stopSoundFile(const std::string& fileName)
{
    proto::Remoting::FilePlayerCapabilityMethod methodData;
    methodData.set_file_name(TString(fileName));

    Json::Value payload;
    payload["method_data"] = methodData.SerializeAsString();

    Directive::Data data(Directives::SOUND_FILE_STOP, "local_action", std::move(payload));
    data.requestId = makeUUID();

    if (auto dp = directiveProcessor_.lock()) {
        dp->addDirectives({std::make_shared<Directive>(std::move(data))});
    }
}

void FilePlayerCapability::handleRemotingMessage(
    const quasar::proto::Remoting& message,
    std::shared_ptr<YandexIO::IRemotingConnection> connection)
{
    if (!isReady_) {
        queuedMessages_.push(QueuedMessage{
            .message = message,
            .connection = connection});
        return;
    }

    if (message.has_file_player_capability_method()) {
        const auto& method = message.file_player_capability_method();
        if (method.method() == quasar::proto::Remoting::FilePlayerCapabilityMethod::HANDSHAKE_REQUEST) {
            quasar::proto::Remoting remoting;
            remoting.set_remote_object_id(TString(s_capabilityName));
            remoting.mutable_file_player_capability_events_method()->mutable_handshake_response();
            connection->sendMessage(remoting);
        } else if (method.method() == quasar::proto::Remoting::FilePlayerCapabilityMethod::PLAY_SOUND_FILE) {
            if (!method.has_file_name()) {
                YIO_LOG_ERROR_EVENT("FilePlayerCapability.RemoteMethodFailed", "fileName is required to call method 'playSoundFile'");
                return;
            }

            const auto optAudioChannel = method.has_channel()
                                             ? std::optional<proto::AudioChannel>(method.channel())
                                             : std::nullopt;

            IFilePlayerCapability::PlayParams playParams;
            if (method.has_play_params()) {
                if (method.play_params().has_play_times()) {
                    playParams.playTimes = method.play_params().play_times();
                }
                if (method.play_params().has_play_looped()) {
                    playParams.playLooped = method.play_params().play_looped();
                }
                if (method.play_params().has_request_id()) {
                    playParams.requestId = method.play_params().request_id();
                }
                if (method.play_params().has_parent_request_id()) {
                    playParams.parentRequestId = method.play_params().parent_request_id();
                }
            }

            if (playParams.requestId.empty()) {
                playParams.requestId = makeUUID();
            }

            std::shared_ptr<IPlaySoundFileListener> listener;
            if (method.has_send_events()) {
                auto onStarted = [requestId = playParams.requestId, connection]() {
                    sendStartedEvent(requestId, connection);
                };
                auto onCompleted = [requestId = playParams.requestId, connection]() {
                    sendCompletedEvent(requestId, connection);
                };
                listener = std::make_shared<PlaySoundFileListener>(onStarted, onCompleted);
            }

            playSoundFile(method.file_name(), optAudioChannel, playParams, listener);

        } else if (method.method() == quasar::proto::Remoting::FilePlayerCapabilityMethod::STOP_SOUND_FILE) {
            if (!method.has_file_name()) {
                YIO_LOG_ERROR_EVENT("FilePlayerCapability.RemoteMethodFailed", "fileName is required to call method 'stopSoundFile'");
                return;
            }
            stopSoundFile(method.file_name());
        }
    }
}

void FilePlayerCapability::sendStartedEvent(const std::string& requestId, std::shared_ptr<YandexIO::IRemotingConnection> connection) {
    quasar::proto::Remoting remoting;
    remoting.set_remote_object_id(TString(s_capabilityName));
    auto method = remoting.mutable_file_player_capability_events_method();
    method->mutable_directive_event()->mutable_started_event();
    method->mutable_directive_event()->mutable_directive()->set_request_id(TString(requestId));

    connection->sendMessage(remoting);
}

void FilePlayerCapability::sendCompletedEvent(const std::string& requestId, std::shared_ptr<YandexIO::IRemotingConnection> connection) {
    quasar::proto::Remoting remoting;
    remoting.set_remote_object_id(TString(s_capabilityName));
    auto method = remoting.mutable_file_player_capability_events_method();
    method->mutable_directive_event()->set_completed_event(proto::DirectiveEvent::SUCCESS);
    method->mutable_directive_event()->mutable_directive()->set_request_id(TString(requestId));

    connection->sendMessage(remoting);
}

const std::string& FilePlayerCapability::getHandlerName() const {
    return s_capabilityName;
}

const std::set<std::string>& FilePlayerCapability::getSupportedDirectiveNames() const {
    static const std::set<std::string> s_names = {
        Directives::SOUND_FILE_PLAY,
        Directives::SOUND_FILE_STOP,
    };

    return s_names;
}

void FilePlayerCapability::handleDirective(const std::shared_ptr<Directive>& directive) {
    const auto& payload = directive->getData().payload;
    proto::Remoting::FilePlayerCapabilityMethod methodData;
    try {
        Y_PROTOBUF_SUPPRESS_NODISCARD methodData.ParseFromString(tryGetString(payload, "method_data"));
    } catch (std::runtime_error& error) {
        YIO_LOG_ERROR_EVENT("FilePlayerCapability.BadMethodData", "payload[\"method_data\"] is invalid. Ignore directive: " << directive->format() << ", error: " << error.what());
        return;
    }

    if (directive->is(Directives::SOUND_FILE_PLAY)) {
        if (!methodData.has_file_name()) {
            YIO_LOG_ERROR_EVENT("FilePlayerCapability.EmptyFileName", "file_name is empty. Ignore directive: " << directive->format());
            handleOnCompletedCallback(directive);
            if (auto directiveProcessor = directiveProcessor_.lock()) {
                directiveProcessor->onHandleDirectiveCompleted(directive, false);
            }
            return;
        }
        const std::string fileName = methodData.file_name();

        TFsPath filePath = fileName;
        if (!filePath.IsAbsolute()) {
            filePath = JoinFsPaths(fileStoragePath_, fileName);
        }

        if (!filePath.Exists()) {
            YIO_LOG_ERROR_EVENT("FilePlayerCapability.FileNotFound", "Sound file not found. Path : " << filePath.GetPath());
            handleOnCompletedCallback(directive);
            if (auto directiveProcessor = directiveProcessor_.lock()) {
                directiveProcessor->onHandleDirectiveCompleted(directive, false);
            }
            return;
        }

        PlayParams playParams;
        if (methodData.has_play_params()) {
            const auto& protoPlayParams = methodData.play_params();
            if (protoPlayParams.has_play_looped()) {
                playParams.playLooped = protoPlayParams.play_looped();
            }
            if (protoPlayParams.has_play_times()) {
                playParams.playTimes = protoPlayParams.play_times();
            }
        }

        const auto& channel = directive->getData().channel;
        const auto session = std::make_shared<FilePlayerSession>(fileName, filePath, channel, std::move(playParams));
        sessionByDirective_[directive] = session;
        if (session->getDescriptor().has_audio_channel()) {
            activityTracker_.addActivity(session);
        }

        std::stringstream log;
        log << "Play sound file " << filePath.GetPath() << ", channel ";
        if (channel.has_value()) {
            log << proto::AudioChannel_Name(channel.value());
        } else {
            log << " nullopt";
        }
        log << ", directive " << directive->format();
        YIO_LOG_INFO(log.str());

        audioClientConnector_->sendMessage(ipc::buildMessage([&session](auto& msg) {
            msg.mutable_media_request()->CopyFrom(session->getPlayRequest());
        }));
    } else if (directive->is(Directives::SOUND_FILE_STOP)) {
        if (!methodData.has_file_name()) {
            YIO_LOG_ERROR_EVENT("FilePlayerCapability.EmptyFileName", "file_name is empty. Ignore directive: " << directive->format());
            return;
        }
        const std::string fileName = methodData.file_name();

        auto copy = sessionByDirective_;
        for (const auto& [directive, session] : copy) {
            if (session->getFileName() == fileName) {
                YIO_LOG_INFO("Stop playing file " << fileName << " by stopSoundFile");
                completeDirective(directive);
                handleOnCompletedCallback(directive);
                if (auto directiveProcessor = directiveProcessor_.lock()) {
                    directiveProcessor->onHandleDirectiveCompleted(directive, true);
                }
                return;
            }
        }
        YIO_LOG_INFO("File " << fileName << " is not playing. Ignore stopSoundFile");
    }
}

void FilePlayerCapability::cancelDirective(const std::shared_ptr<Directive>& directive) {
    completeDirective(directive);
    handleOnCompletedCallback(directive);
}

void FilePlayerCapability::completeDirective(const std::shared_ptr<Directive>& directive) {
    auto iter = sessionByDirective_.find(directive);
    if (iter == sessionByDirective_.end()) {
        return;
    }

    const auto& session = iter->second;
    audioClientConnector_->sendMessage(ipc::buildMessage([&session](auto& msg) {
        msg.mutable_media_request()->CopyFrom(session->getCleanRequest());
    }));

    if (session->getDescriptor().has_audio_channel()) {
        activityTracker_.removeActivity(session);
    }

    sessionByDirective_.erase(directive);
}

void FilePlayerCapability::prefetchDirective(const std::shared_ptr<Directive>& /*directive*/) {
    // do nothing
}

void FilePlayerCapability::onAudioClientMessage(const ipc::SharedMessage& message) {
    auto copy = sessionByDirective_;
    for (const auto& [directive, session] : copy) {
        onQuasarMessage(directive, session, message);
    }
}

void FilePlayerCapability::onQuasarMessage(
    const std::shared_ptr<Directive>& directive,
    const std::shared_ptr<FilePlayerSession>& session,
    const ipc::SharedMessage& sharedMessage) {
    const auto& message = *sharedMessage;
    if (!message.has_audio_client_event()) {
        return;
    }

    const auto& event = message.audio_client_event();
    if (event.event() != proto::AudioClientEvent::STATE_CHANGED) {
        return;
    }
    if (!event.has_player_descriptor()) {
        return;
    }
    if (event.player_descriptor().has_type() && event.player_descriptor().type() != proto::AudioPlayerDescriptor::FILE_PLAYER) {
        return;
    }
    if (!event.player_descriptor().has_player_id() ||
        event.player_descriptor().player_id() != session->getDescriptor().player_id()) {
        return;
    }
    switch (event.state()) {
        case proto::AudioClientState::PLAYING:
            if (session->getPlayedCount() == 0) {
                YIO_LOG_INFO("Session started. sound_file: " << session->getFileName() << ", request_id: " << directive->getRequestId());
                handleOnStartedCallback(directive);
                if (auto directiveProcessor = directiveProcessor_.lock()) {
                    directiveProcessor->onHandleDirectiveStarted(directive);
                }
            }
            break;
        case proto::AudioClientState::FINISHED:
        case proto::AudioClientState::STOPPED:
            if (session->onFilePlayFinished()) {
                YIO_LOG_INFO("Replay for the " << session->getPlayedCount() + 1 << " time. sound_file: " << session->getFileName() << ", request_id: " << directive->getRequestId());
                audioClientConnector_->sendMessage(ipc::buildMessage([&session](auto& msg) {
                    msg.mutable_media_request()->CopyFrom(session->getReplayRequest());
                }));
            } else {
                YIO_LOG_INFO("Session stopped. sound_file: " << session->getFileName() << ", request_id: " << directive->getRequestId());
                completeDirective(directive);
                handleOnCompletedCallback(directive);
                if (auto directiveProcessor = directiveProcessor_.lock()) {
                    directiveProcessor->onHandleDirectiveCompleted(directive, true);
                }
            }
            break;
        case proto::AudioClientState::FAILED:
            YIO_LOG_INFO("Session failed. sound_file: " << session->getFileName() << ", request_id: " << directive->getRequestId());
            completeDirective(directive);
            handleOnCompletedCallback(directive);
            if (auto directiveProcessor = directiveProcessor_.lock()) {
                directiveProcessor->onHandleDirectiveCompleted(directive, false);
            }
            break;
        default: {
            // do nothing
            break;
        }
    }
}

void FilePlayerCapability::handleOnStartedCallback(const std::shared_ptr<Directive>& directive) {
    auto iter = listenerByDirective_.find(directive);
    if (iter == listenerByDirective_.end()) {
        return;
    }

    YIO_LOG_DEBUG("Call onStarted() for request_id " << directive->getRequestId());
    const auto& listener = iter->second;
    listener->onStarted();
}

void FilePlayerCapability::handleOnCompletedCallback(const std::shared_ptr<Directive>& directive) {
    auto iter = listenerByDirective_.find(directive);
    if (iter == listenerByDirective_.end()) {
        return;
    }

    YIO_LOG_DEBUG("Call onCompleted() for request_id " << directive->getRequestId());
    const auto& listener = iter->second;
    listener->onCompleted();

    listenerByDirective_.erase(iter);
}
