#include "legacy_iot_capability.h"

#include <yandex_io/libs/base/directives.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

YIO_DEFINE_LOG_MODULE("legacy_iot_capability");

using namespace quasar;

LegacyIotCapability::LegacyIotCapability(std::shared_ptr<ipc::IConnector> toIot)
    : toIot_(std::move(toIot))
{
}

const std::string& LegacyIotCapability::getHandlerName() const {
    static const std::string s_name = "LegacyIotCapability";
    return s_name;
}

const std::set<std::string>& LegacyIotCapability::getSupportedDirectiveNames() const {
    static const std::set<std::string> s_names = {
        Directives::IOT_DISCOVERY_START,
        Directives::IOT_DISCOVERY_STOP,
        Directives::IOT_DISCOVERY_CREDENTIALS,
    };

    return s_names;
}

void LegacyIotCapability::handleDirective(const std::shared_ptr<YandexIO::Directive>& directive)
{
    try {
        const auto& data = directive->getData();
        const std::string& name = data.name;
        const Json::Value& payload = data.payload;

        if (name == Directives::IOT_DISCOVERY_START) {
            handleIotDiscoveryStart(payload);
        } else if (name == Directives::IOT_DISCOVERY_STOP) {
            handleIotDiscoveryStop(payload);
        } else if (name == Directives::IOT_DISCOVERY_CREDENTIALS) {
            handleIotDiscoveryCredentials(payload);
        }
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("LegacyIotCapabitlity.FailedHandleDirective", e.what());
    }
}

void LegacyIotCapability::cancelDirective(const std::shared_ptr<YandexIO::Directive>& /*directive*/)
{
    // do nothing
}

void LegacyIotCapability::prefetchDirective(const std::shared_ptr<YandexIO::Directive>& /*directive*/)
{
    // do nothing
}

void LegacyIotCapability::handleIotDiscoveryStart(const Json::Value& payload) {
    proto::QuasarMessage message;
    auto discovery = message.mutable_iot_request()->mutable_start_discovery();
    try {
        discovery->set_device_type(getString(payload, "device_type"));
        discovery->set_ssid(getString(payload, "ssid"));
    } catch (const std::runtime_error& e) {
        YIO_LOG_ERROR_EVENT("LegacyIotCapability.InvalidIotStartDiscoveryPayload", "Can't set info for iot start request " << e.what());
        return;
    }
    if (payload["timeout_ms"].isInt()) {
        discovery->set_timeout_ms(getInt(payload, "timeout_ms"));
    }
    toIot_->sendMessage(std::move(message));
}

void LegacyIotCapability::handleIotDiscoveryStop(const Json::Value& payload) {
    proto::QuasarMessage message;
    message.mutable_iot_request()->mutable_stop_discovery();
    if (!payload["result"].isNull()) {
        message.mutable_iot_request()->mutable_stop_discovery()->mutable_result()->set_code(parseDiscoveryResultCode(getString(payload, "result")));
    }
    toIot_->sendMessage(std::move(message));
}

void LegacyIotCapability::handleIotDiscoveryCredentials(const Json::Value& payload) {
    proto::QuasarMessage message;
    auto credentials = message.mutable_iot_request()->mutable_credentials();
    try {
        credentials->set_ssid(getString(payload, "ssid"));
        credentials->set_password(getString(payload, "password"));
        credentials->set_token(getString(payload, "token"));
        credentials->set_cipher(getString(payload, "cipher"));
    } catch (const std::runtime_error& e) {
        YIO_LOG_ERROR_EVENT("LegacyIotCapability.InvalidIotDiscoveryCredentialsPayload", "Can't set credentials: " << e.what());
        return;
    }
    toIot_->sendMessage(std::move(message));
}

proto::IotDiscoveryResult::ResultCode LegacyIotCapability::parseDiscoveryResultCode(const std::string& code)
{
    static const std::map<std::string, proto::IotDiscoveryResult::ResultCode> map({{"success", proto::IotDiscoveryResult::SUCCESS},
                                                                                   {"failure", proto::IotDiscoveryResult::FAILURE},
                                                                                   {"timeout", proto::IotDiscoveryResult::TIMEOUT},
                                                                                   {"superceded", proto::IotDiscoveryResult::SUPERCEDED},
                                                                                   {"not_started", proto::IotDiscoveryResult::NOT_STARTED}});

    if (const auto it = map.find(code); it != map.end()) {
        return it->second;
    }
    return proto::IotDiscoveryResult::UNKNOWN;
}
