#include "legacy_player_capability.h"

#include <yandex_io/capabilities/device_state/converters/converters.h>
#include <yandex_io/libs/base/directives.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/ete_metrics/ete_util.h>
#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

#include <fstream>
#include <streambuf>

YIO_DEFINE_LOG_MODULE("legacy_player_capability");

using namespace quasar;
using namespace YandexIO;

LegacyPlayerCapability::LegacyPlayerCapability(
    YandexIO::IDirectiveProcessorWeakPtr directiveProcessor,
    AliceDeviceState& aliceDeviceState,
    std::shared_ptr<YandexIO::IDevice> device,
    DeviceContext& deviceContext,
    std::shared_ptr<quasar::ipc::IConnector> toMedia,
    std::shared_ptr<quasar::ipc::IConnector> toInterface,
    std::shared_ptr<IDeviceStateCapability> deviceStateCapability)
    : directiveProcessor_(std::move(directiveProcessor))
    , aliceDeviceState_(aliceDeviceState)
    , device_(std::move(device))
    , toMedia_(std::move(toMedia))
    , toInterface_(std::move(toInterface))
    , deviceContext_(deviceContext)
    , deviceStateCapability_(std::move(deviceStateCapability))
{
    const auto config = device_->configuration()->getServiceConfig("mediad");
    hasJavaRadio_ = tryGetBool(config, "hasJavaRadio", false);
    hasScreens_ = tryGetBool(config, "hasScreens", false);
}

const std::string& LegacyPlayerCapability::getHandlerName() const {
    static const std::string s_name = "LegacyPlayerCapability";
    return s_name;
}

const std::set<std::string>& LegacyPlayerCapability::getSupportedDirectiveNames() const {
    static const std::set<std::string> s_names = {
        Directives::MUSIC_PLAY,
        Directives::RADIO_PLAY,
        Directives::STOP_LEGACY_PLAYER,
        Directives::PLAYER_PAUSE,
        Directives::PLAYER_CONTINUE,
        Directives::PLAYER_NEXT_TRACK,
        Directives::PLAYER_PREVIOUS_TRACK,
        Directives::PLAYER_REPLAY,
        Directives::PLAYER_LIKE,
        Directives::PLAYER_DISLIKE,
        Directives::PLAYER_REWIND,
    };

    return s_names;
}

void LegacyPlayerCapability::handleDirective(const std::shared_ptr<YandexIO::Directive>& directive)
{
    try {
        const auto& data = directive->getData();
        const std::string& name = data.name;
        const Json::Value& payload = data.payload;
        const std::string& vinsRequestId = data.requestId;

        if (!vinsRequestId.empty()) {
            Json::Value etePayload;
            appendETEHeader(vinsRequestId, etePayload);
            etePayload["command_name"] = name;

            device_->telemetry()->reportEvent("legacy_player_directive_received", jsonToString(etePayload));
        }

        if (name == Directives::MUSIC_PLAY) {
            handleMusicPlay(directive);
        } else if (name == Directives::RADIO_PLAY) {
            handleRadioPlay(directive);
        } else if (name == Directives::PLAYER_PAUSE) {
            handlePlayerPause(payload, vinsRequestId);
        } else if (name == Directives::STOP_LEGACY_PLAYER) {
            handlePlayerPause(payload, vinsRequestId);
        } else if (name == Directives::PLAYER_CONTINUE) {
            handlePlayerContinue(payload, vinsRequestId);
        } else if (name == Directives::PLAYER_NEXT_TRACK) {
            handlePlayerNextTrack(vinsRequestId, payload);
        } else if (name == Directives::PLAYER_PREVIOUS_TRACK) {
            handlePlayerPreviousTrack(vinsRequestId, payload);
        } else if (name == Directives::PLAYER_REPLAY) {
            handlePlayerReplay(vinsRequestId);
        } else if (name == Directives::PLAYER_LIKE) {
            handlePlayerLike(vinsRequestId);
        } else if (name == Directives::PLAYER_DISLIKE) {
            handlePlayerDislike(vinsRequestId);
        } else if (name == Directives::PLAYER_REWIND && !payload.isNull()) {
            handlePlayerRewind(payload, vinsRequestId);
        }
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("LegacyPlayerCapabitlity.FailedHandleDirective", "handleDirective failed: " << e.what());
    }
}

void LegacyPlayerCapability::cancelDirective(const std::shared_ptr<YandexIO::Directive>& /*directive*/)
{
    // do nothing
}

void LegacyPlayerCapability::prefetchDirective(const std::shared_ptr<YandexIO::Directive>& /*directive*/)
{
    // do nothing
}

void LegacyPlayerCapability::handleMediadMessage(const ipc::SharedMessage& sharedMessage)
{
    if (sharedMessage->has_legacy_player_state_changed()) {
        onLegacyPlayerStateChanged(sharedMessage->legacy_player_state_changed());
    }
}

void LegacyPlayerCapability::onUserConfig(const quasar::proto::UserConfig& userConfig)
{
    try {
        const std::string& config = userConfig.config();
        if (!config.empty()) {
            const Json::Value jsonConfig = parseJson(config);
            const Json::Value& systemConfig = jsonConfig["system_config"];

            radioUrlRewriter_.updateRules(systemConfig["YandexRadio"]["urlRewriteRules"]);
        }
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("LegacyPlayerCapability.FailedToApplyUserConfig", e.what());
    }
}

void LegacyPlayerCapability::onInterfacedMessage(const quasar::ipc::SharedMessage& sharedMessage)
{
    if (sharedMessage->has_app_state()) {
        const auto& appState = sharedMessage->app_state();
        if (hasJavaRadio_ && appState.has_radio_state()) {
            deviceStateCapability_->setRadioState(YandexIO::convertRadioState(appState.radio_state(), javaRadioPlayTs_));
        }
    } else if (sharedMessage->has_legacy_player_state_changed()) {
        onLegacyPlayerStateChanged(sharedMessage->legacy_player_state_changed());
    }
}

void LegacyPlayerCapability::onLegacyPlayerStateChanged(const quasar::proto::LegacyPlayerStateChanged& stateChanged)
{
    if (!stateChanged.has_player_type() || !stateChanged.has_vins_request_id()) {
        return;
    }

    std::shared_ptr<YandexIO::Directive> directive;
    if (stateChanged.player_type() == proto::LegacyPlayerStateChanged::YANDEX_MUSIC) {
        directive = musicPlayDirective_;
    } else {
        directive = radioPlayDirective_;
    }

    if (directive == nullptr || directive->getRequestId() != stateChanged.vins_request_id()) {
        YIO_LOG_ERROR_EVENT("LegacyPlayerCapability.UnknownDirectiveConfirmed", "Confirmed media directive with id:" << stateChanged.vins_request_id() << " is not found");
        return;
    }

    if (auto dp = directiveProcessor_.lock()) {
        dp->onHandleDirectiveStarted(directive);
    }
}

void LegacyPlayerCapability::handlePlayerRewind(
    const Json::Value& payload, const std::string& vinsRequestId)
{
    if (!isCurrentScreenSuitable()) {
        return;
    }

    if (payload["amount"].isNull() || payload["type"].isNull()) {
        YIO_LOG_ERROR_EVENT("LegacyPlayerCapability.InvalidPlayerRewindPayload", "Unexpected player_rewind payload: " + jsonToString(payload));
        return;
    }

    double amount = getDouble(payload, "amount");
    proto::MediaRequest::RewindType type;
    std::string type_string = getString(payload, "type");
    if (type_string == "forward") {
        type = proto::MediaRequest::FORWARD;
    } else if (type_string == "backward") {
        type = proto::MediaRequest::BACKWARD;
    } else if (type_string == "absolute") {
        type = proto::MediaRequest::ABSOLUTE;
    } else {
        YIO_LOG_ERROR_EVENT("LegacyPlayerCapability.InvalidPlayerRewindType", "Unexpected player_rewind payload: " + jsonToString(payload));
        return;
    }

    auto message = ipc::buildMessage([&](auto& msg) {
        msg.mutable_media_request()->mutable_rewind()->set_amount(amount);
        msg.mutable_media_request()->mutable_rewind()->set_type(type);
        msg.mutable_media_request()->set_vins_request_id(TString(vinsRequestId));
    });

    toMedia_->sendMessage(message);
    if (toInterface_ != nullptr) {
        toInterface_->sendMessage(message);
    }
}

void LegacyPlayerCapability::handleMusicPlay(const std::shared_ptr<YandexIO::Directive>& directive)
{
    musicPlayDirective_ = directive;

    const Json::Value& payload = directive->getData().payload;
    const std::string& vinsRequestId = directive->getRequestId();

    const std::string uid = tryGetString(payload, "uid");
    const std::string sessionId = tryGetString(payload, "session_id");
    const double offset = tryGetDouble(payload, "offset", 0);

    auto message = ipc::buildMessage([&](auto& msg) {
        auto& mediaRequest = *msg.mutable_media_request();
        mediaRequest.mutable_play_audio()->set_multiroom_token(tryGetString(payload, "multiroom_token"));
        mediaRequest.set_uid(TString(uid));
        mediaRequest.set_session_id(TString(sessionId));
        mediaRequest.set_vins_init_request_id(TString(vinsRequestId));
        mediaRequest.set_vins_request_id(TString(vinsRequestId));
        mediaRequest.set_offset(offset);
    });

    toMedia_->sendMessage(message);
    if (toInterface_ != nullptr) {
        toInterface_->sendMessage(message);
    }

    lastPlayIsMusic_ = true;
    aliceDeviceState_.onLegacyPlayerStart();
}

void LegacyPlayerCapability::handleRadioPlay(const std::shared_ptr<YandexIO::Directive>& directive)
{
    radioPlayDirective_ = directive;

    const Json::Value& payload = directive->getData().payload;
    const std::string& vinsRequestId = directive->getRequestId();

    auto message = ipc::buildMessage([&](auto& msg) {
        const auto uid = tryGetString(payload, "uid");
        const auto forceRestart = getOptionalBool(payload, "force_restart_player");
        const auto streamUrl = trim(getString(payload, "streamUrl"));
        const auto patchedUrl = radioUrlRewriter_.rewriteUrl(streamUrl, true);
        if (streamUrl != patchedUrl) {
            YIO_LOG_INFO("Patch radio url from \"" << streamUrl << "\" to \"" << patchedUrl << "\"");
        }

        auto& mediaRequest = *msg.mutable_media_request();
        auto& playRadio = *mediaRequest.mutable_play_radio();

        playRadio.set_url(TString(patchedUrl));
        playRadio.set_uid(TString(uid));
        playRadio.set_raw(jsonToString(payload));

        if (!payload["radioId"].isNull()) {
            playRadio.set_id(payload["radioId"].asString());
        }
        if (!payload["title"].isNull()) {
            playRadio.set_title(payload["title"].asString());
        }
        if (!payload["imageUrl"].isNull()) {
            playRadio.set_image_url(payload["imageUrl"].asString());
        }
        if (!payload["color"].isNull()) {
            playRadio.set_color(payload["color"].asString());
        }
        if (forceRestart.has_value()) {
            playRadio.set_force_restart_player(forceRestart.value());
        }

        mediaRequest.set_vins_init_request_id(TString(vinsRequestId));
        mediaRequest.set_vins_request_id(TString(vinsRequestId));
    });

    deviceContext_.fireMediaRequest(proto::MediaContentType::MUSIC);
    if (hasJavaRadio_) {
        javaRadioPlayTs_ = getNowTimestampMs();

        auto pauseMessage = ipc::buildMessage([&](auto& msg) {
            msg.mutable_media_request()->mutable_pause();
            msg.mutable_media_request()->mutable_pause()->set_smooth(false);
            msg.mutable_media_request()->set_fire_device_context(false);
            msg.mutable_media_request()->set_vins_request_id(TString(vinsRequestId));
        });
        toMedia_->sendMessage(pauseMessage);

        if (toInterface_ != nullptr) {
            toInterface_->sendMessage(message);
        }
    } else {
        toMedia_->sendMessage(message);
    }

    lastPlayIsMusic_ = false;
    aliceDeviceState_.onRadioPlayerStart();
}

void LegacyPlayerCapability::handlePlayerPause(
    const Json::Value& payload, const std::string& vinsRequestId)
{
    auto message = ipc::buildMessage([&](auto& msg) {
        msg.mutable_media_request()->mutable_pause();
        msg.mutable_media_request()->mutable_pause()->set_smooth(payload.isMember("smooth") && payload["smooth"].asBool());
        msg.mutable_media_request()->set_vins_request_id(TString(vinsRequestId));
    });

    toMedia_->sendMessage(message);
    if (toInterface_ != nullptr) {
        toInterface_->sendMessage(message);
    }

    if (lastPlayIsMusic_) {
        lastPlayIsMusic_ = false;
        aliceDeviceState_.onLegacyPlayerStop();
    }
}

void LegacyPlayerCapability::handlePlayerContinue(
    const Json::Value& payload, const std::string& vinsRequestId)
{
    proto::MediaRequest::PlayerType playerType = proto::MediaRequest::AUDIO;
    if (!payload["player"].isNull()) {
        std::string player = getString(payload, "player");
        if (player == "video") {
            playerType = proto::MediaRequest::VIDEO;
        } else if (player == "current") {
            playerType = proto::MediaRequest::CURRENT;
        }
    }

    auto message = ipc::buildMessage([&](auto& msg) {
        msg.mutable_media_request()->set_resume(playerType);
        msg.mutable_media_request()->set_vins_request_id(TString(vinsRequestId));
    });

    switch (playerType) {
        case proto::MediaRequest::AUDIO: {
            toMedia_->sendMessage(message);

            if (toInterface_ != nullptr &&
                aliceDeviceState_.getCurrentPlayerType() != AudioPlayerType::BLUETOOTH) {
                auto playAudioCommand = ipc::buildMessage([&](auto& msg) {
                    msg.mutable_media_request()->mutable_play_audio();
                });
                toInterface_->sendMessage(playAudioCommand);
            }
            break;
        }
        case proto::MediaRequest::VIDEO: {
            if (toInterface_ != nullptr) {
                toInterface_->sendMessage(message);
            }
            break;
        }
        case proto::MediaRequest::CURRENT: {
            YIO_LOG_ERROR_EVENT("LegacyPlayerCapability.handleContinueCurrent", "player type 'current' must be resolved outside of capability");
        }
    }
}

void LegacyPlayerCapability::handlePlayerNextTrack(
    const std::string& vinsRequestId, const Json::Value& payload)
{
    if (!isCurrentScreenSuitable()) {
        return;
    }

    const auto setPause = tryGetBool(payload, "setPause", false);
    const auto fireDeviceContext = tryGetBool(payload, "fire_device_context", false);
    auto message = ipc::buildMessage([&](auto& msg) {
        msg.mutable_media_request()->set_vins_request_id(TString(vinsRequestId));
        msg.mutable_media_request()->set_fire_device_context(fireDeviceContext);
        msg.mutable_media_request()->mutable_next()->set_set_pause(setPause);
    });

    toMedia_->sendMessage(message);
    if (toInterface_ != nullptr) {
        toInterface_->sendMessage(message);
    }
}

void LegacyPlayerCapability::handlePlayerPreviousTrack(
    const std::string& vinsRequestId, const Json::Value& payload)
{
    if (!isCurrentScreenSuitable()) {
        return;
    }

    const auto forced = tryGetBool(payload, "forced", true);
    const auto setPause = tryGetBool(payload, "setPause", false);
    const auto fireDeviceContext = tryGetBool(payload, "fire_device_context", false);

    auto message = ipc::buildMessage([&](auto& msg) {
        msg.mutable_media_request()->set_vins_request_id(TString(vinsRequestId));
        msg.mutable_media_request()->set_fire_device_context(fireDeviceContext);
        msg.mutable_media_request()->mutable_previous()->set_set_pause(setPause);
        msg.mutable_media_request()->mutable_previous()->set_forced(forced);
    });

    toMedia_->sendMessage(message);
    if (toInterface_ != nullptr) {
        toInterface_->sendMessage(message);
    }
}

void LegacyPlayerCapability::handlePlayerReplay(const std::string& vinsRequestId)
{
    if (isCurrentScreenSuitable()) {
        auto message = ipc::buildMessage([&](auto& msg) {
            msg.mutable_media_request()->mutable_replay();
            msg.mutable_media_request()->set_vins_request_id(TString(vinsRequestId));
        });
        toMedia_->sendMessage(message);
        if (toInterface_ != nullptr) {
            toInterface_->sendMessage(message);
        }
    }
}

void LegacyPlayerCapability::handlePlayerLike(const std::string& vinsRequestId)
{
    if (isCurrentScreenSuitable()) {
        auto message = ipc::buildMessage([&](auto& msg) {
            msg.mutable_media_request()->mutable_like();
            msg.mutable_media_request()->set_vins_request_id(TString(vinsRequestId));
        });
        toMedia_->sendMessage(message);
    }
}

void LegacyPlayerCapability::handlePlayerDislike(const std::string& vinsRequestId)
{
    if (isCurrentScreenSuitable()) {
        auto message = ipc::buildMessage([&](auto& msg) {
            msg.mutable_media_request()->mutable_dislike();
            msg.mutable_media_request()->set_vins_request_id(TString(vinsRequestId));
        });
        toMedia_->sendMessage(message);
    }
}

bool LegacyPlayerCapability::isCurrentScreenSuitable() const {
    if (!hasScreens_) {
        return true;
    }

    const auto& appState = aliceDeviceState_.getAppState();
    if (appState.has_screen_state()) {
        const auto currentScreen = appState.screen_state().screen_type();
        if (currentScreen == proto::AppState_ScreenType_MUSIC_PLAYER ||
            currentScreen == proto::AppState_ScreenType_BLUETOOTH ||
            currentScreen == proto::AppState_ScreenType_VIDEO_PLAYER ||
            currentScreen == proto::AppState_ScreenType_RADIO_PLAYER) {
            return true;
        }
    }

    return false;
}
