#include "url_rewriter.h"

#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

YIO_DEFINE_LOG_MODULE("url_rewriter");

using namespace quasar;

UrlRewriter::UrlRewriter(const Json::Value& arrayOfRules)
{
    updateRules(arrayOfRules);
}

void UrlRewriter::updateRules(const Json::Value& arrayOfRules) noexcept {
    if (arrayOfRules.empty()) {
        if (!rules_.empty()) {
            rules_.clear();
        }
        return;
    }

    std::vector<Rule> rules;
    try {
        if (arrayOfRules.isArray()) {
            rules.reserve(arrayOfRules.size());
            for (const auto& jRrule : arrayOfRules) {
                auto regex = tryGetString(jRrule, "regex", "");
                auto replacement = tryGetString(jRrule, "replacement", "");
                auto next = tryGetBool(jRrule, "next", false);
                if (regex.empty()) {
                    YIO_LOG_ERROR_EVENT("UrlRewriter.InvalidEmptyRegex", "Empty regex in url rewrite rule");
                    continue;
                }
                rules.emplace_back(Rule{regex, std::regex{regex}, std::move(replacement), next});
            }
            rules_ = std::move(rules);
        } else {
            YIO_LOG_ERROR_EVENT("UrlRewriter.InvalidUrlRewriteRules", "Field \"urlRewriteRules\" must be array of strings");
        }
    } catch (std::exception& ex) {
        YIO_LOG_ERROR_EVENT("UrlRewriter.FailedUpdateRules", "Fail to update url rewrite urls: " << ex.what());
    } catch (...) {
        YIO_LOG_ERROR_EVENT("UrlRewriter.FailedUpdateRules", "Fail to update url rewrite urls: unexpected exception");
    }
}

std::string UrlRewriter::rewriteUrl(std::string originUrl, bool verbose) const noexcept {
    if (rules_.empty()) {
        return originUrl;
    }

    std::string result;
    for (const auto& rule : rules_) {
        try {
            result = std::regex_replace(originUrl, rule.regex, rule.replacement);
        } catch (const std::exception& ex) {
            if (verbose) {
                YIO_LOG_DEBUG("Fail rewrite rule \"" << rule.text << "\": " << ex.what());
            }
            continue;
        } catch (...) {
            if (verbose) {
                YIO_LOG_DEBUG("Fail rewrite rule \"" << rule.text << "\": unexpected exception");
            }
            continue;
        }
        if (result != originUrl) {
            if (verbose) {
                YIO_LOG_DEBUG("Rewrite matching: rule=\"" << rule.text << "\", original url=\"" << originUrl << ", rewritten url=" << result);
            }
            if (rule.next) {
                originUrl = std::move(result);
                continue;
            }
            return result;
        } else if (verbose) {
            YIO_LOG_DEBUG("Rewrite miss matching: rule=\"" << rule.text << "\", original url=\"" << originUrl);
        }
    }

    return originUrl;
}
