#include "mrforwarder.h"
#include "mrforwarder_capability.h"

#include <yandex_io/libs/base/directives.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/cryptography/digest.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/json.h>

#include <util/system/yassert.h>

using namespace YandexIO;
using namespace quasar;

YIO_DEFINE_LOG_MODULE("mrforwarder_preprocessor");

MRForwarderCapability::MRForwarderCapability(
    std::shared_ptr<YandexIO::IDevice> device,
    std::shared_ptr<IGlagolClusterProvider> glagolClusterProvider,
    std::shared_ptr<IMultiroomProvider> multiroomProvider)
    : device_(std::move(device))
    , glagolClusterProvider_(std::move(glagolClusterProvider))
    , multiroomProvider_(std::move(multiroomProvider))
{
    Y_VERIFY(device_);
    Y_VERIFY(glagolClusterProvider_);
    Y_VERIFY(multiroomProvider_);
}

MRForwarderCapability::~MRForwarderCapability() = default;

const std::string& MRForwarderCapability::getEndpointId() const {
    return MRFORWARDER_ENDPOINT_ID;
}

const std::string& MRForwarderCapability::getHandlerName() const {
    static const std::string s_name = "MRForwarderCapability";
    return s_name;
}

const std::set<std::string>& MRForwarderCapability::getSupportedDirectiveNames() const {
    static const std::set<std::string> s_allDirectives;
    return s_allDirectives;
}

void MRForwarderCapability::handleDirective(const std::shared_ptr<YandexIO::Directive>& inDirective) {
    auto directive = inDirective;
    Y_VERIFY(directive && directive->getData().endpointId == MRFORWARDER_ENDPOINT_ID);

    YIO_LOG_DEBUG("MRForwarder: Process directive: " << Directive::convertToJsonString(directive));
    auto multiroomState = multiroomProvider_->multiroomState().value();

    auto checkDevice = [&](const std::string& deviceId) {
        if (multiroomState) {
            for (const auto& peer : multiroomState->peers) {
                if (peer.deviceId == deviceId) {
                    return true;
                }
            }
        }
        return false;
    };

    auto forwardDirectiveToDevice = [&](const std::string& targetDeviceId) {
        if (!checkDevice(targetDeviceId)) {
            YIO_LOG_WARN("MRForwarder: Fail to route directive to " << targetDeviceId << ". No target device online: " << Directive::convertToJsonString(directive));
            return;
        }

        auto isYandexMusicMaster = (multiroomState->broadcast && multiroomState->broadcast->music && multiroomState->broadcast->masterDeviceId == targetDeviceId);
        if (isYandexMusicMaster) {
            if (directive->getData().name == Directives::CLEAR_QUEUE) {
                Directive::Data newData{directive->getData()};
                newData.name = Directives::PLAYER_PAUSE;
                directive = std::make_shared<Directive>(std::move(newData));
                YIO_LOG_INFO("MRForwarder: Replace directive \"" << Directives::CLEAR_QUEUE << "\" with \"" << Directives::PLAYER_PAUSE << "\" for yandex music master");
            }
        }

        auto isLegacyDevice = std::find(multiroomState->legacyDeviceIds.begin(), multiroomState->legacyDeviceIds.end(), TString(targetDeviceId));
        if (isLegacyDevice) {
            // ALICE-19341 For JBL 80 version compatibility
            auto ec = YandexIO::Directive::convertToExternalCommandProtobuf(directive);
            if (ec.name() == Directives::PLAYBACK_TOGGLE_PLAY_PAUSE) {
                if (multiroomState->broadcast && multiroomState->broadcast->playingState == MultiroomState::PlayingState::PLAYING) {
                    ec.set_name(Directives::PLAYER_PAUSE);
                } else {
                    ec.set_name(Directives::PLAYER_CONTINUE);
                }
            } else {
                static const auto remap = std::map<std::string, std::string>{
                    {Directives::PLAYBACK_PLAY, Directives::PLAYER_CONTINUE},
                    {Directives::PLAYBACK_PAUSE, Directives::PLAYER_PAUSE},
                    {Directives::PLAYBACK_LIKE, Directives::PLAYER_LIKE},
                    {Directives::PLAYBACK_DISLIKE, Directives::PLAYER_DISLIKE},
                    {Directives::PLAYBACK_NEXT, Directives::PLAYER_NEXT_TRACK},
                    {Directives::PLAYBACK_PREV, Directives::PLAYER_PREVIOUS_TRACK}};
                for (const auto& p : remap) {
                    if (ec.name() == p.first) {
                        ec.set_name(TString(p.second));
                        break;
                    }
                }
            }
            if (ec.name() != directive->getData().name) {
                YIO_LOG_INFO("MRForwarder: Replace directive \"" << directive->getData().name << "\" with \"" << ec.name() << "\" for legacy device");
            }
            auto message = ipc::buildMessage([&](auto& msg) {
                auto& mDirective = *msg.mutable_multiroom_directive();
                mDirective.set_sender_device_id(TString(device_->deviceId()));
                if (!directive->getData().multiroomSessionId.empty() && multiroomState->broadcast && multiroomState->broadcast->masterDeviceId == targetDeviceId) {
                    mDirective.set_master_device_id(TString(multiroomState->broadcast->masterDeviceId));
                    mDirective.set_multiroom_token(TString(multiroomState->broadcast->multiroomToken));
                    mDirective.set_session_timestamp_ms(multiroomState->broadcast->sessionTimestampMs);
                    if (ec.name() == Directives::PLAYER_PAUSE) {
                        mDirective.mutable_stop()->set_reason("Original directive " + directive->getData().name + " translated and forwarded from slave");
                    } else {
                        mDirective.mutable_master_external_command()->CopyFrom(ec);
                        mDirective.mutable_master_external_command()->clear_room_device_ids();
                    }
                } else {
                    mDirective.mutable_execute_external_command()->CopyFrom(ec);
                    mDirective.mutable_execute_external_command()->clear_room_device_ids();
                    mDirective.mutable_execute_external_command()->clear_multiroom_session_id();
                }
            });
            YIO_LOG_INFO("MRForwarder: Forward directive to LEGACY device " << targetDeviceId << ": " << message);
            glagolClusterProvider_->send(targetDeviceId, "multiroomd", message);
        } else {
            auto message = ipc::buildMessage([&](auto& msg) {
                msg.mutable_directive()->CopyFrom(YandexIO::Directive::convertToDirectiveProtobuf(directive));
                msg.mutable_directive()->clear_endpoint_id();
                msg.mutable_directive()->clear_room_device_ids();
                msg.mutable_directive()->set_is_route_locally(true);
            });
            YIO_LOG_INFO("MRForwarder: Forward directive to device " << targetDeviceId << ": " << message);
            glagolClusterProvider_->send(targetDeviceId, "aliced", message);
        }
    };

    if (!directive->getData().roomDeviceIds.empty()) {
        std::map<std::string, std::string> deviceSelectorMap;
        for (const auto& deviceId : directive->getData().roomDeviceIds) {
            if (!checkDevice(deviceId)) {
                continue;
            }
            Md5Hasher md5Hasher;
            md5Hasher.update(directive->getRequestId());
            md5Hasher.update(deviceId);
            auto key = md5Hasher.finalize().hashString(Md5Hasher::Encoder::BASE58, 6);
            deviceSelectorMap[key] = deviceId;
        }
        if (deviceSelectorMap.empty()) {
            YIO_LOG_WARN("MRForwarder: Fail to route directive. No target device online: " << Directive::convertToJsonString(directive));
        } else {
            forwardDirectiveToDevice(deviceSelectorMap.begin()->second);
        }
    } else if (!directive->getData().multiroomSessionId.empty()) {
        if (multiroomState->multiroomSessionId() == directive->getData().multiroomSessionId) {
            forwardDirectiveToDevice(multiroomState->broadcast->masterDeviceId);
        } else {
            YIO_LOG_WARN("MRForwarder: Fail to route directive. Invalid multiroom session id: " << Directive::convertToJsonString(directive));
        }
    } else {
        YIO_LOG_WARN("MRForwarder: Ignore directive. The target device is undefined: " << Directive::convertToJsonString(directive));
    }
}
