#include "multiroom_capability.h"
#include "multiroom_directives.h"
#include "multiroom_endpoint.h"
#include "multiroom_log.h"

#include <yandex_io/services/aliced/capabilities/alice_capability/directives/alice_request_directive.h>

#include <yandex_io/libs/base/directives.h>
#include <yandex_io/libs/base/named_callback_queue.h>
#include <yandex_io/libs/configuration/configuration.h>
#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/debug.h>

#include <util/system/yassert.h>

YIO_DEFINE_LOG_MODULE("multiroom_capability");

using namespace quasar;
using namespace YandexIO;

namespace {
    const std::string MULTIROOM_INTERNAL_UPDATE_ACTIVITY = "multiroom_internal_update_activity";
} // namespace

MultiroomCapability::MultiroomCapability(
    std::shared_ptr<quasar::ICallbackQueue> worker,
    std::shared_ptr<YandexIO::IDevice> device,
    std::shared_ptr<ipc::IIpcFactory> ipcFactory,
    std::shared_ptr<YandexIO::SDKInterface> sdk,
    YandexIO::ActivityTracker& activityTracker,
    YandexIO::IDirectiveProcessorWeakPtr directiveProcessor,
    std::shared_ptr<ipc::IConnector> audioClientConnector,
    std::shared_ptr<IClockTowerProvider> clockTowerProvider,
    std::shared_ptr<IGlagolClusterProvider> glagolClusterProvider,
    std::shared_ptr<IMultiroomProvider> multiroomProvider,
    std::shared_ptr<IStereoPairProvider> stereoPairProvider,
    std::shared_ptr<IUserConfigProvider> userConfigProvider)
    : worker_(std::move(worker))
    , activityTracker_(activityTracker)
    , directiveProcessor_(std::move(directiveProcessor))
    , multiroomProvider_(std::move(multiroomProvider))
    , listener_(std::shared_ptr<YandexIO::ActivityTracker::Listener>(this, [](auto* /*p*/) {}))
    , defaultLatency_(std::chrono::milliseconds{tryGetInt64(device->configuration()->getServiceConfig("multiroomd"), "latency_ms", 0)})
    , multiroomSlavePlayer_(std::move(audioClientConnector))
    , multiroomState_(std::make_shared<MultiroomState>())
    , multiroomEndpoint_(std::make_shared<MultiroomEndpoint>(
          worker_,
          std::move(device),
          std::move(ipcFactory),
          std::move(sdk),
          std::move(clockTowerProvider),
          std::move(glagolClusterProvider),
          std::move(stereoPairProvider),
          userConfigProvider))
{
    Y_VERIFY(multiroomProvider_);

    activityTracker_.addListener(listener_);
    multiroomProvider_->multiroomState().connect(
        [this](const auto& state) {
            worker_->add([this, state] { onMultiroomState(state); });
        }, lifetime_);

    if (!localDialogActivity_) {
        multiroomEndpoint_->localDialogActivityChanged(false);
    }

    if (userConfigProvider) {
        userConfigProvider->jsonChangedSignal(IUserConfigProvider::ConfigScope::SYSTEM, "multiroomd").connect([this](const auto& json) {
            static_assert(std::is_same<std::chrono::milliseconds, std::remove_const<decltype(defaultLatency_)>::type>::value);
            auto latency = std::chrono::milliseconds{tryGetInt64(*json, "latency_ms", defaultLatency_.count())};
            multiroomSlavePlayer_.setLatency(latency);
        }, lifetime_);
    }
}

MultiroomCapability::~MultiroomCapability() {
    lifetime_.die();
    activityTracker_.removeListener(listener_);
}

void MultiroomCapability::onAudioClientEvent(const proto::AudioClientEvent& event) {
    multiroomSlavePlayer_.onAudioClientEvent(event);
}

const std::string& MultiroomCapability::getHandlerName() const {
    static const std::string s_name = "MultiroomCapability";
    return s_name;
}

const std::set<std::string>& MultiroomCapability::getSupportedDirectiveNames() const {
    static std::set<std::string> s_names = {
        MULTIROOM_INTERNAL_UPDATE_ACTIVITY,
        Directives::START_MULTIROOM,
        Directives::STOP_MULTIROOM,
        Directives::MULTIROOM_SEMANTIC_FRAME,
        MultiroomDirectives::SLAVE_PLAY,
        MultiroomDirectives::SLAVE_STOP,
        MultiroomDirectives::SLAVE_PAUSE,
    };
    return s_names;
}

void MultiroomCapability::handleDirective(const std::shared_ptr<YandexIO::Directive>& directive) try {
    LOG_DEBUG_MULTIROOM("MultiroomCapability::handleDirective "
                        << "name=" << directive->getData().name
                        << ", ptr=" << directive.get()
                        << ", body: " << convertMessageToDeepJsonString(YandexIO::Directive::convertToDirectiveProtobuf(directive)));

    const auto& data = directive->getData();
    if (directive->is(Directives::START_MULTIROOM)) {
        auto multiroomToken = getString(data.payload, "multiroom_token");
        auto roomDeviceIds = tryGetVector<std::string>(data.payload, "room_device_ids");
        multiroomEndpoint_->startMultiroom(data.requestId, std::move(multiroomToken), std::move(roomDeviceIds));
    } else if (directive->is(Directives::STOP_MULTIROOM)) {
        LOG_DEBUG_MULTIROOM("handleStopMultiroom: vinsRequestId=" << data.requestId << ", payload=" << jsonToString(data.payload, true));
        multiroomEndpoint_->stopMultiroom(data.requestId);
    } else if (directive->is(MULTIROOM_INTERNAL_UPDATE_ACTIVITY)) {
        updateActivities();
    } else if (directive->is(MultiroomDirectives::SLAVE_PLAY)) {
        auto jsonString = jsonToString(directive->getData().payload, true);
        auto multiroomBroadcast = convertJsonToProtobuf<proto::MultiroomBroadcast>(jsonString);
        if (!multiroomBroadcast) {
            YIO_LOG_ERROR_EVENT("MultiroomCapability.InvalidDirectiveParams",
                                "Fail to execute MultiroomDirectives::SLAVE_PLAY directive. Indalid params: payload=" << jsonString);
        } else {
            auto mpd = multiroomSlavePlayer_.play(*multiroomBroadcast);
            history_[directive] = mpd;
        }
    } else if (directive->is(MultiroomDirectives::SLAVE_STOP)) {
        auto multiroomSessionId = getString(directive->getData().payload, "multiroom_session_id");
        auto reason = getString(directive->getData().payload, "reason");
        multiroomSlavePlayer_.stop(multiroomSessionId, reason);
    } else if (directive->is(MultiroomDirectives::SLAVE_PAUSE)) {
        auto multiroomSessionId = getString(directive->getData().payload, "multiroom_session_id");
        auto reason = getString(directive->getData().payload, "reason");
        multiroomSlavePlayer_.pause(multiroomSessionId, reason);
    } else if (directive->is(Directives::MULTIROOM_SEMANTIC_FRAME)) {
        auto body = tryGetJson(directive->getData().payload, "body");
        YIO_LOG_INFO("Multiroom semantic frame: " << jsonToString(body, true));
        if (body.isNull() || !body.isObject()) {
            throw std::runtime_error("Invalid 'body' argument");
        }
        auto request = VinsRequest::createEventRequest(body, VinsRequest::createSoftwareDirectiveEventSource());
        request->setIsSilent(true);
        auto directive = std::make_shared<AliceRequestDirective>(std::move(request), nullptr, true);
        if (auto dp = directiveProcessor_.lock()) {
            dp->addDirectives({std::move(directive)});
        }
    } else {
        throw std::runtime_error("Unsupported directive");
    }

} catch (const std::exception& ex) {
    YIO_LOG_ERROR_EVENT("MultiroomCapability.Exception",
                        "Fail to execute directive \"" << directive->getData().name << "\": " << ex.what());
}

void MultiroomCapability::cancelDirective(const std::shared_ptr<YandexIO::Directive>& directive)
{
    LOG_DEBUG_MULTIROOM("cancelDirective: name=" << directive->getData().name << ", ptr=" << directive.get());
    auto it = history_.find(directive);
    if (it != history_.end()) {
        LOG_INFO_MULTIROOM("Cancel directive to play multiroom: multiroomSessionId=" << it->second.multiroomSessionId << ", playerId=" << it->second.playerId);
        auto multiroomSessionId = std::move(it->second.multiroomSessionId);
        auto reason = "Multiroom capability cancel directive: playerId=" + it->second.playerId;
        history_.erase(it);
        multiroomSlavePlayer_.stop(multiroomSessionId, reason);
    }
}

void MultiroomCapability::prefetchDirective(const std::shared_ptr<YandexIO::Directive>& /*directive*/)
{
    // do nothing
}

void MultiroomCapability::onActivityAdded(const YandexIO::IActivityConstPtr& activity) {
    if (activity->isLocal() && activity->getAudioChannel() == proto::DIALOG_CHANNEL) {
        if (++localDialogActivity_ == 1) {
            multiroomEndpoint_->localDialogActivityChanged(true);
        }
    }
}

void MultiroomCapability::onActivityRemoved(const YandexIO::IActivityConstPtr& activity) {
    if (activity->isLocal() && activity->getAudioChannel() == proto::DIALOG_CHANNEL) {
        if (--localDialogActivity_ == 0) {
            multiroomEndpoint_->localDialogActivityChanged(false);
        }
    }
}

void MultiroomCapability::updateActivities()
{
    for (const auto& [action, activity] : unprocessedActivities_) {
        if (action == ActivityAction::ADD) {
            activityTracker_.addActivity(activity);
        } else {
            activityTracker_.removeActivity(activity);
        }
    }
    unprocessedActivities_.clear();
}

void MultiroomCapability::onMultiroomState(const std::shared_ptr<const MultiroomState>& multiroomState)
{
    multiroomState_ = multiroomState;

    bool fChanges = false;
    auto changeActivity =
        [&](ActivityAction activtyAction, const YandexIO::ChannelActivityPtr& activity) {
            fChanges = true;
            unprocessedActivities_.push_back(std::make_pair(activtyAction, activity));
        };

    bool hasDialogActivity = !multiroomState_->dialogDeviceIds.empty() && multiroomState_->broadcast && multiroomState_->broadcast->playingState == MultiroomState::PlayingState::PLAYING;
    if (!hasDialogActivity) {
        if (dialogActivity_) {
            changeActivity(ActivityAction::REMOVE, dialogActivity_);
            dialogActivity_ = nullptr;
        }
    } else {
        if (!dialogActivity_) {
            dialogActivity_ = std::make_shared<YandexIO::ChannelActivity>("MultiroomDialog", proto::AudioChannel::DIALOG_CHANNEL, false);
            changeActivity(ActivityAction::ADD, dialogActivity_);
        }
    }

    if (multiroomState_->mode != MultiroomState::Mode::SLAVE) {
        if (slaveActivity_) {
            changeActivity(ActivityAction::REMOVE, slaveActivity_);
            slaveActivity_ = nullptr;
        }
    } else {
        if (!slaveActivity_) {
            slaveActivity_ = std::make_shared<YandexIO::ChannelActivity>("MultiroomSlave", proto::AudioChannel::CONTENT_CHANNEL, true);
            changeActivity(ActivityAction::ADD, slaveActivity_);
        }
    }

    if (fChanges) {
        auto directive = YandexIO::Directive::createLocalAction(MULTIROOM_INTERNAL_UPDATE_ACTIVITY);
        if (auto dp = directiveProcessor_.lock()) {
            dp->addDirectives({std::move(directive)});
        }
    }
}
