#include "multiroom_common.h"

#include <yandex_io/interfaces/multiroom/i_multiroom_provider.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/cryptography/digest.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/protobuf_utils/debug.h>
#include <yandex_io/protos/quasar_proto.pb.h>

#include <iomanip>

using namespace quasar;

namespace {
    int64_t getNowMs()
    {
        const auto now = std::chrono::system_clock::now();
        return static_cast<int64_t>(std::chrono::duration_cast<std::chrono::milliseconds>(now.time_since_epoch()).count());
    }

    template <class T>
    std::optional<MultiroomShortAppState::Normalization> getNormalization(const T& parent)
    {
        std::optional<MultiroomShortAppState::Normalization> res;
        if (parent.has_normalization()) {
            res = MultiroomShortAppState::Normalization{
                .truePeak = parent.normalization().true_peak(),
                .integratedLoudness = parent.normalization().integrated_loudness(),
                .targetLufs = parent.normalization().target_lufs(),
            };
        }
        return res;
    }

} // namespace

Json::Value quasar::multiroomBroadcastToJson(const proto::MultiroomBroadcast& multiroomBroadcast)
{
    Json::Value json;
    json["device_id"] = multiroomBroadcast.device_id();
    json["session_timestamp_ms"] = static_cast<int64_t>(multiroomBroadcast.session_timestamp_ms());
    json["vins_request_id"] = multiroomBroadcast.vins_request_id();
    json["state"] = (int)multiroomBroadcast.state();
    if (multiroomBroadcast.has_multiroom_token()) {
        json["multiroom_token"] = multiroomBroadcast.multiroom_token();
    }

    if (multiroomBroadcast.room_device_ids_size() > 0) {
        json["room_device_ids"] = Json::arrayValue;
        for (const auto& room_device_id : multiroomBroadcast.room_device_ids()) {
            json["room_device_ids"].append(room_device_id);
        }
    }

    if (multiroomBroadcast.has_net_audio_clock_host()) {
        json["net_audio_clock_host"] = multiroomBroadcast.net_audio_clock_host();
    }

    if (multiroomBroadcast.has_net_audio_clock_port())
    {
        json["net_audio_clock_port"] = multiroomBroadcast.net_audio_clock_port();
    }

    if (multiroomBroadcast.has_net_audio_clock_id())
    {
        json["net_audio_clock_id"] = multiroomBroadcast.net_audio_clock_id();
    }

    if (multiroomBroadcast.has_multiroom_params()) {
        json["multiroom_params"] = multiroomParamsToJson(multiroomBroadcast.multiroom_params());
    }
    json["version"] = (int)multiroomBroadcast.version();
    return json;
}

proto::MultiroomBroadcast quasar::jsonToMultiroomBroadcast(const Json::Value& jMultiroomBroadcast)
{
    proto::MultiroomBroadcast multiroomBroadcast;
    try {
        multiroomBroadcast.set_device_id(getString(jMultiroomBroadcast, "device_id"));
    } catch (...) {
    }
    try {
        multiroomBroadcast.set_session_timestamp_ms(getInt64(jMultiroomBroadcast, "session_timestamp_ms"));
    } catch (...) {
    }
    try {
        multiroomBroadcast.set_vins_request_id(getString(jMultiroomBroadcast, "vins_request_id"));
    } catch (...) {
    }
    try {
        multiroomBroadcast.set_state(static_cast<decltype(multiroomBroadcast.state())>(getInt(jMultiroomBroadcast, "state")));
    } catch (...) {
    }
    try {
        multiroomBroadcast.set_multiroom_token(getString(jMultiroomBroadcast, "multiroom_token"));
    } catch (...) {
    }
    try {
        if (jMultiroomBroadcast.isMember("room_device_ids")) {
            const auto& jRoomDeviceIds = jMultiroomBroadcast["room_device_ids"];
            if (jRoomDeviceIds.isArray()) {
                for (const auto& element : jRoomDeviceIds) {
                    multiroomBroadcast.add_room_device_ids(element.asString());
                }
            }
        }
    } catch (...) {
    }
    try {
        multiroomBroadcast.set_net_audio_clock_host(getString(jMultiroomBroadcast, "net_audio_clock_host"));
    } catch (...) {
    }
    try {
        multiroomBroadcast.set_net_audio_clock_port(getInt(jMultiroomBroadcast, "net_audio_clock_port"));
    } catch (...) {
    }
    try {
        multiroomBroadcast.set_net_audio_clock_id(getString(jMultiroomBroadcast, "net_audio_clock_id"));
    } catch (...) {
    }
    if (jMultiroomBroadcast.isMember("multiroom_params")) {
        multiroomBroadcast.mutable_multiroom_params()->CopyFrom(jsonToMultiroomParams(jMultiroomBroadcast["multiroom_params"]));
    }
    try {
        multiroomBroadcast.set_version(getUInt32(jMultiroomBroadcast, "version"));
    } catch (...) {
    }
    return multiroomBroadcast;
}

Json::Value quasar::multiroomParamsToJson(const proto::MultiroomParams& multiroomParams)
{
    Json::Value json;
    if (multiroomParams.has_url()) {
        json["url"] = multiroomParams.url();
    }
    if (multiroomParams.has_basetime_ns()) {
        json["basetime_ns"] = static_cast<int64_t>(multiroomParams.basetime_ns());
    }
    if (multiroomParams.has_latency_ns()) {
        json["latency_ns"] = static_cast<int64_t>(multiroomParams.latency_ns());
    }
    if (multiroomParams.has_normalization()) {
        if (multiroomParams.normalization().has_true_peak()) {
            json["normalization"]["true_peak"] = multiroomParams.normalization().true_peak();
        }
        if (multiroomParams.normalization().has_integrated_loudness()) {
            json["normalization"]["integrated_loudness"] = multiroomParams.normalization().integrated_loudness();
        }
        if (multiroomParams.normalization().has_target_lufs()) {
            json["normalization"]["target_lufs"] = multiroomParams.normalization().target_lufs();
        }
    }

    if (multiroomParams.has_position_ns()) {
        json["position_ns"] = static_cast<int64_t>(multiroomParams.position_ns());
    }
    if (multiroomParams.has_music_params()) {
        if (multiroomParams.music_params().has_current_track_id()) {
            json["music_params"]["current_track_id"] = multiroomParams.music_params().current_track_id();
        }
        if (multiroomParams.music_params().has_json_track_info()) {
            json["music_params"]["json_track_info"] = multiroomParams.music_params().json_track_info();
        }
        if (multiroomParams.music_params().has_uid()) {
            json["music_params"]["uid"] = multiroomParams.music_params().uid();
        }
        if (multiroomParams.music_params().has_timestamp_ms()) {
            json["music_params"]["timestamp_ms"] = multiroomParams.music_params().timestamp_ms();
        }
        if (multiroomParams.music_params().has_session_id()) {
            json["music_params"]["session_id"] = multiroomParams.music_params().session_id();
        }
        if (multiroomParams.music_params().has_is_paused()) {
            json["music_params"]["is_paused"] = multiroomParams.music_params().is_paused();
        }
    }

    if (multiroomParams.has_audio_params()) {
        if (multiroomParams.audio_params().has_audio()) {
            json["audio_params"]["offset_ms"] = multiroomParams.audio_params().audio().position_sec() * 1000;
            json["audio_params"]["played_ms"] = multiroomParams.audio_params().audio().played_sec() * 1000;
            json["audio_params"]["duration_ms"] = multiroomParams.audio_params().audio().duration_sec() * 1000;
            json["audio_params"]["id"] = multiroomParams.audio_params().audio().id();
            json["audio_params"]["type"] = multiroomParams.audio_params().audio().type();
            json["audio_params"]["title"] = multiroomParams.audio_params().audio().metadata().title();
            json["audio_params"]["subtitle"] = multiroomParams.audio_params().audio().metadata().subtitle();
            json["audio_params"]["art_image_url"] = multiroomParams.audio_params().audio().metadata().art_image_url();
            json["audio_params"]["json_context"] = multiroomParams.audio_params().audio().context();
        }
        if (multiroomParams.audio_params().has_last_play_timestamp()) {
            json["audio_params"]["last_play_timestamp"] = multiroomParams.audio_params().last_play_timestamp();
        }
        if (multiroomParams.audio_params().has_last_stop_timestamp()) {
            json["audio_params"]["last_stop_timestamp"] = multiroomParams.audio_params().last_stop_timestamp();
        }
        if (multiroomParams.audio_params().has_state()) {
            json["audio_params"]["state"] = static_cast<int>(multiroomParams.audio_params().state());
        }
    }
    return json;
}

proto::MultiroomParams quasar::jsonToMultiroomParams(const Json::Value& jMultiroomParams)
{
    proto::MultiroomParams multiroomParams;
    try {
        multiroomParams.set_url(getString(jMultiroomParams, "url"));
    } catch (...) {
    }
    try {
        multiroomParams.set_basetime_ns(getInt64(jMultiroomParams, "basetime_ns"));
    } catch (...) {
    }
    try {
        multiroomParams.set_position_ns(getInt64(jMultiroomParams, "position_ns"));
    } catch (...) {
    }
    try {
        multiroomParams.set_latency_ns(getInt64(jMultiroomParams, "latency_ns"));
    } catch (...) {
    }
    if (jMultiroomParams.isMember("normalization")) {
        const auto& n = jMultiroomParams["normalization"];
        try {
            auto tp = getDouble(n, "true_peak");
            auto il = getDouble(n, "integrated_loudness");
            auto tl = getDouble(n, "target_lufs");
            multiroomParams.mutable_normalization()->set_true_peak(tp);
            multiroomParams.mutable_normalization()->set_integrated_loudness(il);
            multiroomParams.mutable_normalization()->set_target_lufs(tl);
        } catch (...) {
        }
    }
    if (jMultiroomParams.isMember("music_params")) {
        const auto& musicParams = jMultiroomParams["music_params"];
        try {
            multiroomParams.mutable_music_params()->set_current_track_id(getString(musicParams, "current_track_id"));
        } catch (...) {
        }
        try {
            multiroomParams.mutable_music_params()->set_json_track_info(getString(musicParams, "json_track_info"));
        } catch (...) {
        }
        try {
            multiroomParams.mutable_music_params()->set_uid(getString(musicParams, "uid"));
        } catch (...) {
        }
        try {
            multiroomParams.mutable_music_params()->set_timestamp_ms(getInt64(musicParams, "timestamp_ms"));
        } catch (...) {
        }
        try {
            multiroomParams.mutable_music_params()->set_session_id(getString(musicParams, "session_id"));
        } catch (...) {
        }
        try {
            multiroomParams.mutable_music_params()->set_is_paused(getBool(musicParams, "is_paused"));
        } catch (...) {
        }
    }
    if (jMultiroomParams.isMember("audio_params")) {
        const auto& audioParams = jMultiroomParams["audio_params"];
        try {
            multiroomParams.mutable_audio_params()->mutable_audio()->set_position_sec(getInt64(audioParams, "offset_ms") / 1000);
        } catch (...) {
        }
        try {
            multiroomParams.mutable_audio_params()->mutable_audio()->set_played_sec(getInt64(audioParams, "played_ms") / 1000);
        } catch (...) {
        }
        try {
            multiroomParams.mutable_audio_params()->mutable_audio()->set_duration_sec(getInt64(audioParams, "duration_ms") / 1000);
        } catch (...) {
        }
        try {
            multiroomParams.mutable_audio_params()->set_last_play_timestamp(getInt64(audioParams, "last_play_timestamp"));
        } catch (...) {
        }
        try {
            multiroomParams.mutable_audio_params()->set_last_stop_timestamp(getInt64(audioParams, "last_stop_timestamp"));
        } catch (...) {
        }
        try {
            multiroomParams.mutable_audio_params()->set_state(static_cast<proto::AudioClientState>(getInt64(audioParams, "state")));
        } catch (...) {
        }
        try {
            multiroomParams.mutable_audio_params()->mutable_audio()->set_id(getString(audioParams, "id"));
        } catch (...) {
        }
        try {
            multiroomParams.mutable_audio_params()->mutable_audio()->set_type(getString(audioParams, "type"));
        } catch (...) {
        }
        try {
            multiroomParams.mutable_audio_params()->mutable_audio()->mutable_metadata()->set_title(getString(audioParams, "title"));
        } catch (...) {
        }
        try {
            multiroomParams.mutable_audio_params()->mutable_audio()->mutable_metadata()->set_subtitle(getString(audioParams, "subtitle"));
        } catch (...) {
        }
        try {
            multiroomParams.mutable_audio_params()->mutable_audio()->mutable_metadata()->set_art_image_url(getString(audioParams, "art_image_url"));
        } catch (...) {
        }
        try {
            multiroomParams.mutable_audio_params()->mutable_audio()->set_context(getString(audioParams, "json_context"));
        } catch (...) {
        }
    }
    return multiroomParams;
}

std::string quasar::appStateHash(const proto::AppState& appState)
{
    return shortUtf8DebugString(appState);
}

bool MultiroomShortAppState::isMaster() const {
    return (player == Player::AUDIO_CLIENT || player == Player::YANDEX_MUSIC) &&
           sync &&
           !vinsRequestId.empty();
}

bool MultiroomShortAppState::isSlave() const {
    return (player == Player::MULTIROOM);
}

const char* MultiroomShortAppState::playerName() const {
    switch (player) {
        case Player::OTHER:
            return "OTHER";
        case Player::AUDIO_CLIENT:
            return "AUDIO_CLIENT";
        case Player::YANDEX_MUSIC:
            return "YANDEX_MUSIC";
        case Player::MULTIROOM:
            return "MULTIROOM";
        case Player::RADIO:
            return "RADIO";
        case Player::BLUETOOTH:
            return "BLUETOOTH";
        case Player::VIDEO:
            return "VIDEO";
    }
    return "INVALID";
}

MultiroomShortAppState quasar::makeMultiroomShortAppState(const proto::AppState& appState)
{
    MultiroomShortAppState multiroomShortAppState;
    int64_t lastTimestamp = 0;

    if (appState.has_audio_player_state()) {
        auto state = appState.audio_player_state().state();
        if (state != proto::AudioClientState::FAILED && appState.audio_player_state().has_audio()) {
            bool sync =
                appState.audio_player_state().has_audio() &&
                appState.audio_player_state().audio().has_url() &&
                appState.audio_player_state().audio().has_basetime_ns() &&
                appState.audio_player_state().audio().has_position_ns();
            bool isPlaying = (state == proto::AudioClientState::PLAYING);
            auto player = appState.audio_player_state().player_descriptor().is_multiroom_slave()
                              ? MultiroomShortAppState::Player::MULTIROOM
                              : MultiroomShortAppState::Player::AUDIO_CLIENT;
            multiroomShortAppState = MultiroomShortAppState{
                .player = player,
                .sync = sync,
                .isPlaying = isPlaying,
                .url = appState.audio_player_state().audio().url(),
                .basetimeNs = appState.audio_player_state().audio().basetime_ns(),
                .positionNs = appState.audio_player_state().audio().position_ns(),
                .normalization = getNormalization(appState.audio_player_state().audio()),
                .multiroomToken = appState.audio_player_state().audio().multiroom_token(),
                .vinsRequestId = (appState.audio_player_state().audio().analytics_context().vins_parent_request_id().empty()
                                      ? appState.audio_player_state().audio().analytics_context().vins_request_id()
                                      : appState.audio_player_state().audio().analytics_context().vins_parent_request_id()),
                .playPauseId = appState.audio_player_state().audio().play_pause_id(),
            };
            lastTimestamp = appState.audio_player_state().last_play_timestamp();
        }
    }

    if (appState.has_music_state()) {
        auto timestamp = (appState.music_state().has_timestamp_ms() ? appState.music_state().timestamp_ms() : getNowMs());
        if (timestamp > lastTimestamp || (!multiroomShortAppState.isPlaying && !appState.music_state().is_paused())) {
            auto jsonTrackInfo = tryParseJsonOrEmpty(appState.music_state().json_track_info());
            bool sync =
                appState.music_state().has_url() &&
                appState.music_state().has_basetime_ns() &&
                appState.music_state().has_position_ns();
            multiroomShortAppState = MultiroomShortAppState{
                .player = MultiroomShortAppState::Player::YANDEX_MUSIC,
                .sync = sync,
                .isPlaying = !appState.music_state().is_paused(),
                .url = appState.music_state().url(),
                .basetimeNs = appState.music_state().basetime_ns(),
                .positionNs = appState.music_state().position_ns(),
                .normalization = getNormalization(appState.music_state()),
                .multiroomToken = tryGetString(jsonTrackInfo, "multiroom_token"),
                .vinsRequestId = appState.music_state().vins_request_id(),
                .playPauseId = appState.music_state().play_pause_id(),
            };
            lastTimestamp = timestamp;
        }
    }

    if (!multiroomShortAppState.isPlaying) {
        if (appState.has_video_state() && !appState.video_state().is_paused() && appState.video_state().video().has_raw()) {
            multiroomShortAppState = MultiroomShortAppState{};
            multiroomShortAppState.isPlaying = true;
            multiroomShortAppState.player = MultiroomShortAppState::Player::VIDEO;
        } else if (appState.has_radio_state() && !appState.radio_state().is_paused()) {
            multiroomShortAppState = MultiroomShortAppState{};
            multiroomShortAppState.isPlaying = true;
            multiroomShortAppState.player = MultiroomShortAppState::Player::RADIO;
        } else if (appState.has_bluetooth_player_state() && !appState.bluetooth_player_state().is_paused()) {
            multiroomShortAppState = MultiroomShortAppState{};
            multiroomShortAppState.isPlaying = true;
            multiroomShortAppState.player = MultiroomShortAppState::Player::BLUETOOTH;
        }
    }

    return multiroomShortAppState;
}

MultiroomShortAppState quasar::makeMultiroomShortAppState(const proto::MultiroomBroadcast& multiroomBroadcast)
{
    MultiroomShortAppState multiroomShortAppState;
    if (multiroomBroadcast.has_multiroom_params()) {
        bool sync =
            multiroomBroadcast.multiroom_params().has_url() &&
            multiroomBroadcast.multiroom_params().has_basetime_ns() &&
            multiroomBroadcast.multiroom_params().has_position_ns();

        if (multiroomBroadcast.multiroom_params().has_music_params()) {
            multiroomShortAppState = MultiroomShortAppState{
                .player = MultiroomShortAppState::Player::YANDEX_MUSIC,
                .sync = sync,
                .isPlaying = !multiroomBroadcast.multiroom_params().music_params().is_paused(),
                .url = multiroomBroadcast.multiroom_params().url(),
                .basetimeNs = multiroomBroadcast.multiroom_params().basetime_ns(),
                .positionNs = multiroomBroadcast.multiroom_params().position_ns(),
                .normalization = getNormalization(multiroomBroadcast.multiroom_params()),
                .multiroomToken = multiroomBroadcast.multiroom_token(),
                .vinsRequestId = multiroomBroadcast.vins_request_id(),
                .playPauseId = "",
            };
        } else if (multiroomBroadcast.multiroom_params().has_audio_params()) {
            bool playing =
                multiroomBroadcast.multiroom_params().audio_params().state() == proto::AudioClientState::BUFFERING ||
                multiroomBroadcast.multiroom_params().audio_params().state() == proto::AudioClientState::PLAYING;

            multiroomShortAppState = MultiroomShortAppState{
                .player = MultiroomShortAppState::Player::AUDIO_CLIENT,
                .sync = sync,
                .isPlaying = playing,
                .url = multiroomBroadcast.multiroom_params().url(),
                .basetimeNs = multiroomBroadcast.multiroom_params().basetime_ns(),
                .positionNs = multiroomBroadcast.multiroom_params().position_ns(),
                .normalization = getNormalization(multiroomBroadcast.multiroom_params()),
                .multiroomToken = multiroomBroadcast.multiroom_token(),
                .vinsRequestId = multiroomBroadcast.vins_request_id(),
                .playPauseId = "",
            };
        }
    }
    return multiroomShortAppState;
}

std::string quasar::multiroomSessionId(const proto::MultiroomBroadcast& multiroomBroadcast)
{
    std::string result;
    result.reserve(
        multiroomBroadcast.device_id().size() +
        (multiroomBroadcast.has_multiroom_token() ? multiroomBroadcast.multiroom_token().size() : 0) +
        24);
    result += multiroomBroadcast.device_id();
    result += ":";
    if (multiroomBroadcast.has_multiroom_token()) {
        result += multiroomBroadcast.multiroom_token();
    }
    result += ":";
    result += std::to_string(static_cast<int64_t>(multiroomBroadcast.session_timestamp_ms()));
    return result;
}

std::optional<MultiroomSessionId> quasar::tryPasrseMultiroomSessionId(const std::string& text)
{
    auto parts = split(text, ":");
    if (parts.size() != 3 ||
        parts[0].empty() ||
        parts[1].empty())
    {
        return std::nullopt;
    }

    MultiroomSessionId multiroomSessionId;
    multiroomSessionId.deviceId = std::move(parts[0]);
    multiroomSessionId.multiroomToken = std::move(parts[1]);
    try {
        multiroomSessionId.sessionTimestampMs = std::stoll(parts[2]);
    } catch (...) {
        return std::nullopt;
    }
    return std::optional<MultiroomSessionId>(std::move(multiroomSessionId));
}

std::string quasar::makeMultiroomSessionId(const proto::MultiroomBroadcast& multiroomBroadcast)
{
    return makeMultiroomSessionId(
        multiroomBroadcast.device_id(),
        multiroomBroadcast.multiroom_token(),
        multiroomBroadcast.session_timestamp_ms());
}

std::string quasar::makeMultiroomTrackId(const proto::MultiroomBroadcast& multiroomBroadcast)
{
    return makeMultiroomTrackId(
        multiroomBroadcast.multiroom_params().url(),
        multiroomBroadcast.multiroom_params().basetime_ns(),
        multiroomBroadcast.multiroom_params().position_ns());
}

namespace {
    Json::Value jsonMultiroomBroadcastCompact(const proto::MultiroomBroadcast& message)
    {
        auto json = multiroomBroadcastToJson(message);
        if (json.isMember("session_timestamp_ms")) {
            if (json["session_timestamp_ms"].asInt64() == 0) {
                json.removeMember("state");
            } else {
                json["ts"] = std::move(json["session_timestamp_ms"]);
            }
            json.removeMember("session_timestamp_ms");
        }

        if (json.isMember("state")) {
            switch (tryGetInt(json, "state", -1)) {
                case (int)proto::MultiroomBroadcast::NONE:
                    json["state"] = "NONE";
                    break;
                case (int)proto::MultiroomBroadcast::PAUSED:
                    json["state"] = "PAUSED";
                    break;
                case (int)proto::MultiroomBroadcast::PLAYING:
                    json["state"] = "PLAYING";
                    break;
                default:
                    json["state"] = "-";
                    break;
            }
        }

        json["clk_addr"] = tryGetString(json, "net_audio_clock_host") + ":" + std::to_string(tryGetInt(json, "net_audio_clock_port", 0));
        json.removeMember("net_audio_clock_host");
        json.removeMember("net_audio_clock_port");
        if (json.isMember("vins_request_id")) {
            json["vins"] = std::move(json["vins_request_id"]);
            json.removeMember("vins_request_id");
        }

        if (json.isMember("multiroom_params")) {
            auto& multiroomParams = json["multiroom_params"];
            bool isPlaying = false;
            if (multiroomParams.isMember("music_params")) {
                isPlaying = !message.multiroom_params().music_params().is_paused();
                multiroomParams["music_params"] = (isPlaying ? "[MusicPlaying]" : "[MusicPaused]");
            }
            if (multiroomParams.isMember("audio_params")) {
                isPlaying =
                    message.multiroom_params().audio_params().state() == proto::AudioClientState::BUFFERING ||
                    message.multiroom_params().audio_params().state() == proto::AudioClientState::PLAYING;
                multiroomParams["audio_params"] = (isPlaying ? "[AudioPlaying]" : "[AudioPaused]");
            }
            auto url = multiroomParams["url"].asString();
            if (!url.empty()) {
                multiroomParams["url"] = "https://MD5(" + calcMD5Digest(url) + ")";
            } else if (isPlaying) {
                multiroomParams["url"] = "<empty>";
            }
            if (multiroomParams.isMember("basetime_ns")) {
                multiroomParams["base"] = std::move(multiroomParams["basetime_ns"]);
                multiroomParams.removeMember("basetime_ns");
            }
            if (multiroomParams.isMember("position_ns")) {
                multiroomParams["pos"] = std::move(multiroomParams["position_ns"]);
                multiroomParams.removeMember("position_ns");
            }

            if (multiroomParams.isMember("normalization")) {
                const auto& n = multiroomParams["normalization"];
                std::stringstream str;
                str << "tp:" << std::fixed << std::setprecision(2) << tryGetDouble(n, "true_peak", std::numeric_limits<double>::quiet_NaN())
                    << ", il:" << std::fixed << std::setprecision(2) << tryGetDouble(n, "integrated_loudness", std::numeric_limits<double>::quiet_NaN());
                multiroomParams["norm"] = str.str();
                multiroomParams.removeMember("normalization");
            }
        }

        if (message.has_master_hints() && message.master_hints().quiet_audio_focus()) {
            json["master_hints"]["quiet_audio_focus"] = true;
        }
        json.removeMember("room_device_ids");

        json["v"] = json["version"]; // @@@
        json.removeMember("version");
        return json;
    }
} // namespace

std::string quasar::printMultiroomBroadcastCompact(const proto::MultiroomBroadcast& message)
{
    return jsonToString(jsonMultiroomBroadcastCompact(message), true);
}

namespace {
    Json::Value jsonMultiroomStateCompact(const proto::MultiroomState& message)
    {
        Json::Value multiroomState;
        multiroomState["device_id"] = message.device_id();
        multiroomState["mode"] = proto::MultiroomState::Mode_Name(message.mode());
        multiroomState["ip"] = (message.has_ip_address() ? message.ip_address() : "<UNDEFINED>");
        multiroomState["clk_id"] = message.net_audio_clock_id();
        multiroomState["clk_addr"] = message.net_audio_clock_host() + ":" + std::to_string(message.net_audio_clock_port());

        if (message.slave_clock_syncing()) {
            multiroomState["clock_syncing"] = true;
        }
        if (message.has_slave_sync_level()) {
            multiroomState["sync_level"] = proto::MultiroomState::SlaveSyncLevel_Name(message.slave_sync_level());
        }
        if (message.has_seqnum()) {
            multiroomState["seq"] = static_cast<int64_t>(message.seqnum());
        }
        for (const auto& peer : message.peers()) {
            multiroomState["peers"].append(peer.device_id());
        }
        if (message.has_multiroom_broadcast()) {
            multiroomState["bcast"] = jsonMultiroomBroadcastCompact(message.multiroom_broadcast());
        }
        if (message.dialog_device_ids_size() > 0) {
            multiroomState["mute"] = Json::arrayValue;
            for (const auto& deviceId : message.dialog_device_ids()) {
                multiroomState["mute"].append(deviceId);
            }
        }

        return multiroomState;
    }
} // namespace

std::string quasar::printMultiroomStateCompact(const proto::MultiroomState& message)
{
    return jsonToString(jsonMultiroomStateCompact(message), true);
}

std::string quasar::printClockTimestamp(int64_t timestampNs)
{
    std::string result;
    int64_t sign = (timestampNs < 0 ? -1 : 1);
    timestampNs = std::abs(timestampNs);
    for (int64_t p = 1, index = 0; timestampNs > p; p *= 10, ++index) {
        if (index) {
            if (index == 9) {
                result += ",";
            } else if (index % 3 == 0) {
                result += ".";
            }
        }
        result += '0' + ((timestampNs / p) % 10);
    }
    if (sign < 0) {
        result += '-';
    }
    std::reverse(result.begin(), result.end());
    return result;
}

std::string quasar::printMultiroomShortAppState(const MultiroomShortAppState& shortAppState)
{
    Json::Value json;
    json["player"] = shortAppState.playerName();
    json["sync"] = shortAppState.sync;
    json["isPlaying"] = shortAppState.isPlaying;
    json["url"] = shortAppState.url;
    json["basetimeNs"] = shortAppState.basetimeNs;
    json["positionNs"] = shortAppState.positionNs;
    json["multiroomToken"] = shortAppState.multiroomToken;
    json["vinsRequestId"] = shortAppState.vinsRequestId;
    json["playPauseId"] = shortAppState.playPauseId;
    json["isMaster"] = shortAppState.isMaster();
    json["isSlave"] = shortAppState.isSlave();
    return jsonToString(json, true);
}

std::string quasar::hashMultiroomBroadcast(const proto::MultiroomBroadcast& message)
{
    return printMultiroomBroadcastCompact(message);
}

std::string quasar::hashMultiroomState(const proto::MultiroomState& message)
{
    auto json = jsonMultiroomStateCompact(message);
    json.removeMember("clk");
    json.removeMember("seq");
    return jsonToString(json, true);
}
