#include "multiroom_endpoint.h"

#include "multiroom_common.h"
#include "multiroom_log.h"

#include <yandex_io/libs/base/directives.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/cryptography/digest.h>
#include <yandex_io/libs/glagol_sdk/avahi_wrapper/avahi_browse_client.h>
#include <yandex_io/libs/ipc/datacratic/public.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/net/network_interfaces.h>
#include <yandex_io/libs/protobuf_utils/debug.h>
#include <yandex_io/protos/model_objects.pb.h>
#include <yandex_io/protos/quasar_proto.pb.h>
#include <yandex_io/services/aliced/capabilities/multiroom_capability/multiroom_directives.h>
#include <yandex_io/capabilities/device_state/converters/converters.h>

#include <stdexcept>
#include <string_view>

YIO_DEFINE_LOG_MODULE("multiroom");

using namespace quasar;

namespace {
    constexpr const char* const STEREO_PAIR_MULTIROOM_TOKEN = "cafeabba-0000-0000-8db6-5889091c839e";

    [[maybe_unused]] constexpr uint32_t MULTIROOM_TOKEN_VERSION = 4;
    [[maybe_unused]] constexpr uint32_t MULTIROOM_BROADCAST_VERSION = 3;
    [[maybe_unused]] constexpr uint32_t MULTIROOM_SLAVE_AUDIO_CLIENT_VERSION = 3;
    [[maybe_unused]] constexpr uint32_t MULTIROOM_STATE_VERSION = 2;
    constexpr std::chrono::seconds MULTIROOM_PENDING_TIMEOUT{5};
    constexpr std::chrono::seconds BROADCAST_PROCESSOR_SHORT_DELAY{1};
    constexpr std::chrono::seconds BROADCAST_PROCESSOR_LONG_DELAY{5};
    constexpr std::chrono::seconds METRICA_REPORTER_PERIOD{60};
    constexpr std::chrono::seconds NO_AUDIO_FOCUS_TIMEOUT{10};
    constexpr std::chrono::seconds PEER_KEEP_ALIVE_TIMEOUT{180};
    constexpr std::chrono::seconds PEER_CHECK_KEEP_ALIVE_PERIOD{15};
    constexpr std::chrono::seconds PEER_SEND_STATUS_PERIOD_MASTER{15};
    constexpr std::chrono::seconds PEER_SEND_STATUS_PERIOD_NON_MASTER{60};
    constexpr std::chrono::seconds SYNC_CLOCK_EXPIRED_PERIOD{60};
    constexpr std::chrono::seconds RESET_INACTIVE_MASTER_STATE_PERIOD{30 * 60};
    constexpr std::chrono::milliseconds MAX_LATENCY{3000};
    constexpr std::chrono::minutes VERBOSE_LOG_PERIOD{5};

    const char* modeName(MultiroomEndpoint::Mode mode)
    {
        switch (mode) {
            case MultiroomEndpoint::Mode::NONE:
                return "NONE";
            case MultiroomEndpoint::Mode::MASTER:
                return "MASTER";
            case MultiroomEndpoint::Mode::SLAVE:
                return "SLAVE";
        }
        return "UNDEFINED";
    }

} // namespace

const std::string MultiroomEndpoint::SERVICE_NAME = "multiroomd";

class MultiroomEndpoint::Peer2PeerConnector {
public:
    Peer2PeerConnector(const std::shared_ptr<IGlagolClusterProvider>& glagolClusterProvider, std::string deviceId)
        : glagolClusterProvider_(glagolClusterProvider)
        , deviceIds_(std::vector<std::string>{std::move(deviceId)})
    {
    }

    void sendMessage(const ipc::SharedMessage& message)
    {
        if (message->has_multiroom_state()) {
            LOG_DEBUG_MULTIROOM("Send multiroom_state via glagol to " << deviceIds_.front() << ": " << printMultiroomStateCompact(message->multiroom_state()));
        } else if (message->has_multiroom_broadcast()) {
            LOG_DEBUG_MULTIROOM("Send multiroom_broadcast via glagol to " << deviceIds_.front() << ": " << printMultiroomBroadcastCompact(message->multiroom_broadcast()));
        } else {
            LOG_DEBUG_MULTIROOM("Send message via glagol to " << deviceIds_.front() << ": " << shortUtf8DebugString(*message));
        }
        glagolClusterProvider_->send(deviceIds_, SERVICE_NAME, message);
    }

private:
    std::shared_ptr<IGlagolClusterProvider> glagolClusterProvider_;
    std::vector<std::string> deviceIds_;
};

MultiroomEndpoint::MultiroomEndpoint(
    std::shared_ptr<ICallbackQueue> lifecycle,
    std::shared_ptr<YandexIO::IDevice> device,
    std::shared_ptr<ipc::IIpcFactory> ipcFactory,
    std::shared_ptr<YandexIO::SDKInterface> sdk,
    std::shared_ptr<IClockTowerProvider> clockTowerProvider,
    std::shared_ptr<IGlagolClusterProvider> glagolClusterProvider,
    std::shared_ptr<IStereoPairProvider> stereoPairProvider,
    std::shared_ptr<IUserConfigProvider> userConfigProvider)
    : lifecycle_(std::move(lifecycle))
    , device_(std::move(device))
    , ipcFactory_(std::move(ipcFactory))
    , sdk_(std::move(sdk))
    , deviceState_(sdk_->getDeviceStateCapability())
    , clockTowerProvider_(std::move(clockTowerProvider))
    , glagolClusterProvider_(std::move(glagolClusterProvider))
    , stereoPairProvider_(std::move(stereoPairProvider))
    , myDeviceId_(device_->deviceId())
    , multiroomServer_(ipcFactory_->createIpcServer("multiroomd"))
    , toAliced_(ipcFactory_->createIpcConnector("aliced"))
    , lastAppState_(std::make_shared<proto::AppState>())
    , clockTowerState_(ClockTowerState::createDefault())
    , stereoPairState_(std::make_shared<StereoPairState>())
    , lastBroadcastedMasterKeepAlive_(std::chrono::steady_clock::now())
    , appStateUpdate_(lifecycle_, UniqueCallback::ReplaceType::REPLACE_FIRST)
    , resetMasterStateCallback_(lifecycle_, UniqueCallback::ReplaceType::INSERT_BACK)
{
    Y_VERIFY(lifecycle_);
    Y_VERIFY(sdk_);
    Y_VERIFY(clockTowerProvider_);
    Y_VERIFY(glagolClusterProvider_);

    clearMaster();

    multiroomServer_->setClientConnectedHandler(
        [this](auto& connection) {
            std::weak_ptr<ipc::IServer::IClientConnection> wConnection = connection.share();
            lifecycle_->add([this, wConnection] {
                sendMasterState(true, wConnection);
                sendMultiroomState(true, wConnection);
            }, lifetime_);
        });
    multiroomServer_->setMessageHandler(
        std::bind(makeSafeCallback(
                      [this](const ipc::SharedMessage& message) {
                          std::string peer;
                          if (message->has_glagol_cluster_sign()) {
                              const auto& deviceId = message->glagol_cluster_sign().from_device_id();
                              if (deviceId == myDeviceId_) {
                                  return; // skip self message
                              }
                              peer = deviceId;
                              if (!peers_.count(deviceId)) {
                                  addPeer(deviceId, false);
                              }
                          } else {
                              peer = "LocalClient";
                          }
                          if (!onClientQuasarMessage(peer, message)) {
                              YIO_LOG_ERROR_EVENT("MultiroomEndpoint.UnexpectedMessage.MultiroomServer", "Fail to process unexpected quasar message: " << shortUtf8DebugString(*message));
                          }
                      }, lifetime_, lifecycle_), std::placeholders::_1));

    lifecycle_->add(
        [this] {
            reconfigureMultirooms();
            multiroomServer_->listenService();
            updateState();
            noAudioFocusInformator();
            keepAlive();
            clockTowerProvider_->clockTowerState().connect(
                [this](const auto& state) {
                    clockTowerState_ = state;
                    sendMasterState(true);
                    sendMultiroomState(true);
                }, lifetime_, lifecycle_);
            glagolClusterProvider_->publicIp().connect(
                [this](const auto& publicIp) {
                    LOG_INFO_MULTIROOM("Public ip changed to " << publicIp);
                    sendMasterState(true);
                    sendMultiroomState(true);
                }, lifetime_, lifecycle_);
            glagolClusterProvider_->deviceList().connect(
                [this](const auto& deviceList) {
                    reconfigurePeers(deviceList);
                }, lifetime_, lifecycle_);
        }, lifetime_);

    toAliced_->setMessageHandler(makeSafeCallback(
        [this](const auto& message) {
            if (message->has_app_state()) {
                auto lastAppState = std::shared_ptr<const proto::AppState>(message.share(), &message->app_state());
                appStateUpdate_.execute([this, lastAppState] { updateState(lastAppState); }, lifetime_);
            }
        }, lifetime_));
    toAliced_->connectToService();

    lifecycle_->addDelayed([this] { metricaReporter(); }, METRICA_REPORTER_PERIOD, lifetime_);

    userConfigProvider->accountDevicesChangedSignal().connect(
        [this](const auto& accountDevices) {
            std::set<std::string> myDeviceIds;
            if (accountDevices) {
                for (const auto& accountDevice : *accountDevices) {
                    myDeviceIds.insert(accountDevice.deviceId);
                }
                if (myDeviceIds != myDeviceIds_) {
                    LOG_DEBUG_MULTIROOM("Update my device list");
                    myDeviceIds_ = std::move(myDeviceIds);
                    if (resetNonAccountPeers()) {
                        sendMasterState(false);
                        sendMultiroomState(false);
                    }
                }
            }
        }, lifetime_, lifecycle_);

    userConfigProvider->jsonChangedSignal(IUserConfigProvider::ConfigScope::SYSTEM, "multiroomd").connect([this](const auto& json) {
        multiroomConfig_ = *json;
        reconfigureMultirooms();
    }, lifetime_, lifecycle_);

    if (stereoPairProvider_) {
        stereoPairProvider_->stereoPairState().connect(
            [this](const auto& state) {
                bool stopMultiroom = false;
                auto oldRole = stereoPairState_->role;
                auto newRole = state->role;
                if (oldRole != newRole) {
                    LOG_INFO_MULTIROOM("Stereo pair role changed from " << StereoPairState::roleName(oldRole) << " to " << StereoPairState::roleName(newRole));
                    if (stereoPairState_->isFollower() != state->isFollower() || stereoPairState_->partnerDeviceId != state->partnerDeviceId) {
                        stopMultiroom = (stereoPairState_->isStereoPair() || state->isStereoPair());
                    }
                }

                stereoPairState_ = state;
                if (stopMultiroom) {
                    if (mode_ == Mode::MASTER) {
                        stopMaster(std::string("Stereo pair role changed from ") + StereoPairState::roleName(oldRole) + " to " + StereoPairState::roleName(newRole));
                    } else if (mode_ == Mode::SLAVE) {
                        stopAllSlaves(std::string("Stereo pair role changed from ") + StereoPairState::roleName(oldRole) + " to " + StereoPairState::roleName(newRole));
                    }
                }
            }, lifetime_, lifecycle_);
    }
}

MultiroomEndpoint::~MultiroomEndpoint()
{
    lifetime_.die();
}

void MultiroomEndpoint::startMultiroom(std::string vinsRequestId, std::string multiroomToken, std::vector<std::string> deviceIdList) {
    Y_ENSURE_THREAD(lifecycle_);

    if (multiroomToken.empty()) {
        YIO_LOG_ERROR_EVENT("MultiroomEndpoint.EmptyMultiroomToken", "Fail to start multiroom session for vins request id " << vinsRequestId << " with empty multiroom token");
        throw std::runtime_error("Empty multiroom token");
    }

    if (deviceIdList.empty()) {
        YIO_LOG_ERROR_EVENT("MultiroomEndpoint.NoDevices", "Fail to start multiroom session for vins request id " << vinsRequestId << " with empty empty device list");
        throw std::runtime_error("Empty multiroom token");
    }

    startMaster(std::move(vinsRequestId), std::move(multiroomToken), std::move(deviceIdList));
    reportEventStartMaster(false);
    updateState();
}

void MultiroomEndpoint::stopMultiroom(std::string vinsRequestId) {
    Y_ENSURE_THREAD(lifecycle_);

    if (mode_ == Mode::MASTER) {
        std::string message = makeString("Stop multiroom vins request id ", vinsRequestId);
        stopMaster(message);
        sendToDirectiveProcessor(Directives::PLAYBACK_PAUSE);
        updateState();
    }
}

void MultiroomEndpoint::localDialogActivityChanged(bool localDialogActivity) {
    Y_ENSURE_THREAD(lifecycle_);

    if (localDialogActivity != localDialogActivity_) {
        LOG_INFO_MULTIROOM("Local dialog activity changed from " << (localDialogActivity_ ? "true" : "false") << " to " << (localDialogActivity ? "true" : "false"));
        localDialogActivity_ = localDialogActivity;

        if (mode_ != Mode::NONE) {
            proto::MultiroomDirective focusDirective;
            focusDirective.mutable_audio_focus()->set_initiator_device_id(TString(myDeviceId_));
            focusDirective.mutable_audio_focus()->set_has_audio_focus(!localDialogActivity_);
            sendDirective(std::move(focusDirective));
        }
    }
}

void MultiroomEndpoint::reconfigureMultirooms() {
    LOG_INFO_MULTIROOM("Reconfigure room set");
    try {
        Y_ENSURE_THREAD(lifecycle_);

        bool multiroomStateChanged = false;

        LOG_INFO_MULTIROOM("Reconfigure multiroom: " << jsonToString(multiroomConfig_));

        const auto& serviceConfig = device_->configuration()->getServiceConfig(SERVICE_NAME);
        const auto defaultLatency = tryGetInt64(serviceConfig, "latency_ms", 0);
        auto latency = std::chrono::milliseconds{tryGetInt64(multiroomConfig_, "latency_ms", defaultLatency)};
        if (std::abs(latency.count()) > MAX_LATENCY.count()) {
            YIO_LOG_ERROR_EVENT("MultiroomEndpoint.InvalidLatencyValue", "Invalid \"latency\" value " << latency.count() << "ms. Value must be in range from " << std::to_string(-MAX_LATENCY.count()) << "ms to " << std::to_string(MAX_LATENCY.count()) << "ms");
            latency = std::chrono::milliseconds{defaultLatency};
        }
        if (latency != latency_.load()) {
            LOG_INFO_MULTIROOM("Latency changed from " << latency_.load().count() << " to " << latency.count());
            latency_ = latency;
            multiroomStateChanged = true;
        }

        const auto defaultWeakStereo = tryGetBool(serviceConfig, "weak_stereo", false);
        auto weakStereo = tryGetBool(multiroomConfig_, "weak_stereo", defaultWeakStereo);
        if (weakStereo_ != weakStereo) {
            LOG_INFO_MULTIROOM("Start stereo pair with weak clock synchronization " << (weakStereo ? "allowed" : "not allowed"));
            weakStereo_ = weakStereo;
        }

        if (multiroomStateChanged) {
            sendMultiroomState(false);
        }

    } catch (const std::exception& ex) {
        YIO_LOG_ERROR_EVENT("MultiroomEndpoint.FailedReconfigure", "Invalid multiroom config: " << ex.what());
    }
}

void MultiroomEndpoint::reconfigurePeers(const std::shared_ptr<const std::vector<std::string>>& deviceList) {
    Y_ENSURE_THREAD(lifecycle_);

    LOG_INFO_MULTIROOM("Reconfigure multiroom peers triggered by \"glagol\"");

    for (const auto& deviceId : *deviceList) {
        if (addPeer(deviceId, true)) {
            sendMasterState(true, deviceId);
            sendMultiroomState(true, deviceId);
        }
    }

    sendMultiroomState(false);
}

bool MultiroomEndpoint::addPeer(const std::string& deviceId, bool discovered) {
    Y_ENSURE_THREAD(lifecycle_);

    if (deviceId == myDeviceId_) {
        return false;
    }

    bool nonAccountDevice = (myDeviceIds_.count(deviceId) == 0);
    if (nonAccountDevice) {
        return false;
    }

    const auto now = std::chrono::steady_clock::now();
    const decltype(now) handshakeTime;

    bool fAdded = false;

    try {
        Y_ENSURE_THREAD(lifecycle_);

        peers_.erase(deviceId);

        std::shared_ptr<Peer2PeerConnector> p2pConnector;
        p2pConnector = std::make_shared<Peer2PeerConnector>(glagolClusterProvider_, deviceId);
        auto [peerIt, _] = peers_.emplace(deviceId, Peer{
                                                        .deviceId = deviceId,
                                                        .netClockId = std::string{},
                                                        .version = 0u,
                                                        .discovered = discovered,
                                                        .multiroomBroadcastProcessor = UniqueCallback{lifecycle_, UniqueCallback::ReplaceType::INSERT_BACK},
                                                        .connector = std::move(p2pConnector),
                                                        .discoveryTime = now,
                                                        .handshakeTime = handshakeTime,
                                                        .keepAliveTime = now,
                                                    });

        fAdded = true;
        sendMasterState(true, deviceId);
        sendMultiroomState(true, deviceId);
        auto message = ipc::UniqueMessage::create();
        message->mutable_multiroom_directive()->mutable_state_request()->set_reason("peer added state exchange");
        peerIt->second.connector->sendMessage(std::move(message));
        LOG_INFO_MULTIROOM("Connect to multiroom:  deviceId=" << deviceId
                                                              << (nonAccountDevice ? " (non account)" : ""));
    } catch (const std::exception& ex) {
        YIO_LOG_ERROR_EVENT("MultiroomEndpoint.FailedAddPeer", "Reconfigure multiroom peers failed: " << ex.what());
    } catch (...) {
        YIO_LOG_ERROR_EVENT("MultiroomEndpoint.FailedAddPeer", "Reconfigure multiroom peers failed: unexpected exception");
    }
    return fAdded;
}

bool MultiroomEndpoint::resetTimedOutPeers()
{
    bool fReset = false;
    auto now = std::chrono::steady_clock::now();
    for (auto it = peers_.begin(); it != peers_.end();) {
        if (it->second.keepAliveTime + PEER_KEEP_ALIVE_TIMEOUT < now) {
            LOG_INFO_MULTIROOM("Reset peer " << it->first << " by timeout");
            resetSlaveSession(it->second.deviceId, "Reset peer by timeout");
            it = peers_.erase(it);
            fReset = true;
        } else {
            ++it;
        }
    }
    return fReset;
}

bool MultiroomEndpoint::resetNonAccountPeers()
{
    bool fReset = false;
    for (auto it = peers_.begin(); it != peers_.end();) {
        if (myDeviceIds_.count(it->second.deviceId) == 0) {
            LOG_INFO_MULTIROOM("Reset peer " << it->first << " it is non-account device");
            resetSlaveSession(it->second.deviceId, "Remove peer from account");
            it = peers_.erase(it);
            fReset = true;
        } else {
            ++it;
        }
    }
    return fReset;
}

bool MultiroomEndpoint::onClientQuasarMessage(const std::string& peer, const ipc::SharedMessage& message) noexcept {
    if (onServerQuasarMessage(peer, message)) {
        return true;
    }

    try {
        Y_ENSURE_THREAD(lifecycle_);

        auto peerIt = peers_.find(peer);
        if (peers_.end() != peerIt) {
            peerIt->second.keepAliveTime = std::chrono::steady_clock::now();
        }

        if (message->has_multiroom_broadcast()) {
            LOG_DEBUG_MULTIROOM("Receive multiroom broadcast from " << peer << ": " << printMultiroomBroadcastCompact(message->multiroom_broadcast()));
            if (peerIt != peers_.end()) {
                peerIt->second.multiroomBroadcastProcessor.executeImmediately(
                    [&] {
                        onMultiroomBroadcast(peer, message->multiroom_broadcast());
                    });
            }
            return true;
        } else {
            LOG_WARN_MULTIROOM("Receive unexpected multiroom message from " << peer);
        }
    } catch (const std::exception& ex) {
        YIO_LOG_ERROR_EVENT("MultiroomEndpoint.FailedProcessClientMessage", "Fail to process quasar message from " << peer << ": " << ex.what());
    } catch (...) {
        YIO_LOG_ERROR_EVENT("MultiroomEndpoint.FailedProcessClientMessage", "Fail to process quasar message from " << peer << ": unexpected exception");
    }
    LOG_DEBUG_MULTIROOM("onClientQuasarMessage: " << shortUtf8DebugString(*message));
    return false;
}

void MultiroomEndpoint::onMultiroomBroadcast(const std::string& peer, const proto::MultiroomBroadcast& multiroomBroadcast)
{
    Y_ENSURE_THREAD(lifecycle_);

    const auto now = std::chrono::steady_clock::now();
    bool doVerboseLog =
        verboseLogOnMultiroomBroadcast_.count(peer) == 0 ||
        now - verboseLogOnMultiroomBroadcast_[peer] > VERBOSE_LOG_PERIOD;
    if (doVerboseLog) {
        verboseLogOnMultiroomBroadcast_[peer] = now;
    }
    auto it_peer = peers_.find(peer);
    if (it_peer == peers_.end()) {
        if (doVerboseLog) {
            LOG_WARN_MULTIROOM("Verbose for \"" << peer << "\": no visible peers");
        }
        return;
    }

    auto& peerRef = it_peer->second;
    bool noHandshake = (peerRef.handshakeTime.time_since_epoch().count() == 0);
    bool differentDeviceId = (!peerRef.deviceId.empty() && peerRef.deviceId != multiroomBroadcast.device_id());
    bool differentClockId = (multiroomBroadcast.has_net_audio_clock_id() &&
                             !multiroomBroadcast.net_audio_clock_id().empty() &&
                             !peerRef.netClockId.empty() &&
                             multiroomBroadcast.net_audio_clock_id() != peerRef.netClockId);
    auto differentClockPeer = (!multiroomBroadcast.net_audio_clock_host().empty() &&
                               multiroomBroadcast.net_audio_clock_port() > 0 &&
                               (peerRef.netClockHost != multiroomBroadcast.net_audio_clock_host() || peerRef.netClockPort != multiroomBroadcast.net_audio_clock_port()));
    auto handshake = false;
    if (noHandshake || differentDeviceId || differentClockId || differentClockPeer) {
        if (noHandshake) {
            LOG_INFO_MULTIROOM("Multiroom handshake from " << multiroomBroadcast.device_id() << ", clockId=" << multiroomBroadcast.net_audio_clock_id() << " (new device)");
        } else if (differentDeviceId) {
            LOG_INFO_MULTIROOM("Multiroom handshake from unexpected device " << multiroomBroadcast.device_id() << " at peer " << peer << " (expected device_id is " << peerRef.deviceId << ")");
        } else if (differentClockId) {
            LOG_INFO_MULTIROOM("Multiroom handshake from " << multiroomBroadcast.device_id() << ", clockId=" << multiroomBroadcast.net_audio_clock_id() << " (clock id changed)");
        } else if (differentClockPeer) {
            LOG_INFO_MULTIROOM("Multiroom handshake from " << multiroomBroadcast.device_id() << ", clockId=" << multiroomBroadcast.net_audio_clock_id() << " (clock address changed)");
        } else {
            LOG_INFO_MULTIROOM("Multiroom handshake from " << multiroomBroadcast.device_id() << ", clockId=" << multiroomBroadcast.net_audio_clock_id());
        }
        peerRef.handshakeTime = now;
        peerRef.deviceId = multiroomBroadcast.device_id();
        peerRef.netClockId = multiroomBroadcast.net_audio_clock_id();
        peerRef.netClockHost = multiroomBroadcast.net_audio_clock_host();
        peerRef.netClockPort = multiroomBroadcast.net_audio_clock_port();
        peerRef.version = (multiroomBroadcast.has_version() ? multiroomBroadcast.version() : 1);

        Json::Value values;
        values["deviceId"] = peerRef.deviceId;
        values["handshakeUs"] = (int64_t)std::chrono::duration_cast<std::chrono::microseconds>(peerRef.handshakeTime - peerRef.discoveryTime).count();
        values["clockId"] = peerRef.netClockId;
        values["clockPeer"] = peerRef.netClockHost + ":" + std::to_string(peerRef.netClockPort);
        values["version"] = peerRef.version;
        reportEvent("multiroomHandshake", values);

        clockTowerProvider_->addClock(peerRef.deviceId, peerRef.netClockHost, peerRef.netClockPort, peerRef.netClockId);
        handshake = true;
    }

    const auto& rdIds = multiroomBroadcast.room_device_ids();
    const bool activeSlave = isActiveMultiroomSession(slave_);
    const bool activeMultiroomBroadcast = isActiveMultiroomSession(multiroomBroadcast);
    const bool sameCluster = std::find(rdIds.begin(), rdIds.end(), myDeviceId_) != rdIds.end();
    const bool sameStereoPair = [&] {
        if (stereoPairState_) {
            if (stereoPairState_->role == StereoPairState::Role::FOLLOWER) {
                if (stereoPairState_->partnerDeviceId == multiroomBroadcast.device_id()) {
                    // My LEADER device start multiroom session, join any way!
                    return true;
                }
            }
            if (!stereoPairState_->partnerDeviceId.empty() &&
                std::find(rdIds.begin(), rdIds.end(), stereoPairState_->partnerDeviceId) != rdIds.end()) {
                //  My parthner in multiroom device list, that's why join to it!
                return true;
            }
        }
        return false;
    }();

    const bool isMultiroomPlayerPlaying = lastMultiroomShortAppState_.isSlave() && lastMultiroomShortAppState_.isPlaying;
    bool needToStopMaster = false;
    bool needToStopSlave = false;
    bool needToPauseSlave = false;
    bool needToStartSlave = false;
    bool needToStopSinglePlayer = false;
    bool needToResetSlaveSession = false;

    // If we have an active multiroom session and the master of this
    // session received an empty broadcast or broadcasts in a room in
    // which we don’t, then you need to stop the current playback.
    needToStopSlave = needToStopSlave || (activeSlave &&
                                          slave_.device_id() == multiroomBroadcast.device_id() &&
                                          (!activeMultiroomBroadcast || !(sameCluster || sameStereoPair)));

    // If broadcast came from sessions whose timestamp is late than
    // the current one, then you need to pick it up, switch to SLAVE
    // mode and start broadcasting a new URL
    if (activeMultiroomBroadcast && (sameCluster || sameStereoPair))
    {
        needToStopMaster = needToStopMaster || ((mode_ == Mode::MASTER || isActiveMultiroomSession(master_)) &&
                                                multiroomBroadcast.session_timestamp_ms() > master_.session_timestamp_ms());
        needToStopSlave = needToStopSlave || (activeSlave &&
                                              multiroomBroadcast.session_timestamp_ms() > slave_.session_timestamp_ms());

        const bool sameMultiroomSource = isSameMultiroomSource(slave_.multiroom_params(), multiroomBroadcast.multiroom_params());
        needToStartSlave =
            (mode_ != Mode::MASTER || needToStopMaster) && multiroomBroadcast.state() == proto::MultiroomBroadcast::PLAYING && (!activeSlave || slave_.device_id() != multiroomBroadcast.device_id() || slave_.session_timestamp_ms() < multiroomBroadcast.session_timestamp_ms() || slave_.multiroom_token() != multiroomBroadcast.multiroom_token() || !sameMultiroomSource || (sameMultiroomSource && !isMultiroomPlayerPlaying));
    }

    if (needToStartSlave) {
        if (multiroomPending_) {
            if (multiroomPending_->multiroomTrackId != makeMultiroomTrackId(multiroomBroadcast)) {
                LOG_WARN_MULTIROOM("Cancel multiroom pending " << multiroomPending_->multiroomSessionId << ", track " << multiroomPending_->multiroomTrackId);
                multiroomPending_ = std::nullopt;
            } else if (multiroomPending_->until < std::chrono::steady_clock::now()) {
                LOG_WARN_MULTIROOM("Multiroom pending " << multiroomPending_->multiroomSessionId << " timeout");
                multiroomPending_ = std::nullopt;
            } else {
                needToStartSlave = false;
            }
        }
    }

    auto stereoPairApproved = false;
    if (stereoPairState_ && stereoPairState_->isStereoPair()) {
        if (stereoPairState_->role == StereoPairState::Role::FOLLOWER) {
            if (needToStartSlave) {
                // First of all, we need to make sure that our master started playing the same session
                auto parentPeerIt = peers_.find(stereoPairState_->partnerDeviceId);
                auto parent_state_it = multiroomStates_.find(stereoPairState_->partnerDeviceId);
                bool stereoOk = false;
                if (parent_state_it != multiroomStates_.end()) {
                    if (parent_state_it->second.has_multiroom_broadcast() && multiroomSessionId(parent_state_it->second.multiroom_broadcast()) == multiroomSessionId(multiroomBroadcast)) {
                        stereoOk = true;
                    }
                }

                if (!stereoOk) {
                    needToStartSlave = false;
                    if (isMultiroomPlayerPlaying) {
                        needToPauseSlave = true;
                        LOG_INFO_MULTIROOM("Pause old music because leader broadcast a new track");
                    } else {
                        LOG_INFO_MULTIROOM("Cancel start multiroom because stereo pair leader is not ready");
                    }
                    if (parentPeerIt != peers_.end()) {
                        auto message = ipc::UniqueMessage::create();
                        message->mutable_multiroom_directive()->mutable_state_request()->set_reason("stereo pair follower sync request");
                        parentPeerIt->second.connector->sendMessage(std::move(message));
                    }
                    peerRef.multiroomBroadcastProcessor.executeDelayed(
                        [this, peer, multiroomBroadcast] {
                            LOG_DEBUG_MULTIROOM("Processing deferred (stereo pair leader and follower conflict) multiroom broadcast from " << peer << ": " << printMultiroomBroadcastCompact(multiroomBroadcast));
                            onMultiroomBroadcast(peer, multiroomBroadcast);
                        }, BROADCAST_PROCESSOR_SHORT_DELAY, lifetime_);
                } else {
                    stereoPairApproved = true;
                }
            }
        }
    }

    if (!needToStopSlave) {
        // The master stopped playback, perhaps paused
        needToPauseSlave = needToPauseSlave || (activeSlave &&
                                                slave_.device_id() == multiroomBroadcast.device_id() &&
                                                multiroomBroadcast.state() != proto::MultiroomBroadcast::PLAYING &&
                                                isMultiroomPlayerPlaying);
    } else if (needToPauseSlave) {
        LOG_INFO_MULTIROOM("Reset needToPauseSlave because slave player must stop");
        needToPauseSlave = false;
    }

    needToStopSinglePlayer =
        needToStartSlave &&
        lastMultiroomShortAppState_.isPlaying &&
        (!isSameMultiroomSession(slave_, multiroomBroadcast) || !lastMultiroomShortAppState_.isSlave());

    const auto important = (needToStopMaster || needToStopSlave || needToPauseSlave || needToStartSlave || needToStopSinglePlayer || multiroomPending_);
    doVerboseLog = doVerboseLog || important;
    if (doVerboseLog) {
        LOG_SEPARATOR();
        LOG_VERB_MULTIROOM(important, "Verbose for " << peer << ": before any actions");
        LOG_SEPARATOR();
        LOG_VERB_MULTIROOM(important, "    mode_=" << modeName(mode_));
        LOG_VERB_MULTIROOM(important, "    activeSlave=" << activeSlave);
        LOG_VERB_MULTIROOM(important, "    activeMultiroomBroadcast=" << activeMultiroomBroadcast);
        LOG_VERB_MULTIROOM(important, "    sameCluster=" << sameCluster);
        LOG_VERB_MULTIROOM(important, "    sameStereoPair=" << sameStereoPair);
        LOG_VERB_MULTIROOM(important, "    multiroomPending=" << (multiroomPending_ ? multiroomPending_->multiroomSessionId : "<none>"));
        LOG_VERB_MULTIROOM(important, "    needToStopMaster=" << needToStopMaster);
        LOG_VERB_MULTIROOM(important, "    needToStopSlave=" << needToStopSlave);
        LOG_VERB_MULTIROOM(important, "    needToPauseSlave=" << needToPauseSlave);
        LOG_VERB_MULTIROOM(important, "    needToStartSlave=" << needToStartSlave);
        LOG_VERB_MULTIROOM(important, "    needToStopSinglePlayer=" << needToStopSinglePlayer);
        LOG_VERB_MULTIROOM(important, "    isMultiroomPlayerPlaying=" << isMultiroomPlayerPlaying);
        LOG_VERB_MULTIROOM(important, "    isActiveMultiroomSession(master_)=" << isActiveMultiroomSession(master_));
        LOG_VERB_MULTIROOM(important, "    master_.session_timestamp_ms()=" << master_.session_timestamp_ms());
        LOG_VERB_MULTIROOM(important, "    slave_.session_timestamp_ms()=" << slave_.session_timestamp_ms());
        LOG_VERB_MULTIROOM(important, "    slave_.device_id()=" << slave_.device_id());
        LOG_VERB_MULTIROOM(important, "    slave_.multiroom_token()=" << slave_.multiroom_token());
        LOG_VERB_MULTIROOM(important, "    multiroomBroadcast.device_id()=" << multiroomBroadcast.device_id());
        LOG_VERB_MULTIROOM(important, "    multiroomBroadcast.state()=" << proto::MultiroomBroadcast::State_Name(multiroomBroadcast.state()));
        LOG_VERB_MULTIROOM(important, "    multiroomBroadcast.session_timestamp_ms()=" << multiroomBroadcast.session_timestamp_ms());
        LOG_VERB_MULTIROOM(important, "    isSameMultiroomSource(slave_, multiroomBroadcast)=" << isSameMultiroomSource(slave_.multiroom_params(), multiroomBroadcast.multiroom_params()));
        LOG_VERB_MULTIROOM(important, "    lastAppState_=" << convertMessageToDeepJsonString(*lastAppState_));
        LOG_SEPARATOR();
    }

    if (needToStopSinglePlayer) {
        LOG_INFO_MULTIROOM("It was decided to stop current player " << lastMultiroomShortAppState_.playerName() << " (vins " << lastMultiroomShortAppState_.vinsRequestId << ") before starting slave player");

        sdk_->getPlaybackControlCapability()->pause();

        peerRef.multiroomBroadcastProcessor.executeDelayed(
            [this, peer, multiroomBroadcast] {
                LOG_DEBUG_MULTIROOM("Processing deferred (stopping single player) multiroom broadcast from " << peer << ": " << printMultiroomBroadcastCompact(multiroomBroadcast));
                onMultiroomBroadcast(peer, multiroomBroadcast);
            }, BROADCAST_PROCESSOR_LONG_DELAY, lifetime_);
        LOG_DEBUG_MULTIROOM("Multiroom broadcast processing is deferred to the best times. Slave player will not be stared.");
        needToStartSlave = false;
    }

    if (needToStopSlave) {
        LOG_INFO_MULTIROOM("It was decided to stop multiroom slave player. " << printMultiroomBroadcastCompact(slave_));
        stopSlavePlayer("Receive MultiroomBroadcast from peer=" + peer + ", device_id=" + multiroomBroadcast.device_id());
    }

    if (needToStopMaster) {
        LOG_INFO_MULTIROOM("It was decided to stop master broadcasting: " << printMultiroomBroadcastCompact(master_));
        stopMaster("Receive MultiroomBroadcast from peer=" + peer + ", device_id=" + multiroomBroadcast.device_id());
    }

    bool forcedSlavePause = false;
    slaveClockSyncing_ = std::nullopt;
    slaveClockSyncingExpiredAt_ = now + SYNC_CLOCK_EXPIRED_PERIOD;
    if (needToStartSlave) {
        const auto& mbc = multiroomBroadcast;
        const auto clock = clockTowerState_->findClockById(mbc.net_audio_clock_id());
        if (!clock) {
            LOG_INFO_MULTIROOM("It was decided to start new multiroom slave player but can't get the network clock: host=" << clock->host() << ", port=" << clock->port() << ", clockId=" << clock->clockId());
            if (mbc.net_audio_clock_host().empty()) {
                return;
            }
            Json::Value values;
            values["vinsRequestId"] = mbc.vins_request_id();
            values["multiroomSessionId"] = makeMultiroomSessionId(mbc);
            values["multiroomTrackId"] = makeMultiroomTrackId(mbc);
            reportEvent("multiroomClockFail", values);
            slaveClockSyncing_ = IClock::SyncLevel::NONE;
            clockTowerProvider_->addClock(mbc.device_id(), mbc.net_audio_clock_host(), mbc.net_audio_clock_port(), mbc.net_audio_clock_id());
        } else if (clock->syncLevel() < IClock::SyncLevel::STRONG) {
            LOG_INFO_MULTIROOM("It was decided to start new multiroom slave player but clock synchronization is not good enough, sync level is " << clock->syncLevelAsText());
            slaveClockSyncing_ = clock->syncLevel();
            peerRef.multiroomBroadcastProcessor.executeDelayed(
                [this, peer, mbc] {
                    LOG_DEBUG_MULTIROOM("Processing deferred (bad clock synchronization) multiroom broadcast from " << peer << ": " << printMultiroomBroadcastCompact(mbc));
                    onMultiroomBroadcast(peer, mbc);
                }, BROADCAST_PROCESSOR_LONG_DELAY, lifetime_);
            if (isMultiroomPlayerPlaying) {
                forcedSlavePause = true;
                LOG_INFO_MULTIROOM("Pause old music because master broadcast a new track");
            }
            Json::Value values;
            values["vinsRequestId"] = mbc.vins_request_id();
            values["multiroomSessionId"] = makeMultiroomSessionId(mbc);
            values["multiroomTrackId"] = makeMultiroomTrackId(mbc);
            reportEvent("multiroomClockWeak", values);
            LOG_DEBUG_MULTIROOM("Multiroom broadcast processing is deferred to the best times. Let's wait for the clock to sync.");
        } else {
            LOG_INFO_MULTIROOM("It was decided to start new multiroom slave player. [AudioClient]" << (stereoPairApproved ? " (Stereo pair follower)" : ""));
            slave_.CopyFrom(mbc);
            slave_.mutable_master_hints()->Clear();
            if (slave_.net_audio_clock_host().empty()) {
                slave_.set_net_audio_clock_host(TString(clock->host()));
            }
            setMode(Mode::SLAVE);
            multiroomPending_ =
                MultiroomPending{
                    .multiroomSessionId = makeMultiroomSessionId(slave_),
                    .multiroomTrackId = makeMultiroomTrackId(slave_),
                    .playPauseId = lastMultiroomShortAppState_.playPauseId,
                    .until = now + MULTIROOM_PENDING_TIMEOUT};
            sendToDirectiveProcessor(MultiroomDirectives::createDirectiveSlavePlay(slave_));
            if (mbc.has_master_hints() && mbc.master_hints().quiet_audio_focus()) {
                LOG_DEBUG_MULTIROOM("Multiroom hint: quite audio focus");
                mute(mbc.device_id());
            }
        }
        slaveClockSyncingExpiredAt_ = now + SYNC_CLOCK_EXPIRED_PERIOD;
    }

    if (needToPauseSlave || forcedSlavePause) {
        LOG_INFO_MULTIROOM("It was decided to pause multiroom slave player. " << printMultiroomBroadcastCompact(slave_));
        pauseSlavePlayer("Receive MultiroomBroadcast from peer=" + peer + ", device_id=" + multiroomBroadcast.device_id());
        const auto& clockHost = multiroomBroadcast.net_audio_clock_host();
        slave_.CopyFrom(multiroomBroadcast);
        slave_.mutable_master_hints()->Clear();
        if (slave_.net_audio_clock_host().empty()) {
            slave_.set_net_audio_clock_host(TString(clockHost));
        }
    }

    needToResetSlaveSession = (!slave_.device_id().empty() && multiroomBroadcast.device_id() == slave_.device_id()) && multiroomBroadcast.session_timestamp_ms() != slave_.session_timestamp_ms();
    if (doVerboseLog) {
        LOG_SEPARATOR();
        LOG_VERB_MULTIROOM(important, "Verbose for " << peer << ": after the decision");
        LOG_SEPARATOR();
        LOG_VERB_MULTIROOM(important, "    needToResetSlaveSession=" << needToResetSlaveSession);
        if (forcedSlavePause) {
            LOG_VERB_MULTIROOM(important, "    forcedSlavePause=" << forcedSlavePause);
        }
        if (multiroomPending_) {
            auto durationMs = std::chrono::duration_cast<std::chrono::milliseconds>(multiroomPending_->until - std::chrono::steady_clock::now()).count();
            LOG_VERB_MULTIROOM(important, "    multiroomPending={" << multiroomPending_->multiroomSessionId << ", " << multiroomPending_->playPauseId << ", " << durationMs << " ms}");
        }
        LOG_VERB_MULTIROOM(important, "    mode_=" << modeName(mode_));
        LOG_VERB_MULTIROOM(important, "    master_=" << convertMessageToDeepJsonString(master_));
        LOG_VERB_MULTIROOM(important, "    slave_=" << convertMessageToDeepJsonString(slave_));
        LOG_VERB_MULTIROOM(important, "    lastMultiroomShortAppState_=" << printMultiroomShortAppState(lastMultiroomShortAppState_));
        LOG_VERB_MULTIROOM(important, "    resumeMasterParams_=" << jsonToString(resumeMasterParams_));
        LOG_SEPARATOR();
    }

    if (needToStopSinglePlayer || needToStopSlave || needToStopMaster || needToStartSlave) {
        if (doVerboseLog) {
            LOG_DEBUG_MULTIROOM("Verbose for " << peer << ": call updateState()");
        }
        updateState();
    } else if (handshake) {
        sendMasterState(false);
        sendMultiroomState(false);
    } else if (needToResetSlaveSession) {
        resetSlaveSession(slave_.device_id(), "Master changed multiroom session id");
    }
}

bool MultiroomEndpoint::onServerQuasarMessage(const std::string& peer, const ipc::SharedMessage& message) noexcept {
    try {
        Y_ENSURE_THREAD(lifecycle_);

        if (message->has_multiroom_directive()) {
            const auto& directive = message->multiroom_directive();
            if (directive.has_stop()) {
                onDirectiveStop(peer, directive);
                return true;
            } else if (directive.has_command()) {
                onDirectiveCommand(peer, directive);
                return true;
            } else if (directive.has_audio_focus()) {
                onDirectiveAudioFocus(peer, directive);
                return true;
            } else if (directive.has_state_request()) {
                onDirectiveStateRequest(peer, directive);
                return true;
            } else if (directive.has_resume_broadcast()) {
                // For compatibility with firmware older 127 version
                if (makeMultiroomSessionId(directive.resume_broadcast().multiroom_broadcast()) == makeMultiroomSessionId(master_)) {
                    LOG_DEBUG_MULTIROOM("Old firmware compatibility. Translate ResumeBroadcast to Directives::PLAYER_CONTINUE: " << convertMessageToJsonString(directive));
                    sendToDirectiveProcessor(Directives::PLAYER_CONTINUE);
                }
                return true;
            } else if (directive.has_player_pause()) {
                // For compatibility with firmware older 123 version
                if (isActiveMultiroomSession(master_)) {
                    LOG_DEBUG_MULTIROOM("Old firmware compatibility. Translate PlayerPause to Directives::STOP_LEGACY_PLAYER: " << convertMessageToJsonString(directive));
                    sendToDirectiveProcessor(Directives::STOP_LEGACY_PLAYER);
                }
                return true;
            } else if (directive.has_master_external_command() || directive.has_execute_external_command() || directive.has_cluster_external_command()) {
                // For compatibility with older firmware
                const auto& command = (directive.has_master_external_command()
                                           ? directive.master_external_command()
                                           : (
                                                 directive.has_execute_external_command()
                                                     ? directive.execute_external_command()
                                                     : directive.cluster_external_command()));
                LOG_DEBUG_MULTIROOM("Old firmware compatibility: " << convertMessageToJsonString(directive));
                sendToDirectiveProcessor(YandexIO::Directive::createFromExternalCommandMessage(command));
                return true;
            } else {
                LOG_WARN_MULTIROOM("Receive unknown multiroom directive from " << message->multiroom_directive().sender_device_id() << " (peer " << peer << "): directive=" << shortUtf8DebugString(directive));
            }
        } else if (message->has_multiroom_state()) {
            onMultiroomStateReceived(peer, message);
            return true;
        }
    } catch (const std::exception& ex) {
        YIO_LOG_ERROR_EVENT("MultiroomEndpoint.FailedProcessServerMessage", "Fail to process quasar message for server: " << ex.what());
    } catch (...) {
        YIO_LOG_ERROR_EVENT("MultiroomEndpoint.FailedProcessServerMessage", "Fail to process quasar message for server: unexpected exception");
    }
    return false;
}

void MultiroomEndpoint::onDirectiveStop(const std::string& peer, const proto::MultiroomDirective& directive)
{
    Y_ENSURE_THREAD(lifecycle_);

    bool unprocessed = true;
    if (isSameMultiroomSession(directive, master_)) {
        LOG_INFO_MULTIROOM("Execute \"STOP\" directive on master from " << directive.sender_device_id() << " (peer " << peer << "): mode=" << modeName(mode_));
        stopMasterPlayer("Directive from " + directive.sender_device_id() + "; Reason=[" + directive.stop().reason() + "]");
        unprocessed = false;
    }

    if (isSameMultiroomSession(directive, slave_)) {
        LOG_INFO_MULTIROOM("Execute \"STOP\" directive on slave from " << directive.sender_device_id() << " (peer " << peer << "): mode=" << modeName(mode_));
        stopSlavePlayer("Directive from " + directive.sender_device_id() + "; Reason=[" + directive.stop().reason() + "]");
        unprocessed = false;
    }

    if (unprocessed) {
        LOG_DEBUG_MULTIROOM("Ignore \"STOP\" directive from " << directive.sender_device_id() << " (peer " << peer << "): " << shortUtf8DebugString(directive));
        LOG_DEBUG_MULTIROOM("    MASTR: " << printMultiroomBroadcastCompact(master_));
        LOG_DEBUG_MULTIROOM("    SLAVE: " << printMultiroomBroadcastCompact(slave_));
    }
}

void MultiroomEndpoint::onDirectiveCommand(const std::string& peer, const proto::MultiroomDirective& directive)
{
    Y_ENSURE_THREAD(lifecycle_);

    if (!isSameMultiroomSession(directive, master_)) {
        LOG_INFO_MULTIROOM("Ignore the \"COMMAND\" directive because it expired: " << shortUtf8DebugString(directive) << ", master: " << printMultiroomBroadcastCompact(master_));
        return;
    }
    Json::Value options = parseJson(directive.command().json_options());
    LOG_INFO_MULTIROOM("Process \"COMMAND\" directive from " << directive.sender_device_id() << " (peer " << peer << "): " << shortUtf8DebugString(directive));
    const auto& name = directive.command().command();

    if (name == "like") {
        sdk_->getPlaybackControlCapability()->like();
    } else if (name == "dislike") {
        sdk_->getPlaybackControlCapability()->dislike();
    } else if (name == "next") {
        sdk_->getPlaybackControlCapability()->next();
    } else if (name == "previous") {
        sdk_->getPlaybackControlCapability()->prev();
    }
}

void MultiroomEndpoint::onDirectiveAudioFocus(const std::string& peer, const proto::MultiroomDirective& directive)
{
    LOG_DEBUG_MULTIROOM("Process \"AUDIO_FOCUS\" directive from " << directive.sender_device_id() << " (peer " << peer << ")");
    Y_ENSURE_THREAD(lifecycle_);

    if (!isSameMultiroomSession(directive, master_) && !isSameMultiroomSession(directive, slave_))
    {
        LOG_DEBUG_MULTIROOM("Ignore the \"AUDIO_FOCUS\" directive because different multiroom sessions");
        return;
    }

    /**
     * If this was our directive and she returned to us from the master, then we will ignore it
     */
    if (directive.audio_focus().initiator_device_id() == myDeviceId_) {
        return;
    }

    /**
     * The master received the directive, it is necessary to send it to other peers
     */
    if (isSameMultiroomSession(directive, master_)) {
        LOG_DEBUG_MULTIROOM("Retranslate directive \"AUDIO_FOCUS\" from " << directive.sender_device_id() << " to other slaves");
        sendDirective(directive, master_);
    }

    auto initiatorDeviceId = directive.audio_focus().initiator_device_id();
    if (directive.audio_focus().has_audio_focus()) {
        encourageAudioFocus(initiatorDeviceId);
    } else {
        mute(initiatorDeviceId);
    }
}

void MultiroomEndpoint::onDirectiveStateRequest(const std::string& peer, const proto::MultiroomDirective& directive)
{
    LOG_DEBUG_MULTIROOM("Process \"STATE REQUEST\" directive from " << directive.sender_device_id() << " (peer " << peer << ") reason: " << directive.state_request().reason());
    Y_ENSURE_THREAD(lifecycle_);
    sendMasterState(true, peer);
    sendMultiroomState(true, peer);
}

void MultiroomEndpoint::onMultiroomStateReceived(const std::string& peer, const ipc::SharedMessage& message)
{
    LOG_DEBUG_MULTIROOM("Process multiroom state received from peer " << peer);
    Y_ENSURE_THREAD(lifecycle_);

    const proto::MultiroomState& multiroomState = message->multiroom_state();
    bool fChanged = false;

    if (fChanged) {
        sendMultiroomState(false);
    }

    multiroomStates_[multiroomState.device_id()].CopyFrom(multiroomState);
}

void MultiroomEndpoint::startMaster(std::string vinsRequestId, std::string multiroomToken, std::vector<std::string> roomDeviceIds)
{
    if (isActiveMultiroomSession(slave_)) {
        stopAllSlaves("Start master for vinsRequestId=" + vinsRequestId);
    }

    if (multiroomToken.empty()) {
        Json::Value values;
        values["message"] = "Receive empty MultiroomToken";
        reportEvent("multiroomError", values);
        YIO_LOG_ERROR_EVENT("MultiroomEndpoint.MissingMultiroomToken", "Fail to start multiroom with empty token");
        throw std::runtime_error(makeString("Multiroom token is not specified: vinsRequestId=", vinsRequestId));
    }

    clearMaster();
    master_.set_session_timestamp_ms(std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::system_clock::now().time_since_epoch()).count());
    master_.set_multiroom_token(std::move(multiroomToken));
    if (!roomDeviceIds.empty()) {
        for (auto& deviceId : roomDeviceIds) {
            master_.add_room_device_ids(std::move(deviceId));
        }
    }
    master_.set_vins_request_id(std::move(vinsRequestId));
    master_.set_state(proto::MultiroomBroadcast::PAUSED);
    resumeMasterParams_ = Json::Value{};
    setMode(Mode::MASTER);
}

void MultiroomEndpoint::stopMaster(const std::string& reason)
{
    Y_ENSURE_THREAD(lifecycle_);

    if (isActiveMultiroomSession(master_)) {
        LOG_INFO_MULTIROOM("Stop multiroom master session for multiroom token " << master_.multiroom_token()
                                                                                << " (session " << master_.session_timestamp_ms() << ")"
                                                                                << " Reason: " << reason);
        stopMultiroomSession(master_, reason);

        Json::Value values;
        values["multiroomSessionId"] = makeMultiroomSessionId(master_);
        values["vinsRequestId"] = master_.vins_request_id();
        values["reason"] = reason;
        auto multiroomTrackId = makeMultiroomTrackId(master_);
        if (!multiroomTrackId.empty()) {
            values["multiroomTrackId"] = multiroomTrackId;
        }
        reportEvent("multiroomStopMaster", values);
    }

    if (mode_ == Mode::MASTER) {
        resumeMasterParams_["vins_request_id"] = master_.vins_request_id();
        resumeMasterParams_["multiroom_token"] = master_.multiroom_token();
        if (master_.room_device_ids_size() > 0) {
            resumeMasterParams_["room_device_ids"] = Json::arrayValue;
            for (const auto& deviceId : master_.room_device_ids()) {
                resumeMasterParams_["room_device_ids"].append(deviceId);
            }
        }
        resumeMasterParams_["play_pause_id"] = lastMultiroomShortAppState_.playPauseId;
        muteTimeouts_.clear();
        unmuteMultiroomIfNecessary();
        setMode(Mode::NONE);
    }
    clearMaster();
}

void MultiroomEndpoint::stopMasterPlayer(const std::string& reason)
{
    LOG_INFO_MULTIROOM("Call stopMasterPlayer. Reason: " << reason);
    sdk_->getPlaybackControlCapability()->pause();
}

void MultiroomEndpoint::stopAllSlaves(const std::string& reason)
{
    LOG_INFO_MULTIROOM("Stop multiroom slave session "
                       << "of master " << slave_.device_id()
                       << " with multiroom token " << slave_.multiroom_token()
                       << " (session " << slave_.session_timestamp_ms() << ")"
                       << " Reason: " << reason);
    Y_ENSURE_THREAD(lifecycle_);

    Json::Value values;
    values["multiroomSessionId"] = makeMultiroomSessionId(slave_);
    values["vinsRequestId"] = slave_.vins_request_id();
    values["reason"] = reason;
    auto multiroomTrackId = makeMultiroomTrackId(slave_);
    if (!multiroomTrackId.empty()) {
        values["multiroomTrackId"] = multiroomTrackId;
    }
    reportEvent("multiroomStopSlave", values);

    stopMultiroomSession(slave_, "Stop all slaves; Reason " + reason);
    stopSlavePlayer("Stop all slaves; Reason: " + reason);
    if (mode_ == Mode::SLAVE) {
        setMode(Mode::NONE);
    }
    slave_.Clear();
}

void MultiroomEndpoint::stopSlavePlayer(const std::string& reason)
{
    Y_ENSURE_THREAD(lifecycle_);

    if (isActiveMultiroomSession(slave_)) {
        std::string multiroomSessionId = makeMultiroomSessionId(slave_);

        LOG_INFO_MULTIROOM("Stop multiroom slave player of session " << multiroomSessionId << ", reason: " << reason);

        if (mode_ == Mode::SLAVE) {
            muteTimeouts_.clear();
            unmuteMultiroomIfNecessary();
            setMode(Mode::NONE);
        }
        slave_.Clear();
        auto message = ipc::buildMessage([&](auto& msg) {
            auto directive = MultiroomDirectives::createDirectiveSlaveStop(multiroomSessionId, reason);
            msg.mutable_directive()->CopyFrom(YandexIO::Directive::convertToDirectiveProtobuf(directive));
            msg.mutable_directive()->set_is_route_locally(true);
        });
        toAliced_->sendMessage(message);
        LOG_DEBUG_MULTIROOM("Send directive to DirectiveHandler to stop multiroom: " << convertMessageToDeepJsonString(*message));
    }
}

void MultiroomEndpoint::pauseSlavePlayer(const std::string& reason)
{
    Y_ENSURE_THREAD(lifecycle_);

    if (isActiveMultiroomSession(slave_)) {
        std::string multiroomSessionId = makeMultiroomSessionId(slave_);

        LOG_INFO_MULTIROOM("Pause multiroom slave player of session " << multiroomSessionId << ", reason: " << reason);

        if (mode_ == Mode::SLAVE) {
            muteTimeouts_.clear();
            unmuteMultiroomIfNecessary();
        }
        auto message = ipc::buildMessage([&](auto& msg) {
            auto directive = MultiroomDirectives::createDirectiveSlavePause(multiroomSessionId, reason);
            msg.mutable_directive()->CopyFrom(YandexIO::Directive::convertToDirectiveProtobuf(directive));
            msg.mutable_directive()->set_is_route_locally(true);
        });
        toAliced_->sendMessage(message);
        LOG_DEBUG_MULTIROOM("Send directive to DirectiveHandler to pause multiroom: " << convertMessageToDeepJsonString(*message));
    }
}

void MultiroomEndpoint::stopMultiroomSession(const proto::MultiroomBroadcast& multiroomSession, const std::string& reason)
{
    Y_ENSURE_THREAD(lifecycle_);

    if (isActiveMultiroomSession(multiroomSession)) {
        LOG_INFO_MULTIROOM("Broadcast multiroom directive \"STOP\" for multiroom session of "
                           << "master " << multiroomSession.device_id()
                           << " with multiroom token " << multiroomSession.multiroom_token()
                           << " (session " << multiroomSession.session_timestamp_ms() << ")"
                           << " Reason: " << reason);

        proto::MultiroomDirective directive;
        directive.mutable_stop()->set_reason(TString(reason));
        sendDirective(std::move(directive), multiroomSession);
    }
}

void MultiroomEndpoint::resetSlaveSession(const std::string& deviceId, const std::string& reason) {
    Y_ENSURE_THREAD(lifecycle_);

    if (slave_.device_id() == deviceId) {
        LOG_INFO_MULTIROOM("Reset slave session by device id " << deviceId << ": " << reason);
        pauseSlavePlayer(reason);
        if (mode_ == Mode::SLAVE) {
            setMode(Mode::NONE);
        }
        slave_.Clear();

        sendMasterState(false);
        sendMultiroomState(false);
    }
}

void MultiroomEndpoint::resetMasterSession() {
    Y_ENSURE_THREAD(lifecycle_);

    if (master_.state() == proto::MultiroomBroadcast::PAUSED) {
        if (mode_ == Mode::MASTER) {
            stopMaster("Reset multiroom master state");
        } else {
            LOG_INFO_MULTIROOM("Reset multiroom master state");
            clearMaster();
            sendMasterState(false);
            sendMultiroomState(false);
        }
        resumeMasterParams_ = Json::Value{};
    }
}

void MultiroomEndpoint::updateState(const std::shared_ptr<const proto::AppState>& appState)
{
    Y_ENSURE_THREAD(lifecycle_);

    lastAppState_ = appState;
    lastMultiroomShortAppState_ = makeMultiroomShortAppState(*lastAppState_);
    updateState();
}

void MultiroomEndpoint::updateState()
{
    Y_ENSURE_THREAD(lifecycle_);

    if (multiroomPending_) {
        if (!isActiveMultiroomSession(slave_)) {
            LOG_INFO_MULTIROOM("It no longer need to wait for the start of the multiroom session " << multiroomPending_->multiroomSessionId << " with track id " << multiroomPending_->multiroomTrackId);
        } else if (lastMultiroomShortAppState_.isSlave() && multiroomPending_->playPauseId != lastMultiroomShortAppState_.playPauseId) {
            LOG_INFO_MULTIROOM("Multiroom start notification received: " << printMultiroomShortAppState(lastMultiroomShortAppState_));
        } else if (multiroomPending_->until < std::chrono::steady_clock::now()) {
            LOG_WARN_MULTIROOM("Multiroom did not start in " << std::chrono::duration_cast<std::chrono::milliseconds>(MULTIROOM_PENDING_TIMEOUT).count() << " milliseconds time");
            Json::Value values;
            values["multiroomSessionId"] = multiroomPending_->multiroomSessionId;
            values["multiroomTrackId"] = multiroomPending_->multiroomTrackId;
            reportEvent("multiroomPendingTimeout", values);
        } else {
            LOG_DEBUG_MULTIROOM("Multiroom in pending mode. Waiting for the slave player to start: multiroomSessionId=" << multiroomPending_->multiroomSessionId);
            return;
        }
        multiroomPending_ = std::nullopt;
    }

    bool forceSendState = false;
    if (mode_ == Mode::MASTER) {
        auto newState = (lastMultiroomShortAppState_.isPlaying && lastMultiroomShortAppState_.multiroomToken == master_.multiroom_token() ? proto::MultiroomBroadcast::PLAYING : proto::MultiroomBroadcast::PAUSED);
        if (newState != master_.state()) {
            LOG_INFO_MULTIROOM("Switch master state from " << proto::MultiroomBroadcast::State_Name(master_.state()) << " to " << proto::MultiroomBroadcast::State_Name(newState));
            master_.set_state(newState);
        }
        if (newState == proto::MultiroomBroadcast::PLAYING) {
            auto multiroomParams = makeMultiroomParamsFromAppState();
            master_.mutable_multiroom_params()->Swap(&multiroomParams);
        }
    } else {
        if (lastMultiroomShortAppState_.isPlaying) {
            bool isMasterPlaying = (slave_.state() == proto::MultiroomBroadcast::PLAYING && !slave_.multiroom_token().empty());
            bool isMultiroomTokenMissmatched = (lastMultiroomShortAppState_.multiroomToken != slave_.multiroom_token());
            if (isMasterPlaying && isMultiroomTokenMissmatched) {
                LOG_DEBUG_MULTIROOM("Turn off multiroom session: Master session " << makeMultiroomSessionId(slave_) << ", my player \"" << lastMultiroomShortAppState_.playerName() << "\" and multiroom token \"" << lastMultiroomShortAppState_.multiroomToken << "\"");
                stopAllSlaves(std::string("Another player (playerType=") + lastMultiroomShortAppState_.playerName() + ") is currently active, turn off multiroom");
                forceSendState = true;
            }
        }
    }

    if (!isScheduledMultiroomBroadcast() &&
        mode_ == Mode::NONE &&
        stereoPairState_ &&
        stereoPairState_->role == StereoPairState::Role::LEADER &&
        !stereoPairState_->partnerDeviceId.empty() &&
        lastMultiroomShortAppState_.isPlaying &&
        (lastMultiroomShortAppState_.player == MultiroomShortAppState::Player::AUDIO_CLIENT ||
         lastMultiroomShortAppState_.player == MultiroomShortAppState::Player::YANDEX_MUSIC)) {
        auto spMultiroomToken = (lastMultiroomShortAppState_.multiroomToken.empty() ? STEREO_PAIR_MULTIROOM_TOKEN : lastMultiroomShortAppState_.multiroomToken);
        LOG_INFO_MULTIROOM("Start master multiroom broadcasting for stereo pair vins=" << lastMultiroomShortAppState_.vinsRequestId << ", multiroom_token=" << spMultiroomToken);
        std::vector<std::string> roomDeviceIds{device_->deviceId(), stereoPairState_->partnerDeviceId};
        startMaster(lastMultiroomShortAppState_.vinsRequestId, spMultiroomToken, roomDeviceIds);
        master_.set_state(proto::MultiroomBroadcast::PLAYING);
        reportEventStartMaster(true);
        forceSendState = true;
    }

    auto masterPlayer = [&]() -> std::string {
        if (isActiveMultiroomSession(master_)) {
            return makeString("broadcasting.", lastMultiroomShortAppState_.playerName());
        } else {
            return "inactive";
        }
    };
    auto slavePlayer = [&]() {
        if (isActiveMultiroomSession(slave_)) {
            return (slave_.state() == proto::MultiroomBroadcast::PLAYING ? "playing" : "paused");
        } else {
            return "inactive";
        }
    };
    std::string newMultiroomUpdateLog =
        makeString(
            "Update multiroom state: mode=", modeName(mode_), ", ",
            "player=", lastMultiroomShortAppState_.playerName(), ", ",
            "master=", masterPlayer(), ", ",
            "slave=", slavePlayer(), ", ",
            "vinsRequestId=", lastMultiroomShortAppState_.vinsRequestId);
    if (newMultiroomUpdateLog != lastMultiroomUpdateLog_) {
        lastMultiroomUpdateLog_ = std::move(newMultiroomUpdateLog);
        LOG_DEBUG_MULTIROOM(lastMultiroomUpdateLog_);
    }

    sendMasterState(forceSendState);
    sendMultiroomState(forceSendState);
}

void MultiroomEndpoint::sendMultiroomState(bool force, const Target& target)
{
    static_assert(static_cast<int>(Mode::NONE) == proto::MultiroomState::NONE);
    static_assert(static_cast<int>(Mode::MASTER) == proto::MultiroomState::MASTER);
    static_assert(static_cast<int>(Mode::SLAVE) == proto::MultiroomState::SLAVE);

    Y_ENSURE_THREAD(lifecycle_);

    auto localClock = clockTowerState_->localClock;
    auto message = ipc::buildMessage([&](auto& msg) {
        auto& multiroomState = *msg.mutable_multiroom_state();
        multiroomState.set_version(MULTIROOM_STATE_VERSION);
        multiroomState.set_seqnum(++seqnum_);
        multiroomState.set_device_id(TString(myDeviceId_));
        multiroomState.set_mode(static_cast<proto::MultiroomState::Mode>(mode_));
        if (auto myIp = resolveMyIp(); !myIp.empty()) {
            multiroomState.set_ip_address(std::move(myIp));
            if (localClock) {
                multiroomState.set_net_audio_clock_host(multiroomState.ip_address());
                multiroomState.set_net_audio_clock_port(localClock->port());
                multiroomState.set_net_audio_clock_id(TString(localClock->clockId()));
            }
        }

        if (slaveClockSyncing_ && slaveClockSyncingExpiredAt_ > std::chrono::steady_clock::now()) {
            switch (*slaveClockSyncing_) {
                case IClock::SyncLevel::NONE:
                    multiroomState.set_slave_sync_level(proto::MultiroomState::NOSYNC);
                    break;
                case IClock::SyncLevel::WEAK:
                    multiroomState.set_slave_sync_level(proto::MultiroomState::WEAK);
                    break;
                case IClock::SyncLevel::STRONG:
                    multiroomState.set_slave_sync_level(proto::MultiroomState::STRONG);
                    break;
            }
            multiroomState.set_slave_clock_syncing(true);
        } else {
            multiroomState.set_slave_clock_syncing(false);
        }

        for (const auto& peer : peers_) {
            if (peer.second.handshakeTime.time_since_epoch().count() && !peer.second.deviceId.empty()) {
                auto p = multiroomState.add_peers();
                p->set_device_id(TString(peer.second.deviceId));
                p->set_ip_address(TString(peer.second.netClockHost));
            }

            if (peer.second.version <= 2) {
                // ALICE-19341 For JBL 80 version compatibility
                multiroomState.add_legacy_device_ids(TString(peer.second.deviceId));
            }
        }

        for (const auto& [deviceId, _] : muteTimeouts_) {
            if (!deviceId.empty()) {
                multiroomState.add_dialog_device_ids(TString(deviceId));
            }
        }

        if (mode_ == Mode::MASTER) {
            multiroomState.mutable_multiroom_broadcast()->CopyFrom(master_);
            multiroomState.mutable_multiroom_broadcast()->clear_master_hints();
        } else if (mode_ == Mode::SLAVE) {
            multiroomState.mutable_multiroom_broadcast()->CopyFrom(slave_);
        }
    });

    std::string newMultiroomState = hashMultiroomState(message->multiroom_state());
    if (auto wConnection = std::get_if<std::weak_ptr<ipc::IServer::IClientConnection>>(&target)) {
        if (auto sConnection = wConnection->lock()) {
            LOG_DEBUG_MULTIROOM("Send multiroom state to client \"" << sConnection.get() << "\": " << newMultiroomState);
            sConnection->send(message);
        }
    } else if (auto peer = std::get_if<std::string>(&target)) {
        auto it = peers_.find(*peer);
        if (it != peers_.end()) {
            LOG_DEBUG_MULTIROOM("Send multiroom state to peer \"" << it->first << "\": " << newMultiroomState);
            it->second.connector->sendMessage(message);
        }
    } else {
        if (!force) {
            if (newMultiroomState == lastBroadcastedMultiroomState_) {
                return;
            }
        }

        multiroomServer_->sendToAll(message);
        const auto mrState = YandexIO::convertMultiroomState(message->multiroom_state());
        deviceState_->setMultiroomState(mrState);
        glagolClusterProvider_->send(IGlagolClusterProvider::Target::REMOTE, SERVICE_NAME, message);
        lastBroadcastedMultiroomState_ = std::move(newMultiroomState);
        LOG_DEBUG_MULTIROOM("Send multiroom state: " << printMultiroomStateCompact(message->multiroom_state()));
    }
}

void MultiroomEndpoint::sendMasterState(bool force, const Target& target)
{
    Y_ENSURE_THREAD(lifecycle_);

    const auto clock = clockTowerState_->localClock;
    const auto myIp = resolveMyIp();
    const std::string_view clockId = (clock ? clock->clockId().c_str() : "");
    const int clockPort = (clock ? clock->port() : 0);

    if (!myIp.empty() && myIp != master_.net_audio_clock_host()) {
        LOG_INFO_MULTIROOM("Net clock ip address changed from " << (master_.net_audio_clock_host().empty() ? "<none>" : master_.net_audio_clock_host()) << " to " << myIp);
        master_.set_net_audio_clock_host(TString(myIp));
    }

    if (clockPort && clockPort != master_.net_audio_clock_port()) {
        LOG_INFO_MULTIROOM("Net clock port changed from " << (master_.net_audio_clock_port() ? "<none>" : std::to_string(master_.net_audio_clock_port())) << " to " << clockPort);
        master_.set_net_audio_clock_port(clockPort);
    }

    if (!clockId.empty() && clockId != master_.net_audio_clock_id()) {
        LOG_INFO_MULTIROOM("Net clock id changed from " << (master_.net_audio_clock_id().empty() ? "<none>" : master_.net_audio_clock_id()) << " to " << clockId);
        master_.set_net_audio_clock_id(TString(clockId));
    }

    auto message = ipc::buildMessage([&](auto& msg) {
        auto& broadcast = *msg.mutable_multiroom_broadcast();
        broadcast.CopyFrom(master_);
        broadcast.mutable_master_hints()->Clear();
        if (localDialogActivity_) {
            broadcast.mutable_master_hints()->set_quiet_audio_focus(true);
        }
    });

    std::string newMasterHash = hashMultiroomBroadcast(message->multiroom_broadcast());
    if (auto wConnection = std::get_if<std::weak_ptr<ipc::IServer::IClientConnection>>(&target)) {
        if (auto sConnection = wConnection->lock()) {
            LOG_DEBUG_MULTIROOM("Send master state to client \"" << sConnection.get() << "\": " << newMasterHash);
            sConnection->send(message);
        }
    } else if (auto peer = std::get_if<std::string>(&target)) {
        auto it = peers_.find(*peer);
        if (it != peers_.end()) {
            LOG_DEBUG_MULTIROOM("Send master state to peer " << it->first << ": " << lastBroadcastedMasterHash_);
            it->second.connector->sendMessage(message);
        }
    } else {
        if (!force) {
            if (newMasterHash == lastBroadcastedMasterHash_) {
                return;
            }
        }
        multiroomServer_->sendToAll(message);
        glagolClusterProvider_->send(IGlagolClusterProvider::Target::REMOTE, SERVICE_NAME, message);
        lastBroadcastedMasterHash_ = std::move(newMasterHash);
        lastBroadcastedMasterKeepAlive_ = std::chrono::steady_clock::now();
        LOG_DEBUG_MULTIROOM("Send master state: " << lastBroadcastedMasterHash_);

        if (mode_ == Mode::MASTER && isActiveMultiroomSession(master_) && master_.state() == proto::MultiroomBroadcast::PLAYING) {
            resetMasterStateCallback_.executeDelayed([this] { resetMasterSession(); }, RESET_INACTIVE_MASTER_STATE_PERIOD, lifetime_);
        }
    }
}

void MultiroomEndpoint::sendDirective(proto::MultiroomDirective directive)
{
    sendDirective(std::move(directive), (mode_ == Mode::SLAVE ? slave_ : master_));
}

void MultiroomEndpoint::sendDirective(proto::MultiroomDirective directive, const proto::MultiroomBroadcast& multiroomSession)
{
    auto message = ipc::buildMessage([&](auto& msg) {
        auto& multiroomDirective = *msg.mutable_multiroom_directive();
        multiroomDirective = std::move(directive);
        multiroomDirective.set_sender_device_id(TString(myDeviceId_));
        multiroomDirective.set_master_device_id(multiroomSession.device_id());
        multiroomDirective.set_session_timestamp_ms(multiroomSession.session_timestamp_ms());
        multiroomDirective.set_multiroom_token(multiroomSession.multiroom_token());
    });

    if (multiroomSession.device_id() == myDeviceId_) {
        multiroomServer_->sendToAll(message);
        glagolClusterProvider_->send(IGlagolClusterProvider::Target::REMOTE, SERVICE_NAME, message);
        LOG_INFO_MULTIROOM("Send directive to all slaves: " << message);
    } else {
        auto peerIt = peers_.find(multiroomSession.device_id());
        if (peerIt != peers_.end()) {
            LOG_INFO_MULTIROOM("Send directive to master " << peerIt->second.deviceId << ": " << message);
            peerIt->second.connector->sendMessage(message);
        }
    }
}

void MultiroomEndpoint::sendToDirectiveProcessor(const std::string& directiveName) {
    sendToDirectiveProcessor(YandexIO::Directive::createLocalAction(directiveName));
}

void MultiroomEndpoint::sendToDirectiveProcessor(std::shared_ptr<YandexIO::Directive> directive) {
    auto message = ipc::buildMessage([&](auto& msg) {
        msg.mutable_directive()->CopyFrom(YandexIO::Directive::convertToDirectiveProtobuf(directive));
        msg.mutable_directive()->set_is_route_locally(true);
    });
    LOG_INFO_MULTIROOM("Send directive to directive processor: " << convertMessageToDeepJsonString(*message));
    toAliced_->sendMessage(message);
}

void MultiroomEndpoint::noAudioFocusInformator()
{
    Y_ENSURE_THREAD(lifecycle_);

    if (localDialogActivity_ && mode_ != Mode::NONE) {
        proto::MultiroomDirective directive;
        directive.mutable_audio_focus()->set_initiator_device_id(TString(myDeviceId_));
        directive.mutable_audio_focus()->set_has_audio_focus(false);
        sendDirective(std::move(directive));
    }
    lifecycle_->addDelayed([this] { noAudioFocusInformator(); }, (NO_AUDIO_FOCUS_TIMEOUT * 2) / 3, lifetime_);
}

void MultiroomEndpoint::mute(const std::string& initiatorDeviceId)
{
    Y_ENSURE_THREAD(lifecycle_);

    LOG_DEBUG_MULTIROOM("Make quiet audio focus for " << initiatorDeviceId);
    bool hasInitiatorDeviceId = !!muteTimeouts_.count(initiatorDeviceId);
    muteTimeouts_[initiatorDeviceId] = std::chrono::steady_clock::now();
    if (!hasInitiatorDeviceId) {
        sendMultiroomState(true);
        if (muteTimeouts_.size() == 1) {
            LOG_DEBUG_MULTIROOM("Mute for multiroom");
        }
    }
    lifecycle_->addDelayed([this] { unmuteMultiroomIfNecessary(); }, NO_AUDIO_FOCUS_TIMEOUT, lifetime_);
}

void MultiroomEndpoint::encourageAudioFocus(const std::string& initiatorDeviceId)
{
    Y_ENSURE_THREAD(lifecycle_);
    LOG_DEBUG_MULTIROOM("Release quiet audio focus for " << initiatorDeviceId);
    bool hasInitiatorDeviceId = !!muteTimeouts_.count(initiatorDeviceId);
    muteTimeouts_.erase(initiatorDeviceId);
    if (hasInitiatorDeviceId) {
        sendMultiroomState(true);
    }
    unmuteMultiroomIfNecessary();
}

void MultiroomEndpoint::unmuteMultiroomIfNecessary()
{
    bool hasChanges = false;
    auto now = std::chrono::steady_clock::now();
    for (auto it = muteTimeouts_.begin(); it != muteTimeouts_.end();) {
        if (now - it->second >= NO_AUDIO_FOCUS_TIMEOUT) {
            LOG_DEBUG_MULTIROOM("Auto release quiet audio focus of " << it->first << " due timeout");
            it = muteTimeouts_.erase(it);
            hasChanges = true;
        } else {
            ++it;
        }
    }

    if (hasChanges) {
        sendMultiroomState(true);
    }
}

bool MultiroomEndpoint::isScheduledMultiroomBroadcast() const {
    Y_ENSURE_THREAD(lifecycle_);

    for (const auto& [_, peer] : peers_) {
        if (peer.multiroomBroadcastProcessor.isScheduled()) {
            return true;
        }
    }
    return false;
}

std::string MultiroomEndpoint::resolveMyIp() const {
    auto myIp = glagolClusterProvider_->publicIp().value();
    if (myIp.empty()) {
        myIp = getLocalAddress();
    }
    return myIp;
}

void MultiroomEndpoint::keepAlive()
{
    Y_ENSURE_THREAD(lifecycle_);

    resetTimedOutPeers();

    auto sendStatusPeriod = (mode_ == Mode::MASTER ? PEER_SEND_STATUS_PERIOD_MASTER : PEER_SEND_STATUS_PERIOD_NON_MASTER);
    if (lastBroadcastedMasterKeepAlive_ + sendStatusPeriod < std::chrono::steady_clock::now()) {
        sendMultiroomState(true);
        sendMasterState(true);
    }

    lifecycle_->addDelayed([this] { keepAlive(); }, PEER_CHECK_KEEP_ALIVE_PERIOD, lifetime_);
}

void MultiroomEndpoint::setMode(Mode mode)
{
    if (mode_ != mode) {
        mode_ = mode;
        std::string multiroomSessionId;
        bool resetDeferredBroadcasts = false;
        if (mode_ == Mode::MASTER) {
            multiroomSessionId = makeMultiroomSessionId(master_);
            resetDeferredBroadcasts = true;
        } else if (mode == Mode::SLAVE) {
            multiroomSessionId = makeMultiroomSessionId(slave_);
            resetDeferredBroadcasts = true;
        }
        LOG_INFO_MULTIROOM("Mode change to " << modeName(mode_) << ", multiroomSessionId=" << multiroomSessionId);
        if (resetDeferredBroadcasts) {
            for (auto& [ignore, peerRef] : peers_) {
                peerRef.multiroomBroadcastProcessor.reset();
            }
        }
    }
}

void MultiroomEndpoint::clearMaster()
{
    master_.Clear();
    master_.set_device_id(TString(myDeviceId_));
    master_.set_state(proto::MultiroomBroadcast::NONE);
    master_.set_version(MULTIROOM_BROADCAST_VERSION);
    master_.set_firmware(TString(device_->softwareVersion()));
}

void MultiroomEndpoint::metricaReporter()
{
    Y_ENSURE_THREAD(lifecycle_);

    const auto now = std::chrono::steady_clock::now();

    /*
     * Multiroom around
     */
    const auto localClockId = clockTowerState_->localClock->clockId();
    Json::Value values;
    std::set<std::string> totalVisiblePeers;
    for (const auto& p : peers_) {
        if (p.second.handshakeTime.time_since_epoch().count() &&
            p.second.keepAliveTime + PEER_KEEP_ALIVE_TIMEOUT > now &&
            !p.second.deviceId.empty())
        {
            totalVisiblePeers.insert(p.second.deviceId);
        }
    }
    values["totalVisiblePeers"] = totalVisiblePeers.size();

    if (mode_ == Mode::MASTER) {
        values["multiroomSessionId"] = makeMultiroomSessionId(master_);
    } else if (mode_ == Mode::SLAVE) {
        values["multiroomSessionId"] = makeMultiroomSessionId(slave_);
    }
    values["mode"] = modeName(mode_);
    values["clockId"] = localClockId;
    reportEvent("multiroomAround", values);

    lifecycle_->addDelayed([this] { metricaReporter(); }, METRICA_REPORTER_PERIOD, lifetime_);
}

void MultiroomEndpoint::reportEvent(const std::string& eventName, const Json::Value& values)
{
    device_->telemetry()->reportEvent(eventName, jsonToString(values));
    LOG_DEBUG_MULTIROOM("REPORT EVENT: event=" << eventName << ", json=" << jsonToString(values));
}

void MultiroomEndpoint::reportEventStartMaster(bool resume)
{
    Y_ENSURE_THREAD(lifecycle_);

    const auto now = std::chrono::steady_clock::now();
    Json::Value values;
    values["multiroomSessionId"] = makeMultiroomSessionId(master_);
    values["vinsRequestId"] = master_.vins_request_id();
    auto multiroomTrackId = makeMultiroomTrackId(master_);
    if (!multiroomTrackId.empty()) {
        values["multiroomTrackId"] = multiroomTrackId;
    }
    if (stereoPairState_->isStereoPair()) {
        values["stereoPair"] = StereoPairState::roleName(stereoPairState_->role);
    }
    values["visiblePeers"] = Json::arrayValue;
    for (const auto& p : peers_) {
        if (p.second.handshakeTime.time_since_epoch().count() &&
            p.second.keepAliveTime + PEER_KEEP_ALIVE_TIMEOUT > now &&
            !p.second.deviceId.empty())
        {
            values["visiblePeers"].append(p.second.deviceId);
        }
    }
    if (resume) {
        reportEvent("multiroomResumeMaster", values);
    }
    reportEvent("multiroomStartMaster", values);
    if (values["visiblePeers"].empty()) {
        values.removeMember("visiblePeers");
        reportEvent("multiroomStartMasterWithoutPeers", values);
    }
}

proto::MultiroomParams MultiroomEndpoint::makeMultiroomParamsFromAppState() const {
    proto::MultiroomParams multiroomParams;
    try {
        if (!lastMultiroomShortAppState_.isMaster()) {
            return multiroomParams;
        }
        multiroomParams.set_latency_ns(std::chrono::duration_cast<std::chrono::nanoseconds>(latency_.load()).count());
        multiroomParams.set_url(TString(lastMultiroomShortAppState_.url));
        if (lastMultiroomShortAppState_.sync) {
            multiroomParams.set_basetime_ns(lastMultiroomShortAppState_.basetimeNs);
            multiroomParams.set_position_ns(lastMultiroomShortAppState_.positionNs);
            if (lastMultiroomShortAppState_.normalization) {
                const auto& n = *lastMultiroomShortAppState_.normalization;
                multiroomParams.mutable_normalization()->set_true_peak(n.truePeak);
                multiroomParams.mutable_normalization()->set_integrated_loudness(n.integratedLoudness);
                multiroomParams.mutable_normalization()->set_target_lufs(n.targetLufs);
            }
        }

        if (lastMultiroomShortAppState_.player == MultiroomShortAppState::Player::AUDIO_CLIENT && lastAppState_->has_audio_player_state()) {
            multiroomParams.mutable_audio_params()->CopyFrom(lastAppState_->audio_player_state());
        }

        if (lastMultiroomShortAppState_.player == MultiroomShortAppState::Player::YANDEX_MUSIC && lastAppState_->has_music_state()) {
            auto musicParams = multiroomParams.mutable_music_params();
            musicParams->set_current_track_id(lastAppState_->music_state().current_track_id());
            musicParams->set_json_track_info(lastAppState_->music_state().json_track_info());
            musicParams->set_uid(lastAppState_->music_state().uid());
            if (lastAppState_->music_state().has_timestamp_ms()) {
                musicParams->set_timestamp_ms(lastAppState_->music_state().timestamp_ms());
            }
            musicParams->set_session_id(lastAppState_->music_state().session_id());
            musicParams->set_is_paused(lastAppState_->music_state().is_paused());
        }
    } catch (...) {
        multiroomParams.Clear();
    }
    return multiroomParams;
}

bool MultiroomEndpoint::isActiveMultiroomSession(const proto::MultiroomBroadcast& multiroomSession) noexcept {
    return multiroomSession.has_device_id() &&
           multiroomSession.has_multiroom_token() &&
           multiroomSession.has_session_timestamp_ms() &&
           multiroomSession.has_multiroom_params() &&
           !multiroomSession.multiroom_params().url().empty() &&
           multiroomSession.multiroom_params().has_basetime_ns();
}

bool MultiroomEndpoint::isSameMultiroomSession(const proto::MultiroomBroadcast& mb1, const proto::MultiroomBroadcast& mb2) noexcept {
    return mb1.device_id() == mb2.device_id() &&
           mb1.session_timestamp_ms() == mb2.session_timestamp_ms() &&
           mb1.multiroom_token() == mb2.multiroom_token();
}

bool MultiroomEndpoint::isSameMultiroomSession(const proto::MultiroomDirective& directive, const proto::MultiroomBroadcast& mb) noexcept {
    return directive.master_device_id() == mb.device_id() &&
           directive.session_timestamp_ms() == mb.session_timestamp_ms() &&
           directive.multiroom_token() == mb.multiroom_token();
}

bool MultiroomEndpoint::isSameMultiroomSource(const proto::MultiroomParams& mp1, const proto::MultiroomParams& mp2) noexcept {
    return mp1.url() == mp2.url() &&
           mp1.basetime_ns() == mp2.basetime_ns() &&
           mp1.position_ns() == mp2.position_ns();
}
