#pragma once

#include "multiroom_common.h"

#include <yandex_io/services/mediad/audio_clock_manager/i_audio_clock_manager.h>

#include <yandex_io/capabilities/device_state/interfaces/i_device_state_capability.h>

#include <yandex_io/sdk/interfaces/directive.h>
#include <yandex_io/sdk/sdk_interface.h>

#include <yandex_io/interfaces/clock_tower/i_clock_tower_provider.h>
#include <yandex_io/interfaces/glagol/i_glagol_cluster_provider.h>
#include <yandex_io/interfaces/stereo_pair/i_stereo_pair_provider.h>
#include <yandex_io/interfaces/user_config/i_user_config_provider.h>
#include <yandex_io/libs/audio_player/base/audio_player.h>
#include <yandex_io/libs/base/named_callback_queue.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/glagol_sdk/discovery.h>
#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/libs/signals/signal.h>
#include <yandex_io/libs/threading/lifetime.h>
#include <yandex_io/libs/threading/unique_callback.h>

#include <atomic>
#include <deque>
#include <unordered_map>
#include <variant>

namespace quasar {

    class MultiroomEndpoint {
    public:
        static const std::string SERVICE_NAME;
        enum class Mode {
            NONE,
            MASTER,
            SLAVE,
        };

        MultiroomEndpoint(
            std::shared_ptr<ICallbackQueue> lifecycle,
            std::shared_ptr<YandexIO::IDevice> device,
            std::shared_ptr<ipc::IIpcFactory> ipcFactory,
            std::shared_ptr<YandexIO::SDKInterface> sdk,
            std::shared_ptr<IClockTowerProvider> clockTowerProvider,
            std::shared_ptr<IGlagolClusterProvider> glagolClusterProvider,
            std::shared_ptr<IStereoPairProvider> stereoPairProvider,
            std::shared_ptr<IUserConfigProvider> userConfigProvider);
        ~MultiroomEndpoint();

        void startMultiroom(std::string vinsRequestId, std::string multiroomToken, std::vector<std::string> deviceIdList);
        void stopMultiroom(std::string vinsRequestId);
        void localDialogActivityChanged(bool localDialogActivity);

    private:
        using Target = std::variant<std::nullptr_t, std::string, std::weak_ptr<ipc::IServer::IClientConnection>>;

        void reconfigureMultirooms();
        void reconfigurePeers(const std::shared_ptr<const std::vector<std::string>>& deviceList);
        bool addPeer(const std::string& deviceId, bool discovered);
        bool resetTimedOutPeers();
        bool resetNonAccountPeers();

        bool onClientQuasarMessage(const std::string& peer, const ipc::SharedMessage& message) noexcept;
        void onMultiroomBroadcast(const std::string& peer, const proto::MultiroomBroadcast& multiroomBroadcast);
        bool onServerQuasarMessage(const std::string& peer, const ipc::SharedMessage& message) noexcept;

        void onDirectiveStop(const std::string& peer, const proto::MultiroomDirective& directive);
        void onDirectiveStopBroadcast(const std::string& peer, const proto::MultiroomDirective& directive);
        void onDirectiveCommand(const std::string& peer, const proto::MultiroomDirective& directive);
        void onDirectiveAudioFocus(const std::string& peer, const proto::MultiroomDirective& directive);
        void onDirectiveStateRequest(const std::string& peer, const proto::MultiroomDirective& directive);

        void onMultiroomStateReceived(const std::string& peer, const ipc::SharedMessage& message);

        void startMaster(std::string vinsRequestId, std::string multiroomToken, std::vector<std::string> roomDeviceIds);
        void stopMaster(const std::string& reason);
        void stopMasterPlayer(const std::string& reason);
        void stopAllSlaves(const std::string& reason);
        void stopSlavePlayer(const std::string& reason);
        void pauseSlavePlayer(const std::string& reason);
        void stopMultiroomSession(const proto::MultiroomBroadcast& multiroomSession, const std::string& reason);
        void resetSlaveSession(const std::string& deviceId, const std::string& reason);
        void resetMasterSession();
        void updateState(const std::shared_ptr<const proto::AppState>& appState);
        void updateState();
        void sendMultiroomState(bool force, const Target& target = Target{});
        void sendMasterState(bool force, const Target& target = Target{});
        void sendDirective(proto::MultiroomDirective directive);
        void sendDirective(proto::MultiroomDirective directive, const proto::MultiroomBroadcast& multiroomSession);
        void sendToDirectiveProcessor(const std::string& directiveName);
        void sendToDirectiveProcessor(std::shared_ptr<YandexIO::Directive> directive);

        void noAudioFocusInformator();
        void mute(const std::string& initiatorDeviceId);
        void encourageAudioFocus(const std::string& initiatorDeviceId);
        void unmuteMultiroomIfNecessary();
        void keepAlive();

        void setMode(Mode /*mode*/);
        void clearMaster();

        void metricaReporter();
        void reportEvent(const std::string& eventName, const Json::Value& values);
        void reportEventStartMaster(bool resume);
        bool isScheduledMultiroomBroadcast() const;
        std::string resolveMyIp() const;

        proto::MultiroomParams makeMultiroomParamsFromAppState() const;
        static bool isActiveMultiroomSession(const proto::MultiroomBroadcast& /*multiroomSession*/) noexcept;
        static bool isSameMultiroomSession(const proto::MultiroomBroadcast& mb1, const proto::MultiroomBroadcast& mb2) noexcept;
        static bool isSameMultiroomSession(const proto::MultiroomDirective& directive, const proto::MultiroomBroadcast& mb) noexcept;
        static bool isSameMultiroomSource(const proto::MultiroomParams& mp1, const proto::MultiroomParams& mp2) noexcept;

    private:
        Lifetime lifetime_;
        const std::shared_ptr<ICallbackQueue> lifecycle_;
        const std::shared_ptr<YandexIO::IDevice> device_;
        const std::shared_ptr<ipc::IIpcFactory> ipcFactory_;
        const std::shared_ptr<YandexIO::SDKInterface> sdk_;
        const std::shared_ptr<YandexIO::IDeviceStateCapability> deviceState_;
        const std::shared_ptr<IClockTowerProvider> clockTowerProvider_;
        const std::shared_ptr<IGlagolClusterProvider> glagolClusterProvider_;
        const std::shared_ptr<IStereoPairProvider> stereoPairProvider_;
        const std::string myDeviceId_;
        const std::shared_ptr<ipc::IServer> multiroomServer_;
        const std::shared_ptr<ipc::IConnector> toAliced_;

        // Only lifecycle_ thread access
        Json::Value multiroomConfig_;
        Mode mode_{Mode::NONE};
        class Peer2PeerConnector;
        struct Peer {
            std::string deviceId;
            std::string netClockId;
            std::string netClockHost;
            int netClockPort{0};
            uint32_t version{0};
            bool discovered{false};
            UniqueCallback multiroomBroadcastProcessor;
            std::shared_ptr<Peer2PeerConnector> connector;
            std::chrono::steady_clock::time_point discoveryTime;
            std::chrono::steady_clock::time_point handshakeTime;
            std::chrono::steady_clock::time_point keepAliveTime;
        };
        std::unordered_map<std::string, Peer> peers_;
        std::set<std::string> myDeviceIds_;
        std::atomic<std::chrono::milliseconds> latency_;
        std::atomic<bool> weakStereo_{false};
        std::shared_ptr<const proto::AppState> lastAppState_;
        MultiroomShortAppState lastMultiroomShortAppState_;
        std::shared_ptr<const ClockTowerState> clockTowerState_;
        std::shared_ptr<const StereoPairState> stereoPairState_;

        proto::MultiroomBroadcast master_;
        proto::MultiroomBroadcast slave_;
        uint32_t seqnum_{0};
        bool localDialogActivity_{false};
        std::unordered_map<std::string, std::chrono::steady_clock::time_point> muteTimeouts_;
        Json::Value resumeMasterParams_;
        struct MultiroomPending {
            std::string multiroomSessionId;
            std::string multiroomTrackId;
            std::string playPauseId;
            std::chrono::steady_clock::time_point until;
        };
        std::optional<MultiroomPending> multiroomPending_;
        std::optional<IClock::SyncLevel> slaveClockSyncing_;
        std::chrono::steady_clock::time_point slaveClockSyncingExpiredAt_;

        std::chrono::steady_clock::time_point lastBroadcastedMasterKeepAlive_;
        std::string lastBroadcastedMasterHash_;
        std::string lastBroadcastedMultiroomState_;
        std::string lastMultiroomUpdateLog_; // @@@ remove in release

        std::unordered_map<std::string, proto::MultiroomState> multiroomStates_; // for app metrica
        std::unordered_map<std::string, std::chrono::steady_clock::time_point> verboseLogOnMultiroomBroadcast_;

        UniqueCallback appStateUpdate_;
        UniqueCallback resetMasterStateCallback_;
    };

} // namespace quasar
