#include "multiroom_preprocessor.h"

#include "multiroom_directives.h"
#include "multiroom_log.h"

#include <yandex_io/services/aliced/capabilities/mrforwarder_capability/mrforwarder.h>

#include <yandex_io/interfaces/multiroom/i_multiroom_provider.h>

#include <yandex_io/libs/base/directives.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/json.h>

#include <util/system/yassert.h>

using namespace YandexIO;
using namespace quasar;

YIO_DEFINE_LOG_MODULE("multiroom_preprocessor");

namespace {
    const std::set<std::string> EACH_HOST_DIRECTIVES{
        Directives::SOUND_LOUDER,
        Directives::SOUND_QUITER,
        Directives::SOUND_MUTE,
        Directives::SOUND_UNMUTE,
        Directives::SOUND_SET_LEVEL,
    };

    const std::set<std::string> STOP_PLAYBACK_DIRECTIVES{
        Directives::CLEAR_QUEUE,
        Directives::PLAYBACK_NEXT,
        Directives::PLAYBACK_PREV,
        Directives::PLAYBACK_PAUSE,
        Directives::PLAYBACK_REWIND,
        Directives::PLAYBACK_TOGGLE_PLAY_PAUSE,
    };

    const std::set<std::string> MASTER_PLAYBACK_DIRECTIVES{
        Directives::CLEAR_QUEUE,
        Directives::PLAYBACK_NEXT,
        Directives::PLAYBACK_PREV,
        Directives::PLAYBACK_PLAY,
        Directives::PLAYBACK_PAUSE,
        Directives::PLAYBACK_REWIND,
        Directives::PLAYBACK_LIKE,
        Directives::PLAYBACK_DISLIKE,
        Directives::PLAYBACK_TOGGLE_PLAY_PAUSE,
        Directives::PLAYER_CONTINUE,       // Until we remove the fat multiroom https://st.yandex-team.ru/ALICE-19149
        Directives::PLAYER_NEXT_TRACK,     // https://st.yandex-team.ru/ALICE-19283
        Directives::PLAYER_PREVIOUS_TRACK, // https://st.yandex-team.ru/ALICE-19283
        Directives::PLAYER_REPLAY,         // https://st.yandex-team.ru/ALICE-19283
        Directives::PLAYER_LIKE,           // https://st.yandex-team.ru/ALICE-19283
        Directives::PLAYER_DISLIKE,        // https://st.yandex-team.ru/ALICE-19283
        Directives::PLAYER_REWIND,         // https://st.yandex-team.ru/ALICE-19283
    };
} // namespace

MultiroomPreprocessor::MultiroomPreprocessor(
    std::string deviceId,
    std::shared_ptr<IMultiroomProvider> multiroomProvider,
    std::shared_ptr<const AliceDeviceState> aliceDeviceState)
    : deviceId_(std::move(deviceId))
    , multiroomProvider_(std::move(multiroomProvider))
    , aliceDeviceState_(std::move(aliceDeviceState))
{
    Y_VERIFY(multiroomProvider_);
}

const std::string& MultiroomPreprocessor::getPreprocessorName() const {
    static const std::string s_name = "MultiroomPreprocessor";
    return s_name;
}

void MultiroomPreprocessor::preprocessDirectives(std::list<std::shared_ptr<Directive>>& directives)
{
    forwardDirectives(directives);
    ensureMultiroomToken(directives);

    auto state = multiroomProvider_->multiroomState().value();
    auto isMultiroomPlayer = (aliceDeviceState_->getCurrentPlayerType() == AudioPlayerType::MULTIROOM);
    auto isMultiroomPlaying = (isMultiroomPlayer && aliceDeviceState_->isMediaPlaying());

    for (auto it = directives.begin(), itNext = it; it != directives.end(); it = itNext) {
        ++(itNext = it);
        const auto& data = (*it)->getData();
        if (!data.endpointId.empty() && data.endpointId != deviceId_) {
            continue;
        }

        if (!data.multiroomSessionId.empty()) {
            if (state->multiroomSessionId() != data.multiroomSessionId) {
                LOG_INFO_MULTIROOM("Drop the \"" << data.name << "\" directive as the multiroom session id does not match the current one");
                LOG_INFO_MULTIROOM("    Current multiroom session id: " << (state->multiroomSessionId().empty() ? "<empty>" : state->multiroomSessionId()));
                LOG_INFO_MULTIROOM("    Expected multiroom session id: " << data.multiroomSessionId);
                directives.erase(it);
                continue;
            }

            if (state->mode == MultiroomState::Mode::MASTER) {
                if (EACH_HOST_DIRECTIVES.count(data.name)) {
                    LOG_DEBUG_MULTIROOM("Broadcast directive to all slaves "
                                        << "{" << join(state->peers, ", ", [](const auto& p) { return p.deviceId; }) << "}"
                                        << ": " << Directive::convertToJsonString(*it));
                    for (const auto& peer : state->peers) {
                        if (peer.deviceId == deviceId_) {
                            continue;
                        }
                        auto slaveData = data;
                        // ** NOTE ** While there is no normal support for endpoint_id, we have to send
                        //            the directive to your slaves, in such an exotic way. In the future,
                        //            the MRFORWARDER_ENDPOINT_ID value must be replaced with device_id
                        //            and roomDeviceIds field must be clear
                        Y_VERIFY(slaveData.roomDeviceIds.empty());
                        slaveData.endpointId = MRFORWARDER_ENDPOINT_ID;
                        slaveData.roomDeviceIds.clear();
                        slaveData.roomDeviceIds.push_back(peer.deviceId);
                        // ** NOTE ** There is currently no other way to break the cycle loop as the
                        //            endpoint_id is also calculated from the multiroom_session_id. In
                        //            the future, when the endpoint_id field is filled in by the server,
                        //            then only it will identify the device, and multiroom_session_id will
                        //            only be a function to accept or reject the directive.
                        slaveData.multiroomSessionId.clear();
                        auto slaveDirective = std::make_shared<Directive>(std::move(slaveData));
                        directives.insert(it, std::move(slaveDirective));
                    }
                }
            }
        }

        if (isMultiroomPlayer && state->broadcast && state->mode == MultiroomState::Mode::SLAVE) {
            // ** NOTE ** All playback directives from the slave must be forwarded to the master
            if (MASTER_PLAYBACK_DIRECTIVES.contains(data.name)) {
                auto masterData = data;
                masterData.endpointId = MRFORWARDER_ENDPOINT_ID;
                masterData.roomDeviceIds.clear();
                masterData.roomDeviceIds.push_back(state->broadcast->masterDeviceId);
                masterData.multiroomSessionId = state->multiroomSessionId();
                auto masterDirective = std::make_shared<Directive>(std::move(masterData));
                LOG_DEBUG_MULTIROOM("Forward playback directive \"" << data.name << "\" to master device " << state->broadcast->masterDeviceId << ": " << Directive::convertToJsonString(masterDirective));
                directives.insert(it, std::move(masterDirective));

                if (isMultiroomPlaying && STOP_PLAYBACK_DIRECTIVES.contains(data.name)) {
                    auto pause = MultiroomDirectives::createDirectiveSlavePause(state->multiroomSessionId(), "Playback stop directive (" + data.name + ")");
                    LOG_DEBUG_MULTIROOM("Replace directive \"" << data.name << "\" with multiroom one: " << Directive::convertToJsonString(pause));
                    directives.insert(it, std::move(pause));
                }
                directives.erase(it);
                continue;
            }
        }

        if ((*it)->is(Directives::ALARM_START)) {
            if (tryGetBool((*it)->getData().payload, "is_media_alarm", false)) {
                LOG_INFO_MULTIROOM("Add stop multiroom directive");
                auto directive = YandexIO::Directive::createLocalAction(Directives::STOP_MULTIROOM);
                directives.insert(it, std::move(directive));
            }
        }
    }
}

void MultiroomPreprocessor::ensureMultiroomToken(std::list<std::shared_ptr<YandexIO::Directive>>& directives)
{
    auto state = multiroomProvider_->multiroomState().value();

    for (auto iter = directives.begin(); iter != directives.end(); ++iter) {
        auto directive = *iter;
        auto& originData = directive->getData();
        if (!originData.endpointId.empty() && originData.endpointId != deviceId_) {
            continue;
        }

        if (directive->is(Directives::START_MULTIROOM) || directive->is(Directives::AUDIO_PLAY)) {
            std::string multiroomToken = tryGetString(originData.payload, "multiroom_token");
            if (multiroomToken.empty()) {
                const auto& vinsRequestId = originData.parentRequestId.empty() ? originData.requestId : originData.parentRequestId;
                if (lastMultiroomVinsRequestId_ == vinsRequestId) {
                    multiroomToken = lastMultiroomToken_;
                } else {
                    multiroomToken = makeMultiroomToken(vinsRequestId);
                }

                auto data = originData;
                data.payload.removeMember("room_id");
                data.payload["multiroom_token"] = multiroomToken;
                *iter = std::make_shared<Directive>(std::move(data));
            }

            if (directive->is(Directives::START_MULTIROOM)) {
                lastMultiroomToken_ = multiroomToken;
                lastMultiroomVinsRequestId_ = directive->getRequestId();
            }
        } else if (directive->is(Directives::MUSIC_PLAY)) {
            auto data = originData;
            if (state->mode == MultiroomState::Mode::MASTER && state->broadcast && state->broadcast->playingState == MultiroomState::PlayingState::PLAYING) {
                data.payload["multiroom_token"] = state->broadcast->multiroomToken;
            } else {
                data.payload["multiroom_token"] = makeMultiroomToken(originData.parentRequestId.empty() ? originData.requestId : originData.parentRequestId);
            }

            *iter = std::make_shared<Directive>(std::move(data));
        }

        directive = *iter;
        LOG_INFO_MULTIROOM(directive->getData().name << ":multiroom_token=" << tryGetString(directive->getData().payload, "multiroom_token"));
    }
}

void MultiroomPreprocessor::forwardDirectives(std::list<std::shared_ptr<YandexIO::Directive>>& directives)
{
    for (auto iter = directives.begin(); iter != directives.end(); ++iter) {
        const auto& originData = (*iter)->getData();
        if (!originData.endpointId.empty()) {
            continue;
        }

        if (!originData.roomDeviceIds.empty()) {
            auto newData = originData;
            for (const auto& deviceId : originData.roomDeviceIds) {
                if (deviceId == deviceId_) {
                    newData.roomDeviceIds.clear();
                    break;
                }
            }
            if (!newData.roomDeviceIds.empty()) {
                newData.endpointId = MRFORWARDER_ENDPOINT_ID;
            }

            *iter = std::make_shared<Directive>(std::move(newData));
        } else if (!originData.multiroomSessionId.empty()) {
            auto multiroomState = multiroomProvider_->multiroomState().value();
            if (multiroomState->broadcast->masterDeviceId == deviceId_ ||
                multiroomState->multiroomSessionId() != originData.multiroomSessionId) {
                continue;
            }

            if (originData.name == Directives::PLAYER_PAUSE) {
                directives.insert(iter, std::make_shared<Directive>(originData));
            }

            auto newData = originData;
            newData.endpointId = MRFORWARDER_ENDPOINT_ID;
            *iter = std::make_shared<Directive>(std::move(newData));
        }
    }
}
