#include "multiroom_log.h"
#include "multiroom_slave_player.h"

#include <yandex_io/interfaces/multiroom/i_multiroom_provider.h>

#include <yandex_io/libs/base/named_callback_queue.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/protobuf_utils/json.h>

#include <util/system/yassert.h>

YIO_DEFINE_LOG_MODULE("multiroom_slave_player");

using namespace quasar;

namespace {
    constexpr size_t MAX_HISTORY_SIZE = 4;
} // namespace

MultiroomSlavePlayer::MultiroomSlavePlayer(
    std::shared_ptr<ipc::IConnector> audioClientConnector)
    : audioClientConnector_(std::move(audioClientConnector))
{
    Y_VERIFY(audioClientConnector_);
}

MultiroomSlavePlayer::~MultiroomSlavePlayer() {
}

void MultiroomSlavePlayer::setLatency(std::chrono::milliseconds latencyMs) {
    latencyNs_ = std::chrono::duration_cast<std::chrono::nanoseconds>(latencyMs).count();
}

void MultiroomSlavePlayer::onAudioClientEvent(const proto::AudioClientEvent& event) {
    if (!playerState_) {
        return;
    }

    if (event.player_descriptor().player_id() != playerState_->player_descriptor().player_id()) {
        return;
    }

    if (event.event() != proto::AudioClientEvent::STATE_CHANGED) {
        return;
    }

    LOG_INFO_MULTIROOM("Slave multiroom player " << proto::AudioClientState_Name(event.state())
                                                 << " multiroomSessionId=" << makeMultiroomSessionId(multiroomBroadcast_->device_id(), multiroomBroadcast_->multiroom_token(), multiroomBroadcast_->session_timestamp_ms())
                                                 << ", multiroomTrackId=" << makeMultiroomTrackId(event.audio().url(), event.audio().basetime_ns(), event.audio().position_ns())
                                                 << ": " << convertMessageToDeepJsonString(event.player_descriptor()));
    switch (event.state()) {
        case proto::AudioClientState::FINISHED:
        case proto::AudioClientState::STOPPED:
        case proto::AudioClientState::FAILED:
            playerState_ = std::nullopt;
            multiroomBroadcast_ = std::nullopt;
            break;
        default: {
            playerState_->CopyFrom(event);
            break;
        }
    }
}

MultiroomPlayerDescriptor MultiroomSlavePlayer::play(const proto::MultiroomBroadcast& multiroomBroadcast) {
    LOG_INFO_MULTIROOM("MultiroomSlavePlayer.play: " << convertMessageToDeepJsonString(multiroomBroadcast));
    if (!multiroomBroadcast.has_multiroom_params()) {
        throw std::runtime_error("Invalid MultiroomBroadcast argument for MultiroomSlavePlayer::play");
    }
    auto message = ipc::buildMessage([&](auto& message) {
        auto& request = *message.mutable_media_request();
        auto& audio = *request.mutable_play_audio();

        buildAudio(audio, multiroomBroadcast);

        /*
         * Fix multiroom basetime offset due local latency settings
         */
        auto remoteLatencyNs = multiroomBroadcast.multiroom_params().latency_ns();
        auto basetimeOffsetNs = latencyNs_.load() - remoteLatencyNs;
        if (basetimeOffsetNs) {
            audio.set_basetime_ns(audio.basetime_ns() + basetimeOffsetNs);
            LOG_DEBUG_MULTIROOM("Create multiroom player with basetime offset: master latency=" << remoteLatencyNs << ", my latency=" << latencyNs_.load() << ", offset=" << (basetimeOffsetNs > 0 ? "+" : "") << basetimeOffsetNs);
        }

        /*
         * Validate multiroom params
         */
        if (audio.url().empty()) {
            const char* msg = "Fail to play multiroom: url is empty";
            YIO_LOG_ERROR_EVENT("MultiroomSlavePlayer.FailToPlay.EmptyUrl", msg);
            throw std::runtime_error(msg);
        }

        if (audio.clock_id().empty()) {
            const char* msg = "Fail to play multiroom: clock is empty";
            YIO_LOG_ERROR_EVENT("MultiroomSlavePlayer.FailToPlay.EmptyClock", msg);
            throw std::runtime_error(msg);
        }

        if (audio.clock_id() != multiroomBroadcast.net_audio_clock_id()) {
            const char* msg = "Fail to play multiroom: clock is mismatched";
            YIO_LOG_ERROR_EVENT("MultiroomSlavePlayer.FailToPlay.MismatchedClock", msg);
            throw std::runtime_error(msg);
        }

        if (audio.basetime_ns() <= 0 || audio.position_ns() < 0) {
            const char* msg = "Fail to play multiroom: basetime or position is invalid";
            YIO_LOG_ERROR_EVENT("MultiroomSlavePlayer.FailToPlay.InvalidParams", msg);
            throw std::runtime_error(msg);
        }

        auto& descriptor = *request.mutable_player_descriptor();
        descriptor.set_type(proto::AudioPlayerDescriptor::AUDIO);
        descriptor.set_is_multiroom_slave(true);
        descriptor.set_player_id(TString(makeUUID()));
        descriptor.set_stream_id(TString(audio.id()));
        descriptor.set_audio_channel(proto::AudioChannel::CONTENT_CHANNEL);
    });

    const auto& playerId = message->media_request().player_descriptor().player_id();
    const auto& streamId = message->media_request().player_descriptor().stream_id();
    const auto& audio = message->media_request().play_audio();

    playerState_.emplace();
    playerState_->set_state(proto::AudioClientState::IDLE);
    playerState_->mutable_audio()->CopyFrom(message->media_request().play_audio());
    playerState_->mutable_player_descriptor()->CopyFrom(message->media_request().player_descriptor());

    multiroomBroadcast_.emplace();
    multiroomBroadcast_->CopyFrom(multiroomBroadcast);

    std::string msid = makeMultiroomSessionId(multiroomBroadcast_->device_id(), multiroomBroadcast_->multiroom_token(), multiroomBroadcast_->session_timestamp_ms());
    std::string mtid = makeMultiroomTrackId(audio.url(), audio.basetime_ns(), audio.position_ns());
    MultiroomPlayerDescriptor mpd{std::move(msid), std::move(mtid), playerId, streamId};
    while (history_.size() >= MAX_HISTORY_SIZE) {
        history_.pop_front();
    }
    history_.push_back(mpd);

    LOG_DEBUG_MULTIROOM("Slave multiroom mediad request for playerId=" << playerId
                                                                       << ", multiroomSessionId=" << mpd.multiroomSessionId
                                                                       << ", multiroomTrackId=" << mpd.multiroomTrackId
                                                                       << ": REQ=" << convertMessageToDeepJsonString(*message)
                                                                       << ", BCST=" << convertMessageToDeepJsonString(multiroomBroadcast));

    audioClientConnector_->sendMessage(message);
    return mpd;
}

void MultiroomSlavePlayer::stop(std::string_view multiroomSessionId, std::string_view reason) {
    stop(multiroomSessionId, reason, StopAction::STOP);
}

void MultiroomSlavePlayer::pause(std::string_view multiroomSessionId, std::string_view reason) {
    stop(multiroomSessionId, reason, StopAction::PAUSE);
}

void MultiroomSlavePlayer::stop(std::string_view multiroomSessionId, std::string_view reason, StopAction stopAction) {
    const auto stopActionName = [](StopAction stopAction) {
        switch (stopAction) {
            case StopAction::STOP:
                return "STOP";
            case StopAction::PAUSE:
                return "PAUSE";
        }
        return "UNDEFINED";
    }(stopAction);

    LOG_INFO_MULTIROOM("MultiroomSlavePlayer.stop(" << stopActionName << "): multiroomSessionId=" << multiroomSessionId << ", reason=" << reason);

    bool fMatched = false;
    for (const auto& mpd : history_) {
        if (mpd.multiroomSessionId != multiroomSessionId) {
            continue;
        }
        auto message = ipc::buildMessage([&](auto& message) {
            auto& request = *message.mutable_media_request();
            auto& descriptor = *request.mutable_player_descriptor();

            descriptor.set_is_multiroom_slave(true);
            descriptor.set_player_id(TString(mpd.playerId));
            descriptor.set_stream_id(TString(mpd.streamId));
            if (stopAction == StopAction::PAUSE) {
                request.mutable_pause();
                request.set_fire_device_context(true);
            } else if (stopAction == StopAction::STOP) {
                request.mutable_clean_players();
            } else {
                throw std::runtime_error("Stop action not implemented");
            }
        });
        LOG_DEBUG_MULTIROOM("Slave multiroom mediad request for "
                            << "playerId=" << mpd.playerId
                            << ", multiroomSessionId=" << mpd.multiroomSessionId
                            << ", multiroomTrackId=" << mpd.multiroomTrackId
                            << ": REQ=" << convertMessageToDeepJsonString(*message));
        audioClientConnector_->sendMessage(message);
        fMatched = true;
    }

    if (!fMatched) {
        LOG_WARN_MULTIROOM("Fail to " << stopActionName << " multiroom player because requested multiroom session id \"" << multiroomSessionId << "\" mismatched with any one in history");
    }
}

void MultiroomSlavePlayer::buildAudio(proto::Audio& outAudio, const proto::MultiroomBroadcast& multiroomBroadcast)
{
    outAudio.Clear();

    const auto& multiroomParams = multiroomBroadcast.multiroom_params();
    if (multiroomParams.has_audio_params()) {
        outAudio.CopyFrom(multiroomParams.audio_params().audio());
        outAudio.clear_initial_offset_ms();
        outAudio.clear_set_pause();
        outAudio.clear_play_pause_id();
    } else {
        const auto& masterMusic = multiroomParams.music_params();
        const auto& trackInfo = tryParseJsonOrEmpty(masterMusic.json_track_info());

        outAudio.set_id(tryGetString(trackInfo, "id"));
        outAudio.set_type("Track");
        outAudio.set_url(multiroomParams.url());
        outAudio.clear_initial_offset_ms();
        outAudio.set_format(proto::Audio::MP3);
        // outAudio.set_position_sec(xxx);
        // outAudio.set_duration_sec(xxx);
        outAudio.set_context("{}");
        if (auto analyticsContext = outAudio.mutable_analytics_context()) {
            analyticsContext->set_vins_request_id(multiroomBroadcast.vins_request_id());
            analyticsContext->set_name("multiroom_play");
        }
        if (auto metadata = outAudio.mutable_metadata()) {
            metadata->set_title(tryGetString(trackInfo, "title"));
            metadata->set_art_image_url(tryGetString(trackInfo, "coverUri"));
            metadata->set_hide_progress_bar(true);
            if (trackInfo.isMember("artists")) {
                const auto& artists = trackInfo["artists"];
                if (artists.isArray() && artists.size() > 0) {
                    const auto& artist = artists[0];
                    metadata->mutable_music_metadata()->set_id(tryGetString(artist, "id"));
                    metadata->mutable_music_metadata()->set_type("Artist");
                    metadata->mutable_music_metadata()->set_description(tryGetString(artist, "name"));
                    metadata->set_subtitle(tryGetString(artist, "name"));
                }
            }
        }
        // outAudio->set_provider_name(xxx);
        outAudio.set_screen_type(proto::Audio::Screen::Audio_Screen_DEFAULT);
        // outAudio->set_played_sec(xxx);
        outAudio.set_basetime_ns(multiroomParams.basetime_ns());
        outAudio.set_position_ns(multiroomParams.position_ns());
        if (multiroomParams.has_normalization()) {
            outAudio.mutable_normalization()->CopyFrom(multiroomParams.normalization());
        }
        outAudio.set_clock_id(multiroomBroadcast.net_audio_clock_id());
        outAudio.set_multiroom_token(multiroomBroadcast.multiroom_token());
    }
}
