#include "playback_control_capability.h"

#include <yandex_io/capabilities/device_state/converters/converters.h>
#include <yandex_io/libs/base/directives.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/services/aliced/capabilities/alice_capability/directives/alice_request_directive.h>
#include <yandex_io/services/aliced/capabilities/multiroom_capability/multiroom_directives.h>

YIO_DEFINE_LOG_MODULE("playback_control_capability");

using namespace quasar;
using namespace YandexIO;

PlaybackControlCapability::PlaybackControlCapability(
    std::shared_ptr<quasar::ICallbackQueue> worker,
    const std::shared_ptr<YandexIO::IDevice>& device,
    std::shared_ptr<YandexIO::IDirectiveProcessor> directiveProcessor,
    const quasar::AliceDeviceState& aliceDeviceState,
    std::shared_ptr<ipc::IConnector> glagolConnector,
    std::shared_ptr<quasar::ipc::IConnector> interfaceConnector,
    std::weak_ptr<IRemotingRegistry> remoteRegistry)
    : IRemoteObject(std::move(remoteRegistry))
    , worker_(std::move(worker))
    , directiveProcessor_(std::move(directiveProcessor))
    , aliceDeviceState_(aliceDeviceState)
    , hasInterfaced_(device->configuration()->hasServiceConfig("interfaced"))
    , glagolConnector_(std::move(glagolConnector))
    , interfaceConnector_(std::move(interfaceConnector))
{
}

void PlaybackControlCapability::init() {
    if (const auto registry = getRemotingRegistry().lock()) {
        registry->addRemoteObject(getHandlerName(), weak_from_this());
    }
}

void PlaybackControlCapability::onPassportUidChanged()
{
    broadcastSimplePlayerState();
}

void PlaybackControlCapability::onQuasarMessage(const quasar::ipc::SharedMessage& sharedMessage)
{
    if (sharedMessage->has_simple_player_state()) { // from interfaced
        showSimplePlayer_ = sharedMessage->simple_player_state().show_player();
        glagolConnector_->sendMessage(sharedMessage);
    }
}

void PlaybackControlCapability::onDeviceStatePart(const yandex_io::proto::TDeviceStatePart& statePart)
{
    if (statePart.has_video()) {
        const auto newScreen = statePart.video().currentscreen();

        showSimplePlayer_ = newScreen == yandex_io::proto::TVideo_ScreenType_music_player ||
                            newScreen == yandex_io::proto::TVideo_ScreenType_radio_player ||
                            newScreen == yandex_io::proto::TVideo_ScreenType_video_player;

        const auto prevScreen = std::exchange(currentScreenType_, newScreen);
        if (showSimplePlayer_ || prevScreen != newScreen) {
            const auto message = formatVideoSimplePlayerState(statePart.video());
            glagolConnector_->sendMessage(message);
        }
    }
}

void PlaybackControlCapability::setTestpointPeer(std::shared_ptr<TestpointPeer> testpointPeer)
{
    testpointPeer_ = std::move(testpointPeer);
}

const std::string& PlaybackControlCapability::getHandlerName() const {
    static const std::string NAME = "PlaybackControlCapability";
    return NAME;
}

const std::set<std::string>& PlaybackControlCapability::getSupportedDirectiveNames() const {
    static const std::set<std::string> SUPPORTED_DIRECTIVES = {
        Directives::PLAYBACK_PLAY,
        Directives::PLAYBACK_PAUSE,
        Directives::PLAYBACK_LIKE,
        Directives::PLAYBACK_DISLIKE,
        Directives::PLAYBACK_NEXT,
        Directives::PLAYBACK_PREV,
        Directives::PLAYBACK_REWIND,
        Directives::PLAYBACK_TOGGLE_PLAY_PAUSE,
    };
    return SUPPORTED_DIRECTIVES;
}

void PlaybackControlCapability::handleDirective(const std::shared_ptr<Directive>& directive) {
    if (directive->is(Directives::PLAYBACK_NEXT)) {
        handleNext(std::make_shared<Directive>(*directive));
    } else if (directive->is(Directives::PLAYBACK_PREV)) {
        handlePrev(std::make_shared<Directive>(*directive));
    } else if (directive->is(Directives::PLAYBACK_PLAY)) {
        handlePlay(std::make_shared<Directive>(*directive));
    } else if (directive->is(Directives::PLAYBACK_PAUSE)) {
        handlePause(std::make_shared<Directive>(*directive));
    } else if (directive->is(Directives::PLAYBACK_REWIND)) {
        handleRewind(std::make_shared<Directive>(*directive));
    } else if (directive->is(Directives::PLAYBACK_LIKE)) {
        handleLike(std::make_shared<Directive>(*directive));
    } else if (directive->is(Directives::PLAYBACK_DISLIKE)) {
        handleDislike(std::make_shared<Directive>(*directive));
    } else if (directive->is(Directives::PLAYBACK_TOGGLE_PLAY_PAUSE)) {
        handleTogglePlayPause(std::make_shared<Directive>(*directive));
    }
}

void PlaybackControlCapability::cancelDirective(const std::shared_ptr<Directive>& directive) {
    Y_UNUSED(directive);
}

void PlaybackControlCapability::prefetchDirective(const std::shared_ptr<Directive>& directive) {
    Y_UNUSED(directive);
}

void PlaybackControlCapability::onCapabilityStateChanged(
    const std::shared_ptr<YandexIO::ICapability>& capability, const NAlice::TCapabilityHolder& state) {
    Y_UNUSED(capability);

    if (!state.HasDeviceStateCapability()) {
        return;
    }

    state_ = state.GetDeviceStateCapability().GetState();

    broadcastSimplePlayerState();
}

void PlaybackControlCapability::onCapabilityEvents(
    const std::shared_ptr<YandexIO::ICapability>& capability, const std::vector<NAlice::TCapabilityEvent>& events) {
    Y_UNUSED(capability);
    Y_UNUSED(events);
}

void PlaybackControlCapability::togglePlayPause(bool canRequestMusic) {
    YandexIO::Directive::Data data(Directives::PLAYBACK_TOGGLE_PLAY_PAUSE, "local_action");
    data.payload["can_request_music"] = canRequestMusic;
    directiveProcessor_->addDirectives({std::make_shared<YandexIO::Directive>(std::move(data))});
}

void PlaybackControlCapability::play() {
    YandexIO::Directive::Data data(Directives::PLAYBACK_PLAY, "local_action");
    data.payload["fire_device_context"] = true;
    directiveProcessor_->addDirectives({std::make_shared<YandexIO::Directive>(std::move(data))});
}

void PlaybackControlCapability::pause() {
    YandexIO::Directive::Data data(Directives::PLAYBACK_PAUSE, "local_action");
    data.payload["fire_device_context"] = true;
    directiveProcessor_->addDirectives({std::make_shared<YandexIO::Directive>(std::move(data))});
}

void PlaybackControlCapability::rewind(std::uint32_t time) {
    YandexIO::Directive::Data data(Directives::PLAYBACK_REWIND, "local_action");
    data.payload["amount"] = time;
    data.payload["type"] = "absolute";
    directiveProcessor_->addDirectives({std::make_shared<YandexIO::Directive>(std::move(data))});
}

void PlaybackControlCapability::like() {
    YandexIO::Directive::Data data(Directives::PLAYBACK_LIKE, "local_action");
    data.payload["fire_device_context"] = true;
    directiveProcessor_->addDirectives({std::make_shared<YandexIO::Directive>(std::move(data))});
}

void PlaybackControlCapability::dislike() {
    YandexIO::Directive::Data data(Directives::PLAYBACK_DISLIKE, "local_action");
    data.payload["fire_device_context"] = true;
    directiveProcessor_->addDirectives({std::make_shared<YandexIO::Directive>(std::move(data))});
}

void PlaybackControlCapability::next() {
    YandexIO::Directive::Data data(Directives::PLAYBACK_NEXT, "local_action");
    data.payload["fire_device_context"] = true;
    data.isLedSilent = false;
    directiveProcessor_->addDirectives({std::make_shared<YandexIO::Directive>(std::move(data))});
}

void PlaybackControlCapability::prev() {
    YandexIO::Directive::Data data(Directives::PLAYBACK_PREV, "local_action");
    data.payload["fire_device_context"] = true;
    data.payload["forced"] = false;
    data.isLedSilent = false;
    directiveProcessor_->addDirectives({std::make_shared<YandexIO::Directive>(std::move(data))});
}

bool PlaybackControlCapability::isLegacyPlayer(const AudioPlayerType& playerType) const {
    return playerType == AudioPlayerType::NONE || playerType == AudioPlayerType::MUSIC || playerType == AudioPlayerType::RADIO || aliceDeviceState_.hasVideoPlayerScreen();
}

void PlaybackControlCapability::handleNext(std::shared_ptr<YandexIO::Directive> directive)
{
    const auto playerType = aliceDeviceState_.getCurrentPlayerType();

    if (playerType == AudioPlayerType::BLUETOOTH) {
        directive->setName(Directives::BLUETOOTH_PLAYER_NEXT);
    } else if (isLegacyPlayer(playerType) && playerType != AudioPlayerType::RADIO) {
        directive->setName(Directives::PLAYER_NEXT_TRACK);
    } else {
        directive->setName(Directives::AUDIO_CLIENT_NEXT_TRACK);
    }

    directiveProcessor_->addDirectives({std::move(directive)});
}

void PlaybackControlCapability::handlePrev(std::shared_ptr<YandexIO::Directive> directive)
{
    const auto playerType = aliceDeviceState_.getCurrentPlayerType();

    if (playerType == AudioPlayerType::BLUETOOTH) {
        directive->setName(Directives::BLUETOOTH_PLAYER_PREV);
    } else if (isLegacyPlayer(playerType) && playerType != AudioPlayerType::RADIO) {
        const auto progress = aliceDeviceState_.getMusicStateProgress();
        const bool forced = tryGetBool(directive->getData().payload, "forced", false);
        const bool fireDeviceContext = tryGetBool(directive->getData().payload, "fire_device_context", false);

        if (!forced && fireDeviceContext && progress > std::chrono::seconds(5)) {
            YandexIO::Directive::Data data(Directives::PLAYER_REWIND, "local_action");
            data.payload["amount"] = 0;
            data.payload["type"] = "absolute";

            auto rewindDirective = std::make_shared<YandexIO::Directive>(std::move(data));
            directiveProcessor_->addDirectives({std::move(rewindDirective)});

            return;
        } else {
            directive->setName(Directives::PLAYER_PREVIOUS_TRACK);
        }
    } else {
        directive->setName(Directives::AUDIO_CLIENT_PREV_TRACK);
    }

    directiveProcessor_->addDirectives({std::move(directive)});
}

void PlaybackControlCapability::handlePlay(std::shared_ptr<YandexIO::Directive> directive) {
    const auto playerType = aliceDeviceState_.getCurrentPlayerType();
    const bool isPlayerScreenOrNoScreens =
        !hasInterfaced_ || aliceDeviceState_.hasMusicOrBluetoothPlayerScreen();

    const bool hasAudioPlayer = playerType != AudioPlayerType::NONE;
    YIO_LOG_INFO("currentPlayerType=" << audioPlayerTypeName(playerType) << ", isPlayerScreenOrNoScreens=" << isPlayerScreenOrNoScreens);

    if (isPlayerScreenOrNoScreens && hasAudioPlayer) {
        if (playerType == AudioPlayerType::BLUETOOTH) {
            directive->setName(Directives::BLUETOOTH_PLAYER_PLAY);
        } else if (!isLegacyPlayer(playerType)) {
            auto request = VinsRequest::createEventRequest(
                VinsRequest::buildPlayerContinueSemanticFrame(),
                VinsRequest::createSoftwareDirectiveEventSource(),
                directive->getRequestId());
            request->setIsSilent(true);
            auto requestDirective = std::make_shared<AliceRequestDirective>(std::move(request), nullptr, true);

            directiveProcessor_->addDirectives({requestDirective});
            return;
        } else {
            directive->setName(Directives::PLAYER_CONTINUE);
            directive->setPayload(Json::Value(Json::objectValue));
        }
    } else {
        Json::Value payload;
        payload["player"] = "video";
        directive->setPayload(std::move(payload));
        directive->setName(Directives::PLAYER_CONTINUE);
    }

    directiveProcessor_->addDirectives({std::move(directive)});
}

void PlaybackControlCapability::handlePause(std::shared_ptr<YandexIO::Directive> directive) {
    const auto playerType = aliceDeviceState_.getCurrentPlayerType();

    if (playerType == AudioPlayerType::BLUETOOTH) {
        directive->setName(Directives::BLUETOOTH_PLAYER_PAUSE);
    } else if (isLegacyPlayer(playerType)) {
        directive->setName(Directives::PLAYER_PAUSE);
    } else {
        directive->setName(Directives::AUDIO_STOP);
    }

    directiveProcessor_->addDirectives({std::move(directive)});
}

void PlaybackControlCapability::handleRewind(std::shared_ptr<YandexIO::Directive> directive) {
    const auto playerType = aliceDeviceState_.getCurrentPlayerType();
    if (isLegacyPlayer(playerType)) {
        directive->setName(Directives::PLAYER_REWIND);
    } else {
        directive->setName(Directives::AUDIO_REWIND);
    }

    directiveProcessor_->addDirectives({std::move(directive)});
}

void PlaybackControlCapability::handleLike(std::shared_ptr<YandexIO::Directive> directive) {
    const auto playerType = aliceDeviceState_.getCurrentPlayerType();
    if (isLegacyPlayer(playerType)) {
        directive->setName(Directives::PLAYER_LIKE);
    } else {
        // TODO
        return;
    }
    directiveProcessor_->addDirectives({std::move(directive)});
}

void PlaybackControlCapability::handleDislike(std::shared_ptr<YandexIO::Directive> directive) {
    const auto playerType = aliceDeviceState_.getCurrentPlayerType();
    if (isLegacyPlayer(playerType)) {
        directive->setName(Directives::PLAYER_DISLIKE);
    } else {
        // TODO
        return;
    }
    directiveProcessor_->addDirectives({std::move(directive)});
}

void PlaybackControlCapability::handleTogglePlayPause(std::shared_ptr<YandexIO::Directive> directive) {
    auto canRequestMusic = tryGetBool(directive->getData().payload, "can_request_music", false);
    if (canRequestMusic && !aliceDeviceState_.hasActivePlayer()) {
        YIO_LOG_INFO("Request music as no player is active");
        // TogglePlay is called only for explicit user actions, mainly buttons pushing.

        VinsRequest::EventSource eventSource;
        eventSource.set_event(NAlice::TSpeechKitRequestProto_TEventSource_EEvent_Click);
        eventSource.set_source(NAlice::TSpeechKitRequestProto_TEventSource_ESource_Hardware);
        eventSource.set_type(NAlice::TSpeechKitRequestProto_TEventSource_EType_Text);
        eventSource.set_id(TString(makeUUID()));
        auto request = VinsRequest::createEventRequest(VinsRequest::buildStartMusicSemanticFrame(), std::move(eventSource));
        auto directive = std::make_shared<AliceRequestDirective>(std::move(request), nullptr, true);
        directiveProcessor_->addDirectives({std::move(directive)});
        return;
    }

    if (aliceDeviceState_.isMediaPlaying()) {
        YIO_LOG_INFO("Pausing current playback");
        YandexIO::Directive::Data data(Directives::PLAYBACK_PAUSE, "local_action");
        data.payload["fire_device_context"] = true;
        handlePause(std::make_shared<YandexIO::Directive>(std::move(data)));
    } else {
        YIO_LOG_INFO("Resuming current playback");
        YandexIO::Directive::Data data(Directives::PLAYBACK_PLAY, "local_action");
        data.payload["fire_device_context"] = true;
        handlePlay(std::make_shared<YandexIO::Directive>(std::move(data)));
    }
}

void PlaybackControlCapability::handleRemotingMessage(const quasar::proto::Remoting& message, std::shared_ptr<YandexIO::IRemotingConnection> /*connection*/) {
    if (message.has_playback_control_capability_method()) {
        const auto& method = message.playback_control_capability_method();

        if (!method.has_method()) {
            YIO_LOG_ERROR_EVENT("PlaybackControlCapability.RemoteMethodFailed", "Missing method in remote call");
            return;
        }

        switch (method.method()) {
            case quasar::proto::Remoting::PlaybackControlCapabilityMethod::PLAY: {
                play();
                return;
            }
            case quasar::proto::Remoting::PlaybackControlCapabilityMethod::PAUSE: {
                pause();
                return;
            }
            case quasar::proto::Remoting::PlaybackControlCapabilityMethod::LIKE: {
                like();
                return;
            }
            case quasar::proto::Remoting::PlaybackControlCapabilityMethod::DISLIKE: {
                dislike();
                return;
            }
            case quasar::proto::Remoting::PlaybackControlCapabilityMethod::NEXT: {
                next();
                return;
            }
            case quasar::proto::Remoting::PlaybackControlCapabilityMethod::PREV: {
                prev();
                return;
            }
            case quasar::proto::Remoting::PlaybackControlCapabilityMethod::REWIND: {
                if (!method.has_rewind_amount()) {
                    YIO_LOG_ERROR_EVENT("PlaybackControlCapability.RemoteMethodFailed", "Missing \"rewind_amount\" argument in REWIND remote call");
                    return;
                }
                rewind(method.rewind_amount());
                return;
            }
            case quasar::proto::Remoting::PlaybackControlCapabilityMethod::TOGGLE_PLAY_PAUSE: {
                if (!method.has_can_request_music()) {
                    YIO_LOG_ERROR_EVENT("PlaybackControlCapability.RemoteMethodFailed", "Missing \"can_request_music\" argument in TOGGLE_PLAY_PAUSE remote call");
                    return;
                }
                togglePlayPause(method.can_request_music());
                return;
            }
            default: {
                const auto* descriptor = quasar::proto::Remoting::PlaybackControlCapabilityMethod::Method_descriptor();
                YIO_LOG_ERROR_EVENT("PlaybackControlCapability.RemoteMethodFailed", "Unknown method: " << descriptor->FindValueByNumber(method.method())->name());
                return;
            }
        }
    }
}

ipc::SharedMessage PlaybackControlCapability::formatVideoSimplePlayerState(const yandex_io::proto::TVideo& video) const {
    return ipc::buildMessage([&](auto& msg) {
        auto simplePlayerState = msg.mutable_simple_player_state();
        simplePlayerState->set_show_player(showSimplePlayer_);

        if (video.has_currentlyplaying()) {
            const auto& currentlyPlaying = video.currentlyplaying();
            if (!currentlyPlaying.has_rawitem()) {
                YIO_LOG_ERROR_EVENT("PlaybackControlCapability.FailedToFormatSimplePlayerState", "No currentlyPlaying.rawitem");
                return;
            }

            const auto& item = currentlyPlaying.rawitem();
            simplePlayerState->set_id(item.provideritemid());
            simplePlayerState->set_type(item.type());
            simplePlayerState->set_title(item.name());

            if (currentlyPlaying.has_rawtvshowitem()) {
                const auto& tvShow = currentlyPlaying.rawtvshowitem();
                simplePlayerState->set_subtitle(tvShow.name());
            }

            if (currentlyPlaying.has_progress()) {
                const auto& progress = currentlyPlaying.progress();
                if (progress.played() > 0) {
                    simplePlayerState->set_position(progress.played());
                }
                if (progress.duration() > 0) {
                    simplePlayerState->set_duration(progress.duration());
                }
                simplePlayerState->set_has_progress_bar(true);
            }
        }

        if (video.has_player()) {
            const auto& player = video.player();
            simplePlayerState->set_has_pause(!player.pause());
            simplePlayerState->set_has_play(player.pause());
        }
        simplePlayerState->set_player_type(video.playername());
    });
}

int64_t PlaybackControlCapability::getAudioPlayerLastPlayTs() const {
    if (state_.HasDeviceState() && state_.GetDeviceState().HasAudioPlayer()) {
        return state_.GetDeviceState().GetAudioPlayer().GetLastPlayTimestamp();
    }

    return 0;
}

int64_t PlaybackControlCapability::getAudioPlayerLastStopTs() const {
    if (state_.HasDeviceState() && state_.GetDeviceState().HasAudioPlayer()) {
        return state_.GetDeviceState().GetAudioPlayer().GetLastStopTimestamp();
    }

    return 0;
}

ipc::SharedMessage PlaybackControlCapability::formatValidSimplePlayerState() const {
    auto state = formatSimplePlayerState();
    if (state->has_simple_player_state()) {
        return state;
    }

    return ipc::UniqueMessage::create();
}

ipc::SharedMessage PlaybackControlCapability::formatSimplePlayerState() const {
    const int64_t oldPlayerLastPlayTime = aliceDeviceState_.getLastLegacyPlayerStart();
    const int64_t newPlayerLastPlayTime = getAudioPlayerLastPlayTs();
    const int64_t oldPlayerLastStopTime = aliceDeviceState_.getLastLegacyPlayerStop();
    const int64_t newPlayerLastStopTime = getAudioPlayerLastStopTs();
    const int64_t radioTs = aliceDeviceState_.getLastRadioPlayerStart();

    if (radioTs > oldPlayerLastPlayTime && radioTs > newPlayerLastPlayTime) {
        return formatRadioSimplePlayerState();
    } else if (!oldPlayerLastPlayTime && !newPlayerLastPlayTime) {
        return ipc::UniqueMessage::create();
    }

    const bool oldPlaying = oldPlayerLastPlayTime > oldPlayerLastStopTime;
    const bool newPlaying = newPlayerLastPlayTime > newPlayerLastStopTime;

    bool isAboutOldPlayer;
    if (oldPlaying && newPlaying) {
        isAboutOldPlayer = oldPlayerLastPlayTime > newPlayerLastPlayTime;
    } else if (!oldPlaying && !newPlaying) {
        isAboutOldPlayer = oldPlayerLastStopTime > newPlayerLastStopTime;
    } else {
        isAboutOldPlayer = oldPlaying;
    }

    if (isAboutOldPlayer) {
        return formatMusicSimplePlayerState();
    } else {
        return formatAudioClientSimplePlayerState();
    }
}

void PlaybackControlCapability::broadcastSimplePlayerState()
{
    if (const auto message = formatValidSimplePlayerState()) {
        glagolConnector_->sendMessage(message);
        if (interfaceConnector_ != nullptr) {
            interfaceConnector_->sendMessage(message);
        }
        if (testpointPeer_ != nullptr) {
            testpointPeer_->sendMessage(message);
        }
    }
}

ipc::SharedMessage PlaybackControlCapability::formatMusicSimplePlayerState() const {
    if (state_.HasDeviceState() && state_.GetDeviceState().HasMusic()) {
        return ipc::buildMessage([&](auto& msg) {
            const auto& music = state_.GetDeviceState().GetMusic();
            msg.mutable_simple_player_state()->CopyFrom(convertMusicToSimplePlayerState(music, showSimplePlayer_));
        });
    }

    return ipc::UniqueMessage::create();
}

ipc::SharedMessage PlaybackControlCapability::formatRadioSimplePlayerState() const {
    if (state_.HasDeviceState() && state_.GetDeviceState().HasRadio()) {
        return ipc::buildMessage([&](auto& msg) {
            const auto& radio = state_.GetDeviceState().GetRadio();
            msg.mutable_simple_player_state()->CopyFrom(convertRadioToSimplePlayerState(radio, showSimplePlayer_));
        });
    }

    return ipc::UniqueMessage::create();
}

ipc::SharedMessage PlaybackControlCapability::formatAudioClientSimplePlayerState() const {
    if (state_.HasDeviceState() && state_.GetDeviceState().HasAudioPlayer()) {
        return ipc::buildMessage([&](auto& msg) {
            const auto& audioPlayer = state_.GetDeviceState().GetAudioPlayer();
            msg.mutable_simple_player_state()->CopyFrom(convertAudioPlayerToSimplePlayerState(audioPlayer, showSimplePlayer_));
        });
    }

    return ipc::UniqueMessage::create();
}
