#include "screen_capability.h"

#include <yandex_io/libs/base/directives.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/ete_metrics/ete_util.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

#include <yandex_io/capabilities/device_state/converters/converters.h>

#include <util/system/yassert.h>

YIO_DEFINE_LOG_MODULE("screen_capability");

using namespace quasar;

namespace YandexIO {

    ScreenCapability::ScreenCapability(
        std::shared_ptr<IDeviceStateCapability> deviceStateCapability,
        std::shared_ptr<quasar::ICallbackQueue> worker,
        IDirectiveProcessorWeakPtr directiveProcessor,
        const quasar::AliceDeviceState& aliceDeviceState,
        std::shared_ptr<quasar::ipc::IConnector> interfacedConnector)
        : deviceStateCapability_(std::move(deviceStateCapability))
        , interfaceConnector_(std::move(interfacedConnector))
        , worker_(std::move(worker))
        , directiveProcessor_(std::move(directiveProcessor))
        , aliceDeviceState_(aliceDeviceState)
    {
        Y_VERIFY(worker_);
        Y_VERIFY(interfaceConnector_);
        Y_VERIFY(deviceStateCapability_);
    }

    const std::string& ScreenCapability::getHandlerName() const {
        static const std::string s_name = "ScreenCapability";
        return s_name;
    }

    const std::set<std::string>& ScreenCapability::getSupportedDirectiveNames() const {
        static const std::set<std::string> s_names = {
            quasar::Directives::SCREEN_ON,
            quasar::Directives::SCREEN_OFF,
            quasar::Directives::VIDEO_PLAY,
            quasar::Directives::NEXT_EPISODE_ANNOUNCE,
            quasar::Directives::SHOW_VIDEO_SETTINGS,
            quasar::Directives::SHOW_GALLERY,
            quasar::Directives::SHOW_SEASON_GALLERY,
            quasar::Directives::SHOW_TV_GALLERY,
            quasar::Directives::SHOW_DESCRIPTION,
            quasar::Directives::SHOW_PAY_PUSH_SCREEN,
            quasar::Directives::SHOW_HOME_SCREEN,
            quasar::Directives::GO_FORWARD,
            quasar::Directives::GO_BACKWARD,
            quasar::Directives::GO_UP,
            quasar::Directives::GO_DOWN,
            quasar::Directives::GO_TO_THE_BEGINNING,
            quasar::Directives::GO_TO_THE_END,
            quasar::Directives::GO_TOP,
            quasar::Directives::MORDOVIA_SHOW,
            quasar::Directives::MORDOVIA_COMMAND,
            quasar::Directives::ONBOARDING_PLAY,
            quasar::Directives::ONBOARDING_SKIP,
            quasar::Directives::CHANGE_AUDIO,
            quasar::Directives::CHANGE_SUBTITLES,
            quasar::Directives::SETUP_RCU,
            quasar::Directives::SETUP_RCU_AUTO,
            quasar::Directives::SETUP_RCU_CHECK,
            quasar::Directives::SETUP_RCU_ADVANCED,
            quasar::Directives::SETUP_RCU_MANUAL};

        return s_names;
    }

    void ScreenCapability::onIsScreenActive(bool isScreenActive) {
        isActive_ = isScreenActive;
        deviceStateCapability_->setIsTvPluggedIn(isActive_ || isHdmiConnected_);
    }

    void ScreenCapability::handleDirective(const std::shared_ptr<YandexIO::Directive>& directive)
    {
        try {
            const auto& data = directive->getData();

            const auto& name = data.name;
            const auto& payload = data.payload;
            const auto& vinsRequestId = data.requestId;

            if (name == quasar::Directives::VIDEO_PLAY) {
                handleVideoPlay(payload, vinsRequestId);
            } else if (name == quasar::Directives::NEXT_EPISODE_ANNOUNCE) {
                handleEpisodeAnnounce(payload);
            } else if (name == quasar::Directives::SHOW_GALLERY && !payload.isNull()) {
                handleGallery(payload, data.displayedText, data.asrText, quasar::proto::GalleryType::SIMPLE);
            } else if (name == quasar::Directives::SHOW_SEASON_GALLERY && !payload.isNull()) {
                handleGallery(payload, data.displayedText, data.asrText, quasar::proto::GalleryType::SEASON);
            } else if (name == quasar::Directives::SHOW_TV_GALLERY && !payload.isNull()) {
                handleGallery(payload, data.displayedText, data.asrText, quasar::proto::GalleryType::TV);
            } else if (name == quasar::Directives::SHOW_DESCRIPTION && !payload.isNull()) {
                handleDescription(payload);
            } else if (name == quasar::Directives::SHOW_PAY_PUSH_SCREEN && !payload.isNull()) {
                handlePayScreen(payload);
            } else if (name == quasar::Directives::SHOW_HOME_SCREEN) {
                handleShowHomeScreen();
            } else if (name == quasar::Directives::GO_FORWARD) {
                handleGoForward(vinsRequestId);
            } else if (name == quasar::Directives::GO_BACKWARD) {
                handleGoBackward(payload, vinsRequestId);
            } else if (name == quasar::Directives::GO_UP) {
                handleGoUp(vinsRequestId);
            } else if (name == quasar::Directives::GO_DOWN) {
                handleGoDown(vinsRequestId);
            } else if (name == quasar::Directives::GO_TO_THE_BEGINNING) {
                handleGoToTheBeginning(vinsRequestId);
            } else if (name == quasar::Directives::GO_TO_THE_END) {
                handleGoToTheEnd(vinsRequestId);
            } else if (name == quasar::Directives::GO_TOP) {
                handleGoTop(vinsRequestId);
            } else if (name == quasar::Directives::SCREEN_ON) {
                handleScreenOn();
            } else if (name == quasar::Directives::SCREEN_OFF) {
                handleScreenOff();
            } else if (name == quasar::Directives::MORDOVIA_SHOW) {
                handleMordoviaShow(payload, vinsRequestId);
            } else if (name == quasar::Directives::MORDOVIA_COMMAND) {
                handleMordoviaCommand(payload, vinsRequestId);
            } else if (name == quasar::Directives::ONBOARDING_PLAY) {
                handleOnboardingPlay();
            } else if (name == quasar::Directives::ONBOARDING_SKIP) {
                handleOnboardingSkip();
            } else if (name == quasar::Directives::CHANGE_AUDIO) {
                handleChangeAudio(payload);
            } else if (name == quasar::Directives::CHANGE_SUBTITLES) {
                handleChangeSubtitles(payload);
            } else if (name == quasar::Directives::SHOW_VIDEO_SETTINGS) {
                handleShowVideoSettings();
            } else if (name == Directives::SETUP_RCU) {
                handleSetupRcu();
            } else if (name == Directives::SETUP_RCU_AUTO) {
                handleSetupRcuAuto(payload);
            } else if (name == Directives::SETUP_RCU_CHECK) {
                handleSetupRcuCheck();
            } else if (name == Directives::SETUP_RCU_ADVANCED) {
                handleSetupRcuAdvanced();
            } else if (name == Directives::SETUP_RCU_MANUAL) {
                handleSetupRcuManual();
            }
        } catch (const std::exception& e) {
            YIO_LOG_ERROR_EVENT("ScreenCapabitlity.FailedHandleDirective", "handleDirective failed: " << e.what());
        }
    }

    void ScreenCapability::cancelDirective(const std::shared_ptr<YandexIO::Directive>& /*directive*/)
    {
    }

    void ScreenCapability::prefetchDirective(const std::shared_ptr<YandexIO::Directive>& /*directive*/)
    {
    }

    const std::string& ScreenCapability::getPreprocessorName() const {
        static const std::string s_name = "ScreenCapabilityPreprocessor";
        return s_name;
    }

    void ScreenCapability::preprocessDirectives(std::list<std::shared_ptr<Directive>>& directives)
    {
        for (auto iter = directives.begin(); iter != directives.end(); ++iter) {
            auto& directive = *iter;

            if (directive->is(Directives::VIDEO_PLAY)) {
                iter = directives.insert(iter, YandexIO::Directive::createClientAction(Directives::STOP_LEGACY_PLAYER));
                std::advance(iter, 1);
            }
        }
    }

    void ScreenCapability::onAppState(const quasar::proto::AppState& appState) {
        if (appState.has_screen_state() && appState.screen_state().has_screen_type()) {
            int screenType = (int)appState.screen_state().screen_type();
            if (screenType_ != screenType) {
                if (screenType_ == (int)proto::AppState::MUSIC_PLAYER && aliceDeviceState_.getCurrentlyPlayingPlayerType() == AudioPlayerType::MUSIC) {
                    if (auto dp = directiveProcessor_.lock()) {
                        dp->addDirectives({Directive::createLocalAction(Directives::STOP_LEGACY_PLAYER)});
                    }
                }
                screenType_ = screenType;
            }
        }
        isHdmiConnected_ = appState.has_screen_state() && appState.screen_state().is_hdmi_connected();
        deviceStateCapability_->setIsTvPluggedIn(isActive_ || isHdmiConnected_);

        if (appState.has_video_state() || appState.has_screen_state()) {
            deviceStateCapability_->setVideoState(YandexIO::convertVideoState(appState.video_state(), appState.screen_state(), lastPlayVideoTs_, isActive_));
        }
        if (appState.has_rcu_state()) {
            deviceStateCapability_->setRcuState(YandexIO::convertRcuState(appState.rcu_state()));
        }
        if (appState.has_packages_state()) {
            deviceStateCapability_->setPackagesState(YandexIO::convertPackagesState(appState.packages_state()));
        }
        if (appState.has_media_info()) {
            deviceStateCapability_->setScreenState(YandexIO::convertScreenState(appState.media_info()));
        }
    }

    void ScreenCapability::handleScreenOn() {
        auto message = ipc::buildMessage([&](auto& msg) {
            msg.mutable_media_message()->mutable_hdmi_on();
        });
        interfaceConnector_->sendMessage(message);
    }

    void ScreenCapability::handleScreenOff() {
        auto message = ipc::buildMessage([&](auto& msg) {
            msg.mutable_media_message()->mutable_hdmi_off();
        });
        interfaceConnector_->sendMessage(message);
    }

    void ScreenCapability::handleMordoviaShow(const Json::Value& payload, const std::string& vinsRequestId) {
        quasar::proto::QuasarMessage message;
        auto mordoviaShow = message.mutable_mordovia_show();
        mordoviaShow->set_vins_request_id(TString(vinsRequestId));

        if (!payload["url"].isNull()) {
            mordoviaShow->set_url(getString(payload, "url"));
        }

        if (!payload["is_fullscreen"].isNull()) {
            mordoviaShow->set_is_full_screen(payload["is_fullscreen"].asBool());
        }

        if (!payload["splash_div"].isNull()) {
            mordoviaShow->set_splash_div(getString(payload, "splash_div"));
        }

        if (!payload["callback_prototype"].isNull()) {
            // NOTE: value is json template, but here is no need to parse it in any form
            auto prototype = getJson(payload, "callback_prototype");
            mordoviaShow->set_server_callback_prototype(jsonToString(prototype));
        }

        if (!payload["view_key"].isNull()) {
            mordoviaShow->set_view_key(getString(payload, "view_key"));
        }

        if (!payload["scenario_name"].isNull()) {
            mordoviaShow->set_scenario_name(getString(payload, "scenario_name"));
        }

        if (!payload["go_back"].isNull()) {
            mordoviaShow->set_go_back(payload["go_back"].asBool());
        }

        interfaceConnector_->sendMessage(std::move(message));
    }

    void ScreenCapability::handleMordoviaCommand(const Json::Value& payload, const std::string& vinsRequestId) {
        quasar::proto::QuasarMessage message;
        auto mordoviaCommand = message.mutable_mordovia_command();
        mordoviaCommand->set_vins_request_id(TString(vinsRequestId));

        if (!payload["command"].isNull()) {
            mordoviaCommand->set_command(getString(payload, "command"));
        }

        mordoviaCommand->set_meta(jsonToString(payload));

        if (!payload["view_key"].isNull()) {
            mordoviaCommand->set_view_key(getString(payload, "view_key"));
        }

        if (!payload["scenario_name"].isNull()) {
            mordoviaCommand->set_scenario_name(getString(payload, "scenario_name"));
        }

        interfaceConnector_->sendMessage(std::move(message));
    }

    void ScreenCapability::handleOnboardingPlay() {
        quasar::proto::QuasarMessage message;
        message.mutable_onboarding_request()->mutable_play_onboarding();

        interfaceConnector_->sendMessage(std::move(message));
    }

    void ScreenCapability::handleOnboardingSkip() {
        quasar::proto::QuasarMessage message;
        message.mutable_onboarding_request()->mutable_skip_onboarding();

        interfaceConnector_->sendMessage(std::move(message));
    }

    void ScreenCapability::handleChangeAudio(const Json::Value& payload) {
        quasar::proto::QuasarMessage message;
        auto changeAudio = message.mutable_media_request()->mutable_change_audio();
        if (!payload["language"].isNull()) {
            changeAudio->set_language(getString(payload, "language"));
        }
        if (!payload["title"].isNull()) {
            changeAudio->set_title(getString(payload, "title"));
        }

        interfaceConnector_->sendMessage(std::move(message));
    }

    void ScreenCapability::handleChangeSubtitles(const Json::Value& payload) {
        quasar::proto::QuasarMessage message;
        auto changeSubtitles = message.mutable_media_request()->mutable_change_subtitles();
        if (!payload["language"].isNull()) {
            changeSubtitles->set_language(getString(payload, "language"));
        }
        if (!payload["title"].isNull()) {
            changeSubtitles->set_title(getString(payload, "title"));
        }
        if (!payload["enable"].isNull()) {
            changeSubtitles->set_enable(getBool(payload, "enable"));
        }

        interfaceConnector_->sendMessage(std::move(message));
    }

    void ScreenCapability::handleVideoPlay(const Json::Value& payload, const std::string& vinsRequestId)
    {
        quasar::proto::QuasarMessage message = buildVideoPlayMessage(payload, vinsRequestId);
        lastPlayVideoTs_ = getNowTimestampMs();
        interfaceConnector_->sendMessage(std::move(message));
    }

    void ScreenCapability::handleEpisodeAnnounce(const Json::Value& payload)
    {
        quasar::proto::QuasarMessage message;

        if (!payload["thumbnail_url"].isNull()) {
            message.mutable_media_message()->mutable_show_tv_announce()->set_thumbnail_url(getString(payload, "thumbnail_url"));
        }

        if (!payload["start_time"].isNull()) {
            message.mutable_media_message()->mutable_show_tv_announce()->set_start_time(getInt64(payload, "start_time"));
        }

        if (!payload["next_episode_name"].isNull()) {
            message.mutable_media_message()->mutable_show_tv_announce()->set_next_episode_name(getString(payload, "next_episode_name"));
        }

        interfaceConnector_->sendMessage(std::move(message));
    }

    void ScreenCapability::handleShowHomeScreen()
    {
        quasar::proto::QuasarMessage message;
        message.mutable_media_message()->mutable_to_home_screen();

        interfaceConnector_->sendMessage(std::move(message));
    }

    void ScreenCapability::handleGoForward(const std::string& vinsRequestId)
    {
        quasar::proto::QuasarMessage message;
        message.mutable_control_request()->mutable_navigation_request()->mutable_go_right();
        message.mutable_control_request()->set_origin(quasar::proto::ControlRequest_Origin_VOICE);
        message.mutable_control_request()->set_vins_request_id(TString(vinsRequestId));

        interfaceConnector_->sendMessage(std::move(message));
    }

    void ScreenCapability::handleGoBackward(const Json::Value& payload, const std::string& vinsRequestId)
    {
        quasar::proto::QuasarMessage message;
        auto goLeftCommand = message.mutable_control_request()->mutable_navigation_request()->mutable_go_left();

        if (!payload["visual"].isNull()) {
            goLeftCommand->mutable_visual_mode();
        } else if (!payload["historical"].isNull() && !payload["historical"]["history_url"].isNull()) {
            auto historicalMode = goLeftCommand->mutable_historical_mode();
            historicalMode->set_history_url(getString(payload["historical"], "history_url"));
        } else if (!payload["native"].isNull()) {
            goLeftCommand->mutable_native_mode();
        }

        message.mutable_control_request()->set_origin(quasar::proto::ControlRequest_Origin_VOICE);
        message.mutable_control_request()->set_vins_request_id(TString(vinsRequestId));

        interfaceConnector_->sendMessage(std::move(message));
    }

    void ScreenCapability::handleGoUp(const std::string& vinsRequestId)
    {
        quasar::proto::QuasarMessage message;
        message.mutable_control_request()->mutable_navigation_request()->mutable_go_up();
        message.mutable_control_request()->set_origin(quasar::proto::ControlRequest_Origin_VOICE);
        message.mutable_control_request()->set_vins_request_id(TString(vinsRequestId));

        interfaceConnector_->sendMessage(std::move(message));
    }

    void ScreenCapability::handleGoDown(const std::string& vinsRequestId)
    {
        quasar::proto::QuasarMessage message;
        message.mutable_control_request()->mutable_navigation_request()->mutable_go_down();
        message.mutable_control_request()->set_origin(quasar::proto::ControlRequest_Origin_VOICE);
        message.mutable_control_request()->set_vins_request_id(TString(vinsRequestId));

        interfaceConnector_->sendMessage(std::move(message));
    }

    void ScreenCapability::handleGoToTheBeginning(const std::string& vinsRequestId)
    {
        quasar::proto::QuasarMessage message;
        quasar::proto::NavigationMessage* direction = message.mutable_control_request()->mutable_navigation_request()->mutable_go_left();
        direction->mutable_scroll_amount()->mutable_till_end();
        message.mutable_control_request()->set_origin(quasar::proto::ControlRequest_Origin_VOICE);
        message.mutable_control_request()->set_vins_request_id(TString(vinsRequestId));

        interfaceConnector_->sendMessage(std::move(message));
    }

    void ScreenCapability::handleGoToTheEnd(const std::string& vinsRequestId)
    {
        quasar::proto::QuasarMessage message;
        quasar::proto::NavigationMessage* direction = message.mutable_control_request()->mutable_navigation_request()->mutable_go_right();
        direction->mutable_scroll_amount()->mutable_till_end();
        message.mutable_control_request()->set_origin(quasar::proto::ControlRequest_Origin_VOICE);
        message.mutable_control_request()->set_vins_request_id(TString(vinsRequestId));

        interfaceConnector_->sendMessage(std::move(message));
    }

    void ScreenCapability::handleGoTop(const std::string& vinsRequestId)
    {
        quasar::proto::QuasarMessage message;
        quasar::proto::NavigationMessage* direction = message.mutable_control_request()->mutable_navigation_request()->mutable_go_up();
        direction->mutable_scroll_amount()->mutable_till_end();
        message.mutable_control_request()->set_origin(quasar::proto::ControlRequest_Origin_VOICE);
        message.mutable_control_request()->set_vins_request_id(TString(vinsRequestId));

        interfaceConnector_->sendMessage(std::move(message));
    }

    void ScreenCapability::handleShowVideoSettings() {
        quasar::proto::QuasarMessage message;
        message.mutable_media_request()->mutable_show_video_settings();

        interfaceConnector_->sendMessage(std::move(message));
    }

    void ScreenCapability::handlePayScreen(const Json::Value& payload)
    {
        proto::QuasarMessage message;

        if (!payload["item"].isNull()) {
            convert(getJson(payload, "item"), message.mutable_media_message()->mutable_show_payment()->mutable_item());
        }
        if (!payload["tv_show_item"].isNull()) {
            convert(getJson(payload, "tv_show_item"), message.mutable_media_message()->mutable_show_payment()->mutable_tvshowitem());
        }
        message.mutable_media_message()->mutable_show_payment()->set_raw(jsonToString(payload));

        interfaceConnector_->sendMessage(std::move(message));
    }

    proto::QuasarMessage ScreenCapability::buildVideoPlayMessage(const Json::Value& payload, const std::string& vinsRequestId)
    {
        std::string uri = getString(payload, "uri");
        uri = trim(uri);
        YIO_LOG_DEBUG("video_play: " + uri);
        proto::QuasarMessage message;
        message.mutable_media_request()->mutable_play_video()->set_url(std::move(uri));
        message.mutable_media_request()->mutable_play_video()->set_vins_init_request_id(TString(vinsRequestId));

        if (!payload["session_token"].isNull()) {
            std::string session = getString(payload, "session_token");
            message.mutable_media_request()->mutable_play_video()->set_session_token(TString(session));
            YIO_LOG_DEBUG("session_token: " + session);
        }

        if (!payload["start_at"].isNull())
        {
            message.mutable_media_request()->mutable_play_video()->set_start_at(
                static_cast<int>(round(getDouble(payload, "start_at"))));
        }

        if (!payload["item"].isNull()) {
            convert(getJson(payload, "item"), message.mutable_media_request()->mutable_play_video()->mutable_item());
        }

        if (!payload["next_item"].isNull()) {
            convert(getJson(payload, "next_item"), message.mutable_media_request()->mutable_play_video()->mutable_next_item());
        }

        if (!payload["tv_show_item"].isNull()) {
            convert(getJson(payload, "tv_show_item"), message.mutable_media_request()->mutable_play_video()->mutable_tv_show_item());
        }

        if (!payload["payload"].isNull()) {
            message.mutable_media_request()->mutable_play_video()->set_payload(getString(payload, "payload"));
        }

        if (!payload["audio_language"].isNull()) {
            auto language = getString(payload, "audio_language");
            if (!language.empty()) {
                message.mutable_media_request()->mutable_play_video()->set_audio_language(std::move(language));
            }
        }

        if (!payload["subtitles_language"].isNull()) {
            auto language = getString(payload, "subtitles_language");
            if (!language.empty()) {
                message.mutable_media_request()->mutable_play_video()->set_subtitles_language(std::move(language));
            }
        }

        message.mutable_media_request()->mutable_play_video()->set_raw(jsonToString(payload));

        return message;
    }

    void ScreenCapability::handleGallery(const Json::Value& payload, const std::string& outputSpeech, const std::string& lastRecognizedPhrase,
                                         proto::GalleryType galleryType)
    {
        proto::QuasarMessage message;
        message.mutable_media_message()->mutable_show_video_list()->set_question(TString(lastRecognizedPhrase));

        if (!outputSpeech.empty()) {
            message.mutable_media_message()->mutable_show_video_list()->set_answer(TString(outputSpeech));
        }

        for (auto& item : payload["items"])
        {
            auto newItem = message.mutable_media_message()->mutable_show_video_list()->add_items();
            convert(item, newItem);
        }

        message.mutable_media_message()->mutable_show_video_list()->set_gallery_type(galleryType);
        message.mutable_media_message()->mutable_show_video_list()->set_raw_result(jsonToString(payload));

        if (!payload["tv_show_item"].isNull()) {
            convert(getJson(payload, "tv_show_item"), message.mutable_media_message()->mutable_show_video_list()->mutable_tv_show_item());
        }

        if (!payload["season"].isNull()) {
            message.mutable_media_message()->mutable_show_video_list()->set_season(getInt(payload, "season"));
        }

        interfaceConnector_->sendMessage(std::move(message));
    }

    void ScreenCapability::handleDescription(const Json::Value& payload)
    {
        proto::QuasarMessage msg;

        Json::Value item = getJson(payload, "item");
        convert(item, msg.mutable_media_message()->mutable_show_description()->mutable_media_item());
        msg.mutable_media_message()->mutable_show_description()->set_raw(jsonToString(payload));

        interfaceConnector_->sendMessage(std::move(msg));
    }

    void ScreenCapability::handleSetupRcu() {
        quasar::proto::QuasarMessage message;
        message.mutable_setup_rcu();

        interfaceConnector_->sendMessage(std::move(message));
    }

    void ScreenCapability::handleSetupRcuAuto(const Json::Value& payload) {
        quasar::proto::QuasarMessage message;
        message.mutable_setup_rcu_auto();
        if (!payload["tv_model"].isNull()) {
            message.mutable_setup_rcu_auto()->set_tv_model(getString(payload, "tv_model"));
        }

        interfaceConnector_->sendMessage(std::move(message));
    }

    void ScreenCapability::handleSetupRcuCheck() {
        quasar::proto::QuasarMessage message;
        message.mutable_setup_rcu_check();

        interfaceConnector_->sendMessage(std::move(message));
    }

    void ScreenCapability::handleSetupRcuAdvanced() {
        quasar::proto::QuasarMessage message;
        message.mutable_setup_rcu_advanced();

        interfaceConnector_->sendMessage(std::move(message));
    }

    void ScreenCapability::handleSetupRcuManual() {
        quasar::proto::QuasarMessage message;
        message.mutable_setup_rcu_manual();

        interfaceConnector_->sendMessage(std::move(message));
    }

    void ScreenCapability::convert(const Json::Value& item, proto::MediaItem* mediaItem)
    {
        YIO_LOG_DEBUG(jsonToString(item));

        if (!item["name"].isNull()) {
            mediaItem->set_name(getString(item, "name"));
        }
        if (!item["source_host"].isNull()) {
            mediaItem->set_source_host(getString(item, "source_host"));
        }

        if (!item["type"].isNull()) {
            mediaItem->set_type(convertType(getString(item, "type")));
        } else {
            mediaItem->set_type(proto::MediaItem::VIDEO); // TODO change later
        }

        if (!item["entref"].isNull()) {
            mediaItem->set_entref(getString(item, "entref"));
        }

        if ((proto::MediaItem::MOVIE == mediaItem->type() || proto::MediaItem::TV_SHOW == mediaItem->type()) && !item["cover_url_2x3"].isNull())
        {
            mediaItem->set_cover_url(getString(item, "cover_url_2x3"));
        } else if ((proto::MediaItem::VIDEO == mediaItem->type() ||
                    proto::MediaItem::TV_SHOW_EPISODE == mediaItem->type() ||
                    proto::MediaItem::TV_STREAM == mediaItem->type()) &&
                   !item["thumbnail_url_16x9"].isNull())
        {
            mediaItem->set_cover_url(getString(item, "thumbnail_url_16x9"));
        }

        if (!item["cover_url_16x9"].isNull()) {
            mediaItem->set_background_url(getString(item, "cover_url_16x9"));
        }

        if (!item["genre"].isNull()) {
            mediaItem->set_genre(getString(item, "genre"));
        }

        if (!item["rating"].isNull()) {
            mediaItem->set_rating(getDouble(item, "rating"));
        }

        if (!item["description"].isNull()) {
            mediaItem->set_description(getString(item, "description"));
        }

        if (!item["progress"].isNull()) {
            mediaItem->set_progress_percents(static_cast<int>(getDouble(item, "progress") * 100));
        }

        if (!item["seasons_count"].isNull()) {
            mediaItem->set_seasons(getInt(item, "seasons_count"));
        }

        if (!item["seasons"].isNull()) {
            for (auto& seasonItem : item["seasons"]) {
                auto season = mediaItem->add_season_numbers();
                if (!seasonItem["number"].isNull()) {
                    season->set_number(getInt(seasonItem, "number"));
                }
            }
        }

        if (!item["release_year"].isNull()) {
            mediaItem->set_release_year(std::to_string(getInt(item, "release_year")));
        }

        if (!item["directors"].isNull()) {
            mediaItem->set_directors(getString(item, "directors"));
        }

        if (!item["actors"].isNull()) {
            mediaItem->set_actors(getString(item, "actors"));
        }

        if (!item["age_limit"].isNull()) {
            mediaItem->set_age_restriction(getString(item, "age_limit") + "+");
        }

        if (!item["view_count"].isNull()) {
            mediaItem->set_view_count(getInt64(item, "view_count"));
        }

        if (!item["provider_name"].isNull()) {
            mediaItem->set_provider(parseProvider(getString(item, "provider_name")));
        }

        if (!item["provider_item_id"].isNull()) {
            mediaItem->set_item_id(getString(item, "provider_item_id"));
        }

        if (!item["available"].isNull()) {
            mediaItem->set_available(getInt(item, "available") > 0);
        } else {
            mediaItem->set_available(true); // this is due to https://st.yandex-team.ru/QUASAR-1534
        }

        if (!item["availability_request"].isNull()) {
            mediaItem->set_availability_request(jsonToString(getJson(item, "availability_request")));
        }

        if (!item["price_from"].isNull()) {
            mediaItem->set_cost_from(getInt(item, "price_from"));
        }

        if (!item["provider_info"].isNull()) {
            for (auto& info : item["provider_info"]) {
                auto providerInfo = mediaItem->add_provider_info();
                if (!info["provider_name"].isNull()) {
                    providerInfo->set_provider(parseProvider(getString(info, "provider_name")));
                }

                if (!info["provider_item_id"].isNull()) {
                    providerInfo->set_item_id(getString(info, "provider_item_id"));
                }

                if (!info["available"].isNull()) {
                    providerInfo->set_available(getInt(info, "available") > 0);
                } else {
                    providerInfo->set_available(true); // this due to https://st.yandex-team.ru/QUASAR-1534
                }

                if (!info["price_from"].isNull()) {
                    providerInfo->set_cost_from(getInt(info, "price_from"));
                }
            }
        }

        if (!item["duration"].isNull()) {
            mediaItem->set_duration(getInt(item, "duration"));
        }

        if (!item["season"].isNull()) {
            mediaItem->set_season(getInt(item, "season"));
        }

        if (!item["episode"].isNull()) {
            mediaItem->set_episode(getInt(item, "episode"));
        }

        if (!item["unauthorized"].isNull()) {
            mediaItem->set_unauthorized(getInt(item, "unauthorized") > 0);
        }

        if (!item["play_uri"].isNull()) {
            mediaItem->set_play_uri(getString(item, "play_uri"));
        }

        if (!item["tv_stream_info"].isNull()) {
            Json::Value streamInfo = getJson(item, "tv_stream_info");
            if (!streamInfo["tv_episode_name"].isNull()) {
                mediaItem->set_tv_episode_name(getString(streamInfo, "tv_episode_name"));
            }
            if (!streamInfo["deep_hd"].isNull()) {
                mediaItem->set_deep_hd(getInt(streamInfo, "deep_hd") > 0);
            }
            if (!streamInfo["channel_type"].isNull()) {
                mediaItem->set_personal_channel(getString(streamInfo, "channel_type") == "personal");
            }
        }

        if (!item["camera_stream_info"].isNull()) {
            Json::Value streamInfo = getJson(item, "camera_stream_info");
            if (!streamInfo["type"].isNull()) {
                mediaItem->set_camera_stream_type(convertStreamType(getString(streamInfo, "type")));
            }
        }

        auto jsonStreamToProto = [](auto& jsonStream, auto protoStream) {
            if (!jsonStream["language"].isNull()) {
                protoStream->set_language(getString(jsonStream, "language"));
            }
            if (!jsonStream["title"].isNull()) {
                protoStream->set_title(getString(jsonStream, "title"));
            }
            if (!jsonStream["index"].isNull()) {
                protoStream->set_index(getInt(jsonStream, "index"));
            }
            if (!jsonStream["suggest"].isNull()) {
                protoStream->set_suggest(getString(jsonStream, "suggest"));
            }
        };
        if (item["audio_streams"].isArray()) {
            for (auto& audioStream : item["audio_streams"]) {
                auto stream = mediaItem->add_audio_streams();
                jsonStreamToProto(audioStream, stream);
            }
        }
        if (item["subtitles"].isArray()) {
            for (auto& subtitles : item["subtitles"]) {
                auto stream = mediaItem->add_subtitles();
                jsonStreamToProto(subtitles, stream);
            }
        }

        if (item["skippable_fragments"].isArray()) {
            for (auto& skippableFragment : item["skippable_fragments"]) {
                auto fragment = mediaItem->add_skippable_fragments();
                fragment->set_type(parseSkippableFragmentType(tryGetString(skippableFragment, "type", "other")));
                fragment->set_start_time_sec(tryGetInt64(skippableFragment, "start_time", 0));
                fragment->set_end_time_sec(tryGetInt64(skippableFragment, "end_time", 0));
            }
        }
        // TODO more fields
    }

    proto::MediaItem::Type ScreenCapability::convertType(const std::string& type)
    {
        static const std::unordered_map<std::string, proto::MediaItem_Type> map = {
            {"movie", proto::MediaItem::MOVIE},
            {"tv_show", proto::MediaItem::TV_SHOW},
            {"tv_show_episode", proto::MediaItem::TV_SHOW_EPISODE},
            {"tv_stream", proto::MediaItem::TV_STREAM},
            {"camera_stream", proto::MediaItem::CAMERA_STREAM}};

        if (const auto it = map.find(type); it != map.end()) {
            return it->second;
        }
        return proto::MediaItem::VIDEO;
    }

    proto::MediaItem::StreamType ScreenCapability::convertStreamType(const std::string& type)
    {
        static const std::unordered_map<std::string, proto::MediaItem_StreamType> map({{"dash", proto::MediaItem::DASH},
                                                                                       {"smooth_streaming", proto::MediaItem::SMOOTH_STREAMING},
                                                                                       {"hls", proto::MediaItem::HLS},
                                                                                       {"other", proto::MediaItem::OTHER}});

        if (const auto it = map.find(type); it != map.end()) {
            return it->second;
        }

        return proto::MediaItem::OTHER;
    }

    proto::Provider ScreenCapability::parseProvider(const std::string& providerName)
    {
        static const std::unordered_map<std::string, proto::Provider> map = {
            {"ivi", proto::Provider::IVI},
            {"amediateka", proto::Provider::AMEDIATEKA},
            {"youtube", proto::Provider::YOUTUBE},
            {"yavideo", proto::Provider::YAVIDEO},
            {"kinopoisk", proto::Provider::KINOPOISK},
            {"strm", proto::Provider::STRM},
            {"yavideo_proxy", proto::Provider::YAVIDEO_PROXY},
            {"camera_stream", proto::Provider::CAMERA_STREAM}};

        if (const auto it = map.find(providerName); it != map.end()) {
            return it->second;
        }
        return proto::Provider::UNKNOWN;
    }

    proto::SkippableFragments::Type ScreenCapability::parseSkippableFragmentType(const std::string& fragmentType) {
        static const std::unordered_map<std::string, proto::SkippableFragments_Type> map = {
            {"intro", proto::SkippableFragments::INTRO},
            {"recap", proto::SkippableFragments::RECAP},
            {"credits", proto::SkippableFragments::CREDITS},
            {"next_episode", proto::SkippableFragments::NEXT_EPISODE},
        };

        if (const auto it = map.find(fragmentType); it != map.end()) {
            return it->second;
        }
        return proto::SkippableFragments::OTHER;
    }

} // namespace YandexIO
